<?php
/**
 * User Model
 * Salon/Spa/Beauty Parlor Booking System
 */

class User extends Model {
    protected $table = 'users';
    protected $fillable = ['company_id', 'branch_id', 'email', 'password_hash', 'first_name', 'last_name', 'phone', 'avatar', 'role', 'status'];
    protected $hidden = ['password_hash'];

    /**
     * Get user with roles and permissions
     */
    public function findWithRoles($id) {
        $user = $this->find($id);
        if ($user) {
            $user['roles'] = $this->getUserRoles($id);
            $user['permissions'] = $this->getUserPermissions($id);
        }
        return $user;
    }

    /**
     * Get user roles
     */
    public function getUserRoles($userId) {
        return $this->db->query("
            SELECT r.name, r.slug
            FROM roles r
            INNER JOIN user_roles ur ON r.id = ur.role_id
            WHERE ur.user_id = ?
        ")->bind(1, $userId)->resultSet();
    }

    /**
     * Get user permissions
     */
    public function getUserPermissions($userId) {
        $permissions = [];
        $roles = $this->getUserRoles($userId);

        foreach ($roles as $role) {
            $roleData = $this->db->query("SELECT permissions FROM roles WHERE slug = ?")
                                ->bind(1, $role['slug'])
                                ->single();

            if ($roleData && $roleData['permissions']) {
                $rolePermissions = json_decode($roleData['permissions'], true);
                if (is_array($rolePermissions)) {
                    $permissions = array_merge($permissions, $rolePermissions);
                }
            }
        }

        return array_unique($permissions);
    }

    /**
     * Find user by email
     */
    public function findByEmail($email) {
        return $this->findBy(['email' => $email]);
    }

    /**
     * Get users by company
     */
    public function getByCompany($companyId, $role = null) {
        $sql = "SELECT * FROM {$this->table} WHERE company_id = ?";
        $params = [$companyId];

        if ($role) {
            $sql .= " AND role = ?";
            $params[] = $role;
        }

        $sql .= " ORDER BY first_name, last_name";

        $query = $this->db->query($sql);
        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->resultSet();
    }

    /**
     * Get staff by branch
     */
    public function getStaffByBranch($branchId) {
        return $this->findBy(['branch_id' => $branchId, 'role' => 'staff', 'status' => 'active']);
    }

    /**
     * Update last login
     */
    public function updateLastLogin($userId) {
        return $this->update($userId, ['last_login' => date('Y-m-d H:i:s')]);
    }

    /**
     * Check if email exists
     */
    public function emailExists($email, $excludeId = null) {
        $sql = "SELECT COUNT(*) as count FROM {$this->table} WHERE email = ?";
        $params = [$email];

        if ($excludeId) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
        }

        $result = $this->db->query($sql)->single();
        return $result['count'] > 0;
    }
}
