<?php
/**
 * Product Model
 * Salon/Spa/Beauty Parlor Booking System
 */

class Product extends Model {
    protected $table = 'products';

    /**
     * Get products by category
     */
    public function getByCategory($categoryId, $limit = null, $offset = 0) {
        $query = "SELECT * FROM {$this->table} WHERE category_id = ? AND is_active = 1 ORDER BY name ASC";

        if ($limit) {
            $query .= " LIMIT ? OFFSET ?";
            return $this->db->query($query)->bind(1, $categoryId)->bind(2, $limit)->bind(3, $offset)->resultSet();
        }

        return $this->db->query($query)->bind(1, $categoryId)->resultSet();
    }

    /**
     * Search products
     */
    public function search($query, $categoryId = null, $limit = 50) {
        $where = "MATCH(name, description, short_description) AGAINST(? IN NATURAL LANGUAGE MODE) AND is_active = 1";
        $params = [$query];

        if ($categoryId) {
            $where .= " AND category_id = ?";
            $params[] = $categoryId;
        }

        $sql = "SELECT *, MATCH(name, description, short_description) AGAINST(? IN NATURAL LANGUAGE MODE) as relevance
                FROM {$this->table}
                WHERE {$where}
                ORDER BY relevance DESC, name ASC
                LIMIT ?";

        $params[] = $limit;

        return $this->db->query($sql, $params)->resultSet();
    }

    /**
     * Get featured products
     */
    public function getFeatured($limit = 4) {
        return $this->db->query("SELECT * FROM {$this->table} WHERE is_featured = 1 AND is_active = 1 ORDER BY created_at DESC LIMIT ?")
                       ->bind(1, $limit)->resultSet();
    }

    /**
     * Get related products
     */
    public function getRelated($productId, $categoryId, $limit = 4) {
        return $this->db->query("SELECT * FROM {$this->table}
                                WHERE category_id = ? AND id != ? AND is_active = 1
                                ORDER BY created_at DESC LIMIT ?")
                       ->bind(1, $categoryId)->bind(2, $productId)->bind(3, $limit)->resultSet();
    }

    /**
     * Update stock quantity
     */
    public function updateStock($productId, $quantity) {
        return $this->db->update($this->table, ['stock_quantity' => $quantity], ['id' => $productId]);
    }

    /**
     * Check if product is in stock
     */
    public function isInStock($productId, $quantity = 1) {
        $product = $this->find($productId);
        if (!$product) return false;

        // If stock_quantity is NULL, assume unlimited stock
        if ($product['stock_quantity'] === null) return true;

        return $product['stock_quantity'] >= $quantity;
    }

    /**
     * Get low stock products
     */
    public function getLowStock($threshold = 5) {
        return $this->db->query("SELECT * FROM {$this->table}
                                WHERE stock_quantity IS NOT NULL AND stock_quantity <= ? AND is_active = 1
                                ORDER BY stock_quantity ASC")
                       ->bind(1, $threshold)->resultSet();
    }

    /**
     * Get product with category info
     */
    public function getWithCategory($productId) {
        return $this->db->query("SELECT p.*, pc.name as category_name, pc.slug as category_slug
                                FROM {$this->table} p
                                LEFT JOIN product_categories pc ON p.category_id = pc.id
                                WHERE p.id = ?")
                       ->bind(1, $productId)->single();
    }

    /**
     * Get products by company
     */
    public function getByCompany($companyId, $limit = null, $offset = 0) {
        $query = "SELECT * FROM {$this->table} WHERE company_id = ? ORDER BY created_at DESC";

        if ($limit) {
            $query .= " LIMIT ? OFFSET ?";
            return $this->db->query($query)->bind(1, $companyId)->bind(2, $limit)->bind(3, $offset)->resultSet();
        }

        return $this->db->query($query)->bind(1, $companyId)->resultSet();
    }
}
