<?php
/**
 * Order Model
 * Salon/Spa/Beauty Parlor Booking System
 */

class Order extends Model {
    protected $table = 'orders';

    /**
     * Get orders by customer
     */
    public function getByCustomer($customerId, $limit = null, $offset = 0) {
        $query = "SELECT * FROM {$this->table} WHERE customer_id = ? ORDER BY created_at DESC";

        if ($limit) {
            $query .= " LIMIT ? OFFSET ?";
            return $this->db->query($query)->bind(1, $customerId)->bind(2, $limit)->bind(3, $offset)->resultSet();
        }

        return $this->db->query($query)->bind(1, $customerId)->resultSet();
    }

    /**
     * Get order with items
     */
    public function getWithItems($orderId) {
        $order = $this->find($orderId);
        if (!$order) return null;

        $items = $this->db->query("SELECT oi.*, p.name, p.slug, p.images
                                  FROM order_items oi
                                  JOIN products p ON oi.product_id = p.id
                                  WHERE oi.order_id = ?
                                  ORDER BY oi.id ASC")
                         ->bind(1, $orderId)->resultSet();

        $order['items'] = $items;
        return $order;
    }

    /**
     * Get orders by status
     */
    public function getByStatus($status, $limit = null, $offset = 0) {
        $query = "SELECT * FROM {$this->table} WHERE status = ? ORDER BY created_at DESC";

        if ($limit) {
            $query .= " LIMIT ? OFFSET ?";
            return $this->db->query($query)->bind(1, $status)->bind(2, $limit)->bind(3, $offset)->resultSet();
        }

        return $this->db->query($query)->bind(1, $status)->resultSet();
    }

    /**
     * Update order status
     */
    public function updateStatus($orderId, $status) {
        return $this->db->update($this->table, [
            'status' => $status,
            'updated_at' => date('Y-m-d H:i:s')
        ], ['id' => $orderId]);
    }

    /**
     * Get order statistics
     */
    public function getStats($customerId = null, $dateFrom = null, $dateTo = null) {
        $where = [];
        $params = [];

        if ($customerId) {
            $where[] = "customer_id = ?";
            $params[] = $customerId;
        }

        if ($dateFrom) {
            $where[] = "created_at >= ?";
            $params[] = $dateFrom . ' 00:00:00';
        }

        if ($dateTo) {
            $where[] = "created_at <= ?";
            $params[] = $dateTo . ' 23:59:59';
        }

        $whereClause = !empty($where) ? "WHERE " . implode(' AND ', $where) : "";

        $stats = $this->db->query("SELECT
            COUNT(*) as total_orders,
            SUM(total_amount) as total_revenue,
            AVG(total_amount) as avg_order_value,
            COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_orders,
            COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_orders,
            COUNT(CASE WHEN status = 'cancelled' THEN 1 END) as cancelled_orders
            FROM {$this->table} {$whereClause}", $params)->single();

        return $stats;
    }

    /**
     * Create order with items
     */
    public function createOrder($orderData, $orderItems) {
        // Start transaction
        $this->db->beginTransaction();

        try {
            // Create order
            $orderId = $this->db->insert($this->table, $orderData);

            // Add order items
            foreach ($orderItems as $item) {
                $item['order_id'] = $orderId;
                $this->db->insert('order_items', $item);
            }

            $this->db->commit();
            return $orderId;

        } catch (Exception $e) {
            $this->db->rollback();
            throw $e;
        }
    }

    /**
     * Cancel order
     */
    public function cancelOrder($orderId) {
        $order = $this->find($orderId);
        if (!$order || !in_array($order['status'], ['pending', 'processing'])) {
            return false;
        }

        // Start transaction
        $this->db->beginTransaction();

        try {
            // Update order status
            $this->updateStatus($orderId, 'cancelled');

            // Restore product stock
            $items = $this->db->query("SELECT * FROM order_items WHERE order_id = ?")
                             ->bind(1, $orderId)->resultSet();

            foreach ($items as $item) {
                $this->db->query("UPDATE products SET stock_quantity = stock_quantity + ? WHERE id = ?")
                        ->bind(1, $item['quantity'])->bind(2, $item['product_id'])->execute();
            }

            $this->db->commit();
            return true;

        } catch (Exception $e) {
            $this->db->rollback();
            return false;
        }
    }

    /**
     * Get recent orders
     */
    public function getRecent($limit = 10) {
        return $this->db->query("SELECT o.*, u.first_name, u.last_name
                                FROM {$this->table} o
                                LEFT JOIN users u ON o.customer_id = u.id
                                ORDER BY o.created_at DESC LIMIT ?")
                       ->bind(1, $limit)->resultSet();
    }

    /**
     * Search orders
     */
    public function search($query, $customerId = null) {
        $where = "o.order_number LIKE ?";
        $params = ["%{$query}%"];

        if ($customerId) {
            $where .= " AND o.customer_id = ?";
            $params[] = $customerId;
        }

        return $this->db->query("SELECT o.*, u.first_name, u.last_name
                                FROM {$this->table} o
                                LEFT JOIN users u ON o.customer_id = u.id
                                WHERE {$where}
                                ORDER BY o.created_at DESC", $params)->resultSet();
    }
}
