<?php
/**
 * Company Model
 * Salon/Spa/Beauty Parlor Booking System
 */

class Company extends Model {
    protected $table = 'companies';
    protected $fillable = ['name', 'slug', 'description', 'logo', 'favicon', 'primary_color', 'secondary_color', 'website', 'email', 'phone', 'whatsapp_number', 'address', 'timezone', 'currency', 'status'];

    /**
     * Get company with branches and services count
     */
    public function findWithStats($id) {
        $company = $this->find($id);
        if ($company) {
            $company['branches_count'] = $this->count(['company_id' => $id]);
            $company['services_count'] = $this->db->query("SELECT COUNT(*) as count FROM services WHERE company_id = ? AND is_active = 1")
                                                  ->bind(1, $id)
                                                  ->single()['count'];
            $company['staff_count'] = $this->db->query("SELECT COUNT(*) as count FROM users WHERE company_id = ? AND role = 'staff' AND status = 'active'")
                                               ->bind(1, $id)
                                               ->single()['count'];
            $company['bookings_today'] = $this->db->query("SELECT COUNT(*) as count FROM bookings WHERE company_id = ? AND DATE(start_at) = CURDATE()")
                                                  ->bind(1, $id)
                                                  ->single()['count'];
        }
        return $company;
    }

    /**
     * Get active companies
     */
    public function getActive() {
        return $this->findBy(['status' => 'active']);
    }

    /**
     * Check if slug exists
     */
    public function slugExists($slug, $excludeId = null) {
        $sql = "SELECT COUNT(*) as count FROM {$this->table} WHERE slug = ?";
        $params = [$slug];

        if ($excludeId) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
        }

        $result = $this->db->query($sql)->single();
        return $result['count'] > 0;
    }

    /**
     * Get company branches
     */
    public function getBranches($companyId) {
        return $this->db->query("
            SELECT b.*, COUNT(s.id) as services_count, COUNT(st.id) as staff_count
            FROM branches b
            LEFT JOIN services s ON b.id = s.branch_id AND s.is_active = 1
            LEFT JOIN users st ON b.id = st.branch_id AND st.role = 'staff' AND st.status = 'active'
            WHERE b.company_id = ? AND b.status = 'active'
            GROUP BY b.id
            ORDER BY b.is_main DESC, b.name ASC
        ")->bind(1, $companyId)->resultSet();
    }

    /**
     * Get company services
     */
    public function getServices($companyId, $activeOnly = true) {
        $sql = "SELECT * FROM services WHERE company_id = ?";
        $params = [$companyId];

        if ($activeOnly) {
            $sql .= " AND is_active = 1";
        }

        $sql .= " ORDER BY sort_order ASC, name ASC";

        return $this->db->query($sql)->bind(1, $companyId)->resultSet();
    }

    /**
     * Get company statistics
     */
    public function getStats($companyId, $dateFrom = null, $dateTo = null) {
        $sql = "
            SELECT
                COUNT(DISTINCT b.id) as total_bookings,
                COUNT(DISTINCT CASE WHEN b.status = 'completed' THEN b.id END) as completed_bookings,
                COUNT(DISTINCT CASE WHEN b.status = 'cancelled' THEN b.id END) as cancelled_bookings,
                COUNT(DISTINCT CASE WHEN b.status = 'no_show' THEN b.id END) as no_show_bookings,
                COUNT(DISTINCT s.id) as total_services,
                COUNT(DISTINCT st.id) as total_staff,
                COUNT(DISTINCT br.id) as total_branches,
                SUM(CASE WHEN b.status = 'completed' THEN b.total_amount ELSE 0 END) as total_revenue,
                AVG(CASE WHEN b.status = 'completed' THEN b.total_amount ELSE NULL END) as avg_booking_value
            FROM companies c
            LEFT JOIN bookings b ON c.id = b.company_id
            LEFT JOIN services s ON c.id = s.company_id AND s.is_active = 1
            LEFT JOIN users st ON c.id = st.company_id AND st.role = 'staff' AND st.status = 'active'
            LEFT JOIN branches br ON c.id = br.company_id AND br.status = 'active'
            WHERE c.id = ?
        ";
        $params = [$companyId];

        if ($dateFrom) {
            $sql .= " AND DATE(b.created_at) >= ?";
            $params[] = $dateFrom;
        }

        if ($dateTo) {
            $sql .= " AND DATE(b.created_at) <= ?";
            $params[] = $dateTo;
        }

        $query = $this->db->query($sql);
        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->single();
    }
}
