<?php
/**
 * Cart Model
 * Salon/Spa/Beauty Parlor Booking System
 */

class Cart extends Model {
    protected $table = 'carts';

    /**
     * Get cart by session or user
     */
    public function getCart($sessionId, $userId = null) {
        if ($userId) {
            // Try to get user's cart first
            $cart = $this->db->query("SELECT * FROM {$this->table} WHERE user_id = ? AND status = 'active'")
                            ->bind(1, $userId)->single();
            if ($cart) return $cart;
        }

        // Get cart by session
        return $this->db->query("SELECT * FROM {$this->table} WHERE session_id = ? AND status = 'active'")
                       ->bind(1, $sessionId)->single();
    }

    /**
     * Create new cart
     */
    public function createCart($sessionId, $userId = null) {
        return $this->db->insert($this->table, [
            'session_id' => $sessionId,
            'user_id' => $userId,
            'status' => 'active'
        ]);
    }

    /**
     * Get cart items
     */
    public function getCartItems($cartId) {
        return $this->db->query("SELECT ci.*, p.name, p.slug, p.images, p.stock_quantity, p.price as current_price
                                FROM cart_items ci
                                JOIN products p ON ci.product_id = p.id
                                WHERE ci.cart_id = ?
                                ORDER BY ci.created_at ASC")
                       ->bind(1, $cartId)->resultSet();
    }

    /**
     * Add item to cart
     */
    public function addItem($cartId, $productId, $quantity, $price, $variantId = null) {
        // Check if item already exists
        $existing = $this->db->query("SELECT id, quantity FROM cart_items WHERE cart_id = ? AND product_id = ?")
                            ->bind(1, $cartId)->bind(2, $productId)->single();

        if ($existing) {
            // Update quantity
            $newQuantity = $existing['quantity'] + $quantity;
            $this->db->update('cart_items', ['quantity' => $newQuantity], ['id' => $existing['id']]);
            return $existing['id'];
        } else {
            // Add new item
            return $this->db->insert('cart_items', [
                'cart_id' => $cartId,
                'product_id' => $productId,
                'quantity' => $quantity,
                'price' => $price,
                'variant_id' => $variantId
            ]);
        }
    }

    /**
     * Update cart item quantity
     */
    public function updateItemQuantity($cartId, $itemId, $quantity) {
        if ($quantity <= 0) {
            // Remove item
            return $this->db->query("DELETE FROM cart_items WHERE id = ? AND cart_id = ?")
                           ->bind(1, $itemId)->bind(2, $cartId)->execute();
        } else {
            // Update quantity
            return $this->db->update('cart_items', ['quantity' => $quantity], ['id' => $itemId, 'cart_id' => $cartId]);
        }
    }

    /**
     * Remove item from cart
     */
    public function removeItem($cartId, $itemId) {
        return $this->db->query("DELETE FROM cart_items WHERE id = ? AND cart_id = ?")
                       ->bind(1, $itemId)->bind(2, $cartId)->execute();
    }

    /**
     * Clear cart
     */
    public function clearCart($cartId) {
        $this->db->query("DELETE FROM cart_items WHERE cart_id = ?")->bind(1, $cartId)->execute();
        return $this->db->update($this->table, ['total' => 0], ['id' => $cartId]);
    }

    /**
     * Get cart total
     */
    public function getCartTotal($cartId) {
        return $this->db->query("SELECT SUM(price * quantity) as total FROM cart_items WHERE cart_id = ?")
                       ->bind(1, $cartId)->single()['total'] ?? 0;
    }

    /**
     * Update cart total
     */
    public function updateCartTotal($cartId) {
        $total = $this->getCartTotal($cartId);
        return $this->db->update($this->table, ['total' => $total, 'updated_at' => date('Y-m-d H:i:s')], ['id' => $cartId]);
    }

    /**
     * Get cart item count
     */
    public function getItemCount($cartId) {
        return $this->db->query("SELECT SUM(quantity) as count FROM cart_items WHERE cart_id = ?")
                       ->bind(1, $cartId)->single()['count'] ?? 0;
    }

    /**
     * Transfer cart to user (when user logs in)
     */
    public function transferToUser($sessionId, $userId) {
        // Find session cart
        $sessionCart = $this->db->query("SELECT * FROM {$this->table} WHERE session_id = ? AND status = 'active'")
                               ->bind(1, $sessionId)->single();

        if (!$sessionCart) return false;

        // Check if user already has a cart
        $userCart = $this->db->query("SELECT * FROM {$this->table} WHERE user_id = ? AND status = 'active'")
                            ->bind(1, $userId)->single();

        if ($userCart) {
            // Merge carts - move items from session cart to user cart
            $this->db->query("UPDATE cart_items SET cart_id = ? WHERE cart_id = ?")
                    ->bind(1, $userCart['id'])->bind(2, $sessionCart['id'])->execute();

            // Update user cart total
            $this->updateCartTotal($userCart['id']);

            // Delete session cart
            $this->db->delete($this->table, ['id' => $sessionCart['id']]);
        } else {
            // Assign session cart to user
            $this->db->update($this->table, ['user_id' => $userId], ['id' => $sessionCart['id']]);
        }

        return true;
    }

    /**
     * Validate cart items (check stock, prices)
     */
    public function validateCart($cartId) {
        $items = $this->getCartItems($cartId);
        $errors = [];

        foreach ($items as $item) {
            // Check if product still exists and is active
            if (!$item['name']) {
                $errors[] = "Product no longer available";
                continue;
            }

            // Check stock
            if ($item['stock_quantity'] !== null && $item['stock_quantity'] < $item['quantity']) {
                $errors[] = "Insufficient stock for {$item['name']}";
            }

            // Check if price has changed
            if ($item['current_price'] != $item['price']) {
                // Could update price automatically or add warning
            }
        }

        return $errors;
    }

    /**
     * Convert cart to order
     */
    public function convertToOrder($cartId, $orderData) {
        $items = $this->getCartItems($cartId);
        if (empty($items)) return false;

        $orderItems = [];
        foreach ($items as $item) {
            $orderItems[] = [
                'product_id' => $item['product_id'],
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                'total' => $item['price'] * $item['quantity']
            ];
        }

        $orderModel = new Order();
        $orderId = $orderModel->createOrder($orderData, $orderItems);

        if ($orderId) {
            // Clear the cart
            $this->clearCart($cartId);
        }

        return $orderId;
    }
}
