<?php
/**
 * Booking Model
 * Salon/Spa/Beauty Parlor Booking System
 */

class Booking extends Model {
    protected $table = 'bookings';
    protected $fillable = ['booking_number', 'company_id', 'branch_id', 'customer_id', 'service_id', 'staff_id', 'start_at', 'end_at', 'status', 'total_amount', 'notes', 'customer_notes', 'cancellation_reason', 'cancelled_at', 'cancelled_by'];

    /**
     * Get booking with related data
     */
    public function findWithDetails($id) {
        $sql = "
            SELECT b.*,
                   c.name as company_name, c.slug as company_slug,
                   br.name as branch_name, br.address as branch_address,
                   s.name as service_name, s.slug as service_slug, s.duration_minutes,
                   u.first_name, u.last_name, u.email, u.phone,
                   st.first_name as staff_first_name, st.last_name as staff_last_name,
                   gc.first_name as guest_first_name, gc.last_name as guest_last_name,
                   gc.email as guest_email, gc.phone as guest_phone
            FROM {$this->table} b
            LEFT JOIN companies c ON b.company_id = c.id
            LEFT JOIN branches br ON b.branch_id = br.id
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN users st ON b.staff_id = st.id
            LEFT JOIN guest_customers gc ON b.id = gc.booking_id
            WHERE b.id = ?
        ";
        return $this->db->query($sql)->bind(1, $id)->single();
    }

    /**
     * Get bookings by company
     */
    public function getByCompany($companyId, $filters = []) {
        $sql = "
            SELECT b.*, s.name as service_name, s.duration_minutes,
                   u.first_name, u.last_name, br.name as branch_name,
                   st.first_name as staff_first_name, st.last_name as staff_last_name
            FROM {$this->table} b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN branches br ON b.branch_id = br.id
            LEFT JOIN users st ON b.staff_id = st.id
            WHERE b.company_id = ?
        ";
        $params = [$companyId];

        // Apply filters
        if (!empty($filters['status'])) {
            $sql .= " AND b.status = ?";
            $params[] = $filters['status'];
        }

        if (!empty($filters['branch_id'])) {
            $sql .= " AND b.branch_id = ?";
            $params[] = $filters['branch_id'];
        }

        if (!empty($filters['staff_id'])) {
            $sql .= " AND b.staff_id = ?";
            $params[] = $filters['staff_id'];
        }

        if (!empty($filters['date_from'])) {
            $sql .= " AND DATE(b.start_at) >= ?";
            $params[] = $filters['date_from'];
        }

        if (!empty($filters['date_to'])) {
            $sql .= " AND DATE(b.start_at) <= ?";
            $params[] = $filters['date_to'];
        }

        $sql .= " ORDER BY b.start_at DESC";

        $query = $this->db->query($sql);
        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->resultSet();
    }

    /**
     * Get bookings by customer
     */
    public function getByCustomer($customerId) {
        $sql = "
            SELECT b.*, s.name as service_name, s.duration_minutes,
                   br.name as branch_name, st.first_name as staff_first_name,
                   st.last_name as staff_last_name
            FROM {$this->table} b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN branches br ON b.branch_id = br.id
            LEFT JOIN users st ON b.staff_id = st.id
            WHERE b.customer_id = ?
            ORDER BY b.start_at DESC
        ";
        return $this->db->query($sql)->bind(1, $customerId)->resultSet();
    }

    /**
     * Get bookings by staff
     */
    public function getByStaff($staffId, $date = null) {
        $sql = "
            SELECT b.*, s.name as service_name, s.duration_minutes,
                   u.first_name, u.last_name, br.name as branch_name
            FROM {$this->table} b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN branches br ON b.branch_id = br.id
            WHERE b.staff_id = ?
        ";
        $params = [$staffId];

        if ($date) {
            $sql .= " AND DATE(b.start_at) = ?";
            $params[] = $date;
        }

        $sql .= " ORDER BY b.start_at ASC";

        $query = $this->db->query($sql);
        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->resultSet();
    }

    /**
     * Check availability for booking
     */
    public function checkAvailability($serviceId, $staffId, $startTime, $endTime, $excludeBookingId = null) {
        // Check staff availability
        if ($staffId) {
            $staffAvailable = $this->isStaffAvailable($staffId, $startTime, $endTime);
            if (!$staffAvailable) {
                return ['available' => false, 'reason' => 'Staff not available'];
            }
        }

        // Check for conflicting bookings
        $conflictingBooking = $this->getConflictingBooking($serviceId, $staffId, $startTime, $endTime, $excludeBookingId);
        if ($conflictingBooking) {
            return ['available' => false, 'reason' => 'Time slot already booked'];
        }

        return ['available' => true];
    }

    /**
     * Check if staff is available
     */
    private function isStaffAvailable($staffId, $startTime, $endTime) {
        $dayOfWeek = date('w', strtotime($startTime)); // 0 = Sunday, 1 = Monday, etc.
        $startTimeOnly = date('H:i:s', strtotime($startTime));
        $endTimeOnly = date('H:i:s', strtotime($endTime));

        $availability = $this->db->query("
            SELECT * FROM staff_availability
            WHERE user_id = ? AND day_of_week = ? AND is_available = 1
        ")->bind(1, $staffId)->bind(2, $dayOfWeek)->single();

        if (!$availability) {
            return false;
        }

        // Check if the requested time falls within staff working hours
        return ($startTimeOnly >= $availability['start_time'] && $endTimeOnly <= $availability['end_time']);
    }

    /**
     * Get conflicting booking
     */
    private function getConflictingBooking($serviceId, $staffId, $startTime, $endTime, $excludeBookingId = null) {
        $sql = "
            SELECT * FROM {$this->table}
            WHERE service_id = ? AND status NOT IN ('cancelled', 'no_show')
            AND (
                (start_at < ? AND end_at > ?) OR
                (start_at < ? AND end_at > ?) OR
                (start_at >= ? AND end_at <= ?)
            )
        ";
        $params = [$serviceId, $endTime, $startTime, $startTime, $endTime, $startTime, $endTime];

        if ($staffId) {
            $sql .= " AND staff_id = ?";
            $params[] = $staffId;
        }

        if ($excludeBookingId) {
            $sql .= " AND id != ?";
            $params[] = $excludeBookingId;
        }

        $query = $this->db->query($sql);
        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->single();
    }

    /**
     * Create booking with conflict checking
     */
    public function createBooking($data) {
        // Start transaction
        $this->db->beginTransaction();

        try {
            // Get service details
            $service = $this->db->query("SELECT * FROM services WHERE id = ?")
                               ->bind(1, $data['service_id'])
                               ->single();

            if (!$service) {
                throw new Exception('Service not found');
            }

            // Calculate end time
            $startTime = $data['start_at'];
            $endTime = date('Y-m-d H:i:s', strtotime($startTime) + ($service['duration_minutes'] * 60));

            // Check availability
            $availability = $this->checkAvailability(
                $data['service_id'],
                $data['staff_id'] ?? null,
                $startTime,
                $endTime
            );

            if (!$availability['available']) {
                throw new Exception($availability['reason']);
            }

            // Generate booking number
            $data['booking_number'] = generateBookingNumber();
            $data['end_at'] = $endTime;
            $data['total_amount'] = $service['price'];

            // Create booking
            $bookingId = $this->create($data);

            // If guest booking, create guest customer record
            if (empty($data['customer_id']) && !empty($data['guest_details'])) {
                $this->db->insert('guest_customers', array_merge(
                    $data['guest_details'],
                    ['booking_id' => $bookingId]
                ));
            }

            $this->db->commit();
            return $bookingId;

        } catch (Exception $e) {
            $this->db->rollback();
            throw $e;
        }
    }

    /**
     * Update booking status
     */
    public function updateStatus($bookingId, $status, $userId = null) {
        $data = ['status' => $status];

        if ($status === 'cancelled') {
            $data['cancelled_at'] = date('Y-m-d H:i:s');
            $data['cancelled_by'] = $userId;
        }

        return $this->update($bookingId, $data);
    }

    /**
     * Get today's bookings for staff
     */
    public function getTodaysBookings($staffId) {
        $today = date('Y-m-d');
        return $this->getByStaff($staffId, $today);
    }

    /**
     * Get upcoming bookings
     */
    public function getUpcomingBookings($companyId, $limit = 10) {
        $sql = "
            SELECT b.*, s.name as service_name,
                   u.first_name, u.last_name,
                   st.first_name as staff_first_name, st.last_name as staff_last_name
            FROM {$this->table} b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN users st ON b.staff_id = st.id
            WHERE b.company_id = ? AND b.start_at > NOW() AND b.status = 'confirmed'
            ORDER BY b.start_at ASC
            LIMIT ?
        ";
        return $this->db->query($sql)
                       ->bind(1, $companyId)
                       ->bind(2, $limit)
                       ->resultSet();
    }

    /**
     * Get booking statistics
     */
    public function getStats($companyId, $dateFrom = null, $dateTo = null) {
        $sql = "
            SELECT
                COUNT(*) as total_bookings,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_bookings,
                SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled_bookings,
                SUM(CASE WHEN status = 'no_show' THEN 1 ELSE 0 END) as no_show_bookings,
                SUM(total_amount) as total_revenue
            FROM {$this->table}
            WHERE company_id = ?
        ";
        $params = [$companyId];

        if ($dateFrom) {
            $sql .= " AND DATE(start_at) >= ?";
            $params[] = $dateFrom;
        }

        if ($dateTo) {
            $sql .= " AND DATE(start_at) <= ?";
            $params[] = $dateTo;
        }

        $query = $this->db->query($sql);
        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->single();
    }
}
