<?php
/**
 * Base Model Class
 * Salon/Spa/Beauty Parlor Booking System
 */

abstract class Model {
    protected $db;
    protected $table;
    protected $primaryKey = 'id';
    protected $fillable = [];
    protected $hidden = [];
    protected $timestamps = true;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Find record by ID
     */
    public function find($id) {
        $sql = "SELECT * FROM {$this->table} WHERE {$this->primaryKey} = ?";
        return $this->db->query($sql)->bind(1, $id)->single();
    }

    /**
     * Find record by conditions
     */
    public function findBy($conditions, $single = true) {
        $whereParts = [];
        $params = [];
        $i = 1;

        foreach ($conditions as $key => $value) {
            $whereParts[] = "{$key} = ?";
            $params[] = $value;
        }

        $whereClause = implode(' AND ', $whereParts);
        $sql = "SELECT * FROM {$this->table} WHERE {$whereClause}";

        if ($single) {
            return $this->db->query($sql)->single();
        } else {
            return $this->db->query($sql)->resultSet();
        }
    }

    /**
     * Get all records
     */
    public function all($orderBy = null, $limit = null) {
        $sql = "SELECT * FROM {$this->table}";

        if ($orderBy) {
            $sql .= " ORDER BY {$orderBy}";
        }

        if ($limit) {
            $sql .= " LIMIT {$limit}";
        }

        return $this->db->query($sql)->resultSet();
    }

    /**
     * Get records with pagination
     */
    public function paginate($page = 1, $perPage = ITEMS_PER_PAGE, $where = null, $orderBy = null) {
        $offset = ($page - 1) * $perPage;

        $sql = "SELECT * FROM {$this->table}";
        $countSql = "SELECT COUNT(*) as total FROM {$this->table}";

        $params = [];
        $paramIndex = 1;

        if ($where) {
            $whereParts = [];
            foreach ($where as $key => $value) {
                $whereParts[] = "{$key} = ?";
                $params[] = $value;
            }
            $whereClause = ' WHERE ' . implode(' AND ', $whereParts);
            $sql .= $whereClause;
            $countSql .= $whereClause;
        }

        if ($orderBy) {
            $sql .= " ORDER BY {$orderBy}";
        }

        $sql .= " LIMIT ? OFFSET ?";
        $params[] = $perPage;
        $params[] = $offset;

        // Get total count
        $totalResult = $this->db->query($countSql);
        foreach ($params as $index => $param) {
            if ($index < count($params) - 2) { // Exclude LIMIT and OFFSET params
                $totalResult->bind($index + 1, $param);
            }
        }
        $total = $totalResult->single()['total'];

        // Get records
        $query = $this->db->query($sql);
        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }
        $data = $query->resultSet();

        return [
            'data' => $data,
            'total' => $total,
            'per_page' => $perPage,
            'current_page' => $page,
            'last_page' => ceil($total / $perPage),
            'from' => $offset + 1,
            'to' => min($offset + $perPage, $total)
        ];
    }

    /**
     * Create new record
     */
    public function create($data) {
        // Filter fillable fields
        if (!empty($this->fillable)) {
            $data = array_intersect_key($data, array_flip($this->fillable));
        }

        // Add timestamps
        if ($this->timestamps) {
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['updated_at'] = date('Y-m-d H:i:s');
        }

        $this->db->insert($this->table, $data);
        return $this->db->lastInsertId();
    }

    /**
     * Update record
     */
    public function update($id, $data) {
        // Filter fillable fields
        if (!empty($this->fillable)) {
            $data = array_intersect_key($data, array_flip($this->fillable));
        }

        // Add updated timestamp
        if ($this->timestamps) {
            $data['updated_at'] = date('Y-m-d H:i:s');
        }

        $where = [$this->primaryKey => $id];
        return $this->db->update($this->table, $data, $where);
    }

    /**
     * Delete record
     */
    public function delete($id) {
        $where = [$this->primaryKey => $id];
        return $this->db->delete($this->table, $where);
    }

    /**
     * Check if record exists
     */
    public function exists($id) {
        $sql = "SELECT COUNT(*) as count FROM {$this->table} WHERE {$this->primaryKey} = ?";
        $result = $this->db->query($sql)->bind(1, $id)->single();
        return $result['count'] > 0;
    }

    /**
     * Count records
     */
    public function count($where = null) {
        $sql = "SELECT COUNT(*) as count FROM {$this->table}";

        if ($where) {
            $whereParts = [];
            $params = [];
            $i = 1;

            foreach ($where as $key => $value) {
                $whereParts[] = "{$key} = ?";
                $params[] = $value;
            }

            $sql .= ' WHERE ' . implode(' AND ', $whereParts);
        }

        $query = $this->db->query($sql);

        if ($where) {
            foreach ($params as $index => $param) {
                $query->bind($index + 1, $param);
            }
        }

        $result = $query->single();
        return $result['count'];
    }

    /**
     * Get records with custom query
     */
    public function query($sql, $params = []) {
        $query = $this->db->query($sql);

        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->resultSet();
    }

    /**
     * Get single record with custom query
     */
    public function querySingle($sql, $params = []) {
        $query = $this->db->query($sql);

        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->single();
    }

    /**
     * Hide sensitive fields from result
     */
    protected function hideFields($data) {
        if (is_array($data) && !empty($this->hidden)) {
            foreach ($data as &$record) {
                if (is_array($record)) {
                    foreach ($this->hidden as $field) {
                        unset($record[$field]);
                    }
                }
            }
        }
        return $data;
    }

    /**
     * Get database instance
     */
    protected function getDb() {
        return $this->db;
    }
}
