<?php
/**
 * Staff Schedule Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class ScheduleController extends Controller {
    public function index() {
        $staffId = $this->auth->id();

        // Get current week
        $weekStart = $this->getQuery('week_start', date('Y-m-d', strtotime('monday this week')));
        $weekEnd = date('Y-m-d', strtotime($weekStart . ' +6 days'));

        // Get staff availability
        $availability = $this->db->query("
            SELECT * FROM staff_availability
            WHERE user_id = ?
            ORDER BY day_of_week ASC, start_time ASC
        ")->bind(1, $staffId)->resultSet();

        // Get bookings for the week
        $bookings = $this->getWeeklyBookings($staffId, $weekStart, $weekEnd);

        // Get time off/leave
        $timeOff = $this->db->query("
            SELECT * FROM staff_leave
            WHERE user_id = ? AND (
                (start_date BETWEEN ? AND ?) OR
                (end_date BETWEEN ? AND ?) OR
                (start_date <= ? AND end_date >= ?)
            ) AND status = 'approved'
            ORDER BY start_date ASC
        ")->bind(1, $staffId)->bind(2, $weekStart)->bind(3, $weekEnd)->bind(4, $weekStart)->bind(5, $weekEnd)->bind(6, $weekStart)->bind(7, $weekEnd)->resultSet();

        $this->staffView('schedule/index', [
            'weekStart' => $weekStart,
            'weekEnd' => $weekEnd,
            'availability' => $availability,
            'bookings' => $bookings,
            'timeOff' => $timeOff,
            'pageTitle' => 'My Schedule'
        ]);
    }

    public function availability() {
        $staffId = $this->auth->id();

        // Get current availability
        $availability = $this->db->query("
            SELECT * FROM staff_availability
            WHERE user_id = ?
            ORDER BY day_of_week ASC, start_time ASC
        ")->bind(1, $staffId)->resultSet();

        // Group by day
        $availabilityByDay = [];
        foreach ($availability as $slot) {
            $day = $slot['day_of_week'];
            if (!isset($availabilityByDay[$day])) {
                $availabilityByDay[$day] = [];
            }
            $availabilityByDay[$day][] = $slot;
        }

        $this->staffView('schedule/availability', [
            'availabilityByDay' => $availabilityByDay,
            'pageTitle' => 'My Availability'
        ]);
    }

    public function updateAvailability() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['error' => 'Method not allowed'], 405);
        }

        $staffId = $this->auth->id();
        $branchId = $this->auth->branchId();

        try {
            // Delete existing availability
            $this->db->query("DELETE FROM staff_availability WHERE user_id = ?")->bind(1, $staffId)->execute();

            // Add new availability
            $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];

            foreach ($days as $dayIndex => $dayName) {
                $enabled = $this->getPost("{$dayName}_enabled");
                $startTime = $this->getPost("{$dayName}_start");
                $endTime = $this->getPost("{$dayName}_end");
                $breakStart = $this->getPost("{$dayName}_break_start");
                $breakEnd = $this->getPost("{$dayName}_break_end");

                if ($enabled && $startTime && $endTime) {
                    // Validate time format
                    if (!$this->isValidTime($startTime) || !$this->isValidTime($endTime)) {
                        continue;
                    }

                    // Validate that end time is after start time
                    if (strtotime($endTime) <= strtotime($startTime)) {
                        continue;
                    }

                    $this->db->insert('staff_availability', [
                        'user_id' => $staffId,
                        'branch_id' => $branchId,
                        'day_of_week' => $dayIndex,
                        'start_time' => $startTime . ':00',
                        'end_time' => $endTime . ':00',
                        'break_start' => $breakStart ? $breakStart . ':00' : null,
                        'break_end' => $breakEnd ? $breakEnd . ':00' : null,
                        'is_available' => true
                    ]);
                }
            }

            $this->json([
                'success' => true,
                'message' => 'Availability updated successfully'
            ]);

        } catch (Exception $e) {
            error_log("Availability update error: " . $e->getMessage());
            $this->json(['error' => 'Failed to update availability'], 500);
        }
    }

    public function requestTimeOff() {
        $staffId = $this->auth->id();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $startDate = $this->getPost('start_date');
            $endDate = $this->getPost('end_date');
            $reason = $this->getPost('reason');

            // Validate input
            $errors = [];
            if (empty($startDate)) $errors['start_date'] = 'Start date is required';
            if (empty($endDate)) $errors['end_date'] = 'End date is required';
            if (strtotime($endDate) < strtotime($startDate)) $errors['end_date'] = 'End date must be after start date';

            if (!empty($errors)) {
                $_SESSION['form_errors'] = $errors;
                $_SESSION['form_data'] = $_POST;
                $this->redirect(BASE_URL . '/staff/schedule/time-off');
            }

            try {
                $this->db->insert('staff_leave', [
                    'user_id' => $staffId,
                    'branch_id' => $this->auth->branchId(),
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                    'reason' => $reason,
                    'status' => 'pending'
                ]);

                $this->redirect(BASE_URL . '/staff/schedule/time-off', 'Time off request submitted successfully!');

            } catch (Exception $e) {
                error_log("Time off request error: " . $e->getMessage());
                $_SESSION['form_errors'] = ['general' => 'Failed to submit time off request'];
                $_SESSION['form_data'] = $_POST;
                $this->redirect(BASE_URL . '/staff/schedule/time-off');
            }
        }

        // Get existing time off requests
        $timeOffRequests = $this->db->query("
            SELECT * FROM staff_leave
            WHERE user_id = ?
            ORDER BY created_at DESC
        ")->bind(1, $staffId)->resultSet();

        $this->staffView('schedule/time-off', [
            'timeOffRequests' => $timeOffRequests,
            'pageTitle' => 'Request Time Off'
        ]);
    }

    public function calendar() {
        $staffId = $this->auth->id();
        $month = $this->getQuery('month', date('m'));
        $year = $this->getQuery('year', date('Y'));

        // Get bookings for the month
        $startDate = date('Y-m-01', strtotime("{$year}-{$month}-01"));
        $endDate = date('Y-m-t', strtotime("{$year}-{$month}-01"));

        $bookings = $this->db->query("
            SELECT DATE(start_at) as date, COUNT(*) as booking_count,
                   GROUP_CONCAT(DISTINCT status) as statuses
            FROM bookings
            WHERE staff_id = ? AND start_at BETWEEN ? AND ?
            GROUP BY DATE(start_at)
        ")->bind(1, $staffId)->bind(2, $startDate)->bind(3, $endDate)->resultSet();

        // Convert to array keyed by date
        $bookingsByDate = [];
        foreach ($bookings as $booking) {
            $bookingsByDate[$booking['date']] = $booking;
        }

        $this->staffView('schedule/calendar', [
            'month' => $month,
            'year' => $year,
            'bookingsByDate' => $bookingsByDate,
            'pageTitle' => 'Schedule Calendar'
        ]);
    }

    public function conflicts() {
        $staffId = $this->auth->id();
        $date = $this->getQuery('date', date('Y-m-d'));

        // Check for scheduling conflicts
        $conflicts = $this->checkScheduleConflicts($staffId, $date);

        // Get potential conflicts (overlapping bookings)
        $overlappingBookings = $this->getOverlappingBookings($staffId, $date);

        $this->staffView('schedule/conflicts', [
            'date' => $date,
            'conflicts' => $conflicts,
            'overlappingBookings' => $overlappingBookings,
            'pageTitle' => 'Schedule Conflicts'
        ]);
    }

    /**
     * Get weekly bookings
     */
    private function getWeeklyBookings($staffId, $startDate, $endDate) {
        $bookings = $this->db->query("
            SELECT b.*, s.name as service_name, s.duration_minutes,
                   u.first_name, u.last_name, u.phone,
                   br.name as branch_name
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN branches br ON b.branch_id = br.id
            WHERE b.staff_id = ? AND DATE(b.start_at) BETWEEN ? AND ?
            ORDER BY b.start_at ASC
        ")->bind(1, $staffId)->bind(2, $startDate)->bind(3, $endDate)->resultSet();

        // Group by date
        $bookingsByDate = [];
        foreach ($bookings as $booking) {
            $date = date('Y-m-d', strtotime($booking['start_at']));
            if (!isset($bookingsByDate[$date])) {
                $bookingsByDate[$date] = [];
            }
            $bookingsByDate[$date][] = $booking;
        }

        return $bookingsByDate;
    }

    /**
     * Check schedule conflicts
     */
    private function checkScheduleConflicts($staffId, $date) {
        $conflicts = [];

        // Get staff availability for the day
        $dayOfWeek = date('w', strtotime($date));
        $availability = $this->db->query("
            SELECT * FROM staff_availability
            WHERE user_id = ? AND day_of_week = ? AND is_available = 1
        ")->bind(1, $staffId)->bind(2, $dayOfWeek)->resultSet();

        if (empty($availability)) {
            $conflicts[] = [
                'type' => 'no_availability',
                'message' => 'No availability scheduled for this day'
            ];
        }

        // Check for approved time off
        $timeOff = $this->db->query("
            SELECT * FROM staff_leave
            WHERE user_id = ? AND ? BETWEEN start_date AND end_date AND status = 'approved'
        ")->bind(1, $staffId)->bind(2, $date)->single();

        if ($timeOff) {
            $conflicts[] = [
                'type' => 'time_off',
                'message' => 'Approved time off: ' . $timeOff['reason'],
                'time_off' => $timeOff
            ];
        }

        return $conflicts;
    }

    /**
     * Get overlapping bookings
     */
    private function getOverlappingBookings($staffId, $date) {
        $bookings = $this->db->query("
            SELECT b1.*, s1.name as service_name,
                   b2.id as overlapping_id, s2.name as overlapping_service
            FROM bookings b1
            JOIN bookings b2 ON b1.id != b2.id
                AND b1.staff_id = b2.staff_id
                AND DATE(b1.start_at) = DATE(b2.start_at)
                AND b1.start_at < b2.end_at
                AND b1.end_at > b2.start_at
            LEFT JOIN services s1 ON b1.service_id = s1.id
            LEFT JOIN services s2 ON b2.service_id = s2.id
            WHERE b1.staff_id = ? AND DATE(b1.start_at) = ?
            ORDER BY b1.start_at ASC
        ")->bind(1, $staffId)->bind(2, $date)->resultSet();

        return $bookings;
    }

    /**
     * Validate time format
     */
    private function isValidTime($time) {
        return preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $time);
    }
}
