<?php
/**
 * Staff Dashboard Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class DashboardController extends Controller {
    public function index() {
        $staffId = $this->auth->id();
        $branchId = $this->auth->branchId();

        // Get today's schedule
        $today = date('Y-m-d');
        $todaySchedule = $this->getTodaySchedule($staffId, $today);

        // Get upcoming bookings
        $upcomingBookings = $this->getUpcomingBookings($staffId, 10);

        // Get today's stats
        $todayStats = $this->getTodayStats($staffId, $today);

        // Get weekly schedule overview
        $weeklySchedule = $this->getWeeklySchedule($staffId);

        // Get recent completed bookings
        $recentCompleted = $this->getRecentCompletedBookings($staffId, 5);

        $this->staffView('dashboard/index', [
            'todaySchedule' => $todaySchedule,
            'upcomingBookings' => $upcomingBookings,
            'todayStats' => $todayStats,
            'weeklySchedule' => $weeklySchedule,
            'recentCompleted' => $recentCompleted,
            'pageTitle' => 'Staff Dashboard'
        ]);
    }

    public function markAttendance() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['error' => 'Method not allowed'], 405);
        }

        $staffId = $this->auth->id();
        $action = $this->getPost('action'); // 'clock_in' or 'clock_out'
        $notes = $this->getPost('notes');

        try {
            $today = date('Y-m-d');

            if ($action === 'clock_in') {
                // Check if already clocked in today
                $existing = $this->db->query("
                    SELECT id FROM attendance
                    WHERE user_id = ? AND DATE(date) = ? AND clock_out IS NULL
                ")->bind(1, $staffId)->bind(2, $today)->single();

                if ($existing) {
                    $this->json(['error' => 'Already clocked in for today'], 400);
                }

                // Clock in
                $this->db->insert('attendance', [
                    'user_id' => $staffId,
                    'date' => date('Y-m-d H:i:s'),
                    'clock_in' => date('Y-m-d H:i:s'),
                    'notes' => $notes
                ]);

                $this->json([
                    'success' => true,
                    'message' => 'Successfully clocked in',
                    'timestamp' => date('H:i:s')
                ]);

            } elseif ($action === 'clock_out') {
                // Find today's attendance record
                $attendance = $this->db->query("
                    SELECT * FROM attendance
                    WHERE user_id = ? AND DATE(date) = ? AND clock_out IS NULL
                    ORDER BY id DESC LIMIT 1
                ")->bind(1, $staffId)->bind(2, $today)->single();

                if (!$attendance) {
                    $this->json(['error' => 'No active clock-in found for today'], 400);
                }

                // Calculate hours worked
                $clockIn = strtotime($attendance['clock_in']);
                $clockOut = time();
                $hoursWorked = ($clockOut - $clockIn) / 3600;

                // Clock out
                $this->db->update('attendance', [
                    'clock_out' => date('Y-m-d H:i:s'),
                    'hours_worked' => round($hoursWorked, 2),
                    'notes' => $attendance['notes'] . ($notes ? "\n" . $notes : '')
                ], ['id' => $attendance['id']]);

                $this->json([
                    'success' => true,
                    'message' => 'Successfully clocked out',
                    'hours_worked' => round($hoursWorked, 2),
                    'timestamp' => date('H:i:s')
                ]);

            } else {
                $this->json(['error' => 'Invalid action'], 400);
            }

        } catch (Exception $e) {
            error_log("Attendance marking error: " . $e->getMessage());
            $this->json(['error' => 'Failed to mark attendance'], 500);
        }
    }

    public function getAttendanceStatus() {
        $staffId = $this->auth->id();
        $today = date('Y-m-d');

        $attendance = $this->db->query("
            SELECT * FROM attendance
            WHERE user_id = ? AND DATE(date) = ?
            ORDER BY id DESC LIMIT 1
        ")->bind(1, $staffId)->bind(2, $today)->single();

        $status = [
            'is_clocked_in' => false,
            'clock_in_time' => null,
            'clock_out_time' => null,
            'hours_worked' => 0,
            'can_clock_in' => true,
            'can_clock_out' => false
        ];

        if ($attendance) {
            $status['is_clocked_in'] = $attendance['clock_out'] === null;
            $status['clock_in_time'] = $attendance['clock_in'] ? date('H:i', strtotime($attendance['clock_in'])) : null;
            $status['clock_out_time'] = $attendance['clock_out'] ? date('H:i', strtotime($attendance['clock_out'])) : null;
            $status['hours_worked'] = $attendance['hours_worked'] ?? 0;

            if ($status['is_clocked_in']) {
                $status['can_clock_out'] = true;
                $status['can_clock_in'] = false;
            } else {
                $status['can_clock_in'] = false;
                $status['can_clock_out'] = false;
            }
        }

        $this->json([
            'success' => true,
            'status' => $status
        ]);
    }

    /**
     * Get today's schedule for staff
     */
    private function getTodaySchedule($staffId, $date) {
        return $this->db->query("
            SELECT b.*, s.name as service_name, s.duration_minutes,
                   u.first_name, u.last_name, u.phone,
                   br.name as branch_name, br.phone as branch_phone
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN branches br ON b.branch_id = br.id
            WHERE b.staff_id = ? AND DATE(b.start_at) = ? AND b.status IN ('confirmed', 'pending')
            ORDER BY b.start_at ASC
        ")->bind(1, $staffId)->bind(2, $date)->resultSet();
    }

    /**
     * Get upcoming bookings
     */
    private function getUpcomingBookings($staffId, $limit = 10) {
        return $this->db->query("
            SELECT b.*, s.name as service_name, s.duration_minutes,
                   u.first_name, u.last_name, u.phone,
                   br.name as branch_name
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN branches br ON b.branch_id = br.id
            WHERE b.staff_id = ? AND b.start_at > NOW() AND b.status IN ('confirmed', 'pending')
            ORDER BY b.start_at ASC
            LIMIT ?
        ")->bind(1, $staffId)->bind(2, $limit)->resultSet();
    }

    /**
     * Get today's stats
     */
    private function getTodayStats($staffId, $date) {
        $stats = $this->db->query("
            SELECT
                COUNT(*) as total_bookings,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_bookings,
                COUNT(CASE WHEN status = 'confirmed' THEN 1 END) as confirmed_bookings,
                COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_bookings,
                COALESCE(SUM(CASE WHEN status = 'completed' THEN total_amount END), 0) as revenue
            FROM bookings
            WHERE staff_id = ? AND DATE(start_at) = ?
        ")->bind(1, $staffId)->bind(2, $date)->single();

        // Get attendance info
        $attendance = $this->db->query("
            SELECT * FROM attendance
            WHERE user_id = ? AND DATE(date) = ?
            ORDER BY id DESC LIMIT 1
        ")->bind(1, $staffId)->bind(2, $date)->single();

        $stats['attendance'] = $attendance;

        return $stats;
    }

    /**
     * Get weekly schedule overview
     */
    private function getWeeklySchedule($staffId) {
        $schedule = [];

        // Get next 7 days
        for ($i = 0; $i < 7; $i++) {
            $date = date('Y-m-d', strtotime("+{$i} days"));
            $dayName = date('l', strtotime($date));

            // Check if staff is available on this day
            $availability = $this->db->query("
                SELECT * FROM staff_availability
                WHERE user_id = ? AND day_of_week = ? AND is_available = 1
            ")->bind(1, $staffId)->bind(2, date('w', strtotime($date)))->single();

            // Get bookings for this day
            $bookings = $this->db->query("
                SELECT COUNT(*) as count FROM bookings
                WHERE staff_id = ? AND DATE(start_at) = ? AND status IN ('confirmed', 'pending')
            ")->bind(1, $staffId)->bind(2, $date)->single()['count'];

            $schedule[] = [
                'date' => $date,
                'day' => $dayName,
                'is_available' => !empty($availability),
                'booking_count' => $bookings,
                'availability' => $availability
            ];
        }

        return $schedule;
    }

    /**
     * Get recent completed bookings
     */
    private function getRecentCompletedBookings($staffId, $limit = 5) {
        return $this->db->query("
            SELECT b.*, s.name as service_name,
                   u.first_name, u.last_name,
                   COALESCE(SUM(b.total_amount), 0) as revenue
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            WHERE b.staff_id = ? AND b.status = 'completed'
            GROUP BY b.id
            ORDER BY b.updated_at DESC
            LIMIT ?
        ")->bind(1, $staffId)->bind(2, $limit)->resultSet();
    }
}
