<?php
/**
 * Service Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class ServiceController extends Controller {
    public function index() {
        // Get company info
        $company = $this->db->query("SELECT * FROM companies WHERE status = 'active' LIMIT 1")->single();

        if (!$company) {
            $this->view('setup');
            return;
        }

        // Get services with optional category filtering
        $query = "
            SELECT s.*, c.name as category_name, c.slug as category_slug
            FROM services s
            LEFT JOIN service_categories c ON s.category_id = c.id
            WHERE s.company_id = ? AND s.is_active = 1
        ";

        $params = [$company['id']];

        if (isset($_GET['category']) && !empty($_GET['category'])) {
            $query .= " AND c.slug = ?";
            $params[] = $_GET['category'];
        }

        $query .= " ORDER BY c.display_order ASC, s.name ASC";

        $services = $this->db->query($query);
        foreach ($params as $index => $param) {
            $services->bind($index + 1, $param);
        }
        $services = $services->resultSet();

        // Get all categories for filtering
        $categories = $this->db->query("
            SELECT * FROM service_categories
            WHERE (company_id = ? OR company_id IS NULL) AND is_active = 1
            ORDER BY display_order ASC, name ASC
        ")->bind(1, $company['id'])->resultSet();

        $this->view('services/index', [
            'company' => $company,
            'services' => $services,
            'categories' => $categories
        ]);
    }

    public function show($slug) {
        // Get company info
        $company = $this->db->query("SELECT * FROM companies WHERE status = 'active' LIMIT 1")->single();

        if (!$company) {
            $this->view('setup');
            return;
        }

        // Get service details
        $service = $this->db->query("
            SELECT s.*, c.name as category_name, c.slug as category_slug
            FROM services s
            LEFT JOIN service_categories c ON s.category_id = c.id
            WHERE s.slug = ? AND s.company_id = ? AND s.is_active = 1
        ")->bind(1, $slug)->bind(2, $company['id'])->single();

        if (!$service) {
            http_response_code(404);
            $this->view('errors/404');
            return;
        }

        // Get related services (same category)
        $relatedServices = $this->db->query("
            SELECT s.*, c.name as category_name
            FROM services s
            LEFT JOIN service_categories c ON s.category_id = c.id
            WHERE s.category_id = ? AND s.id != ? AND s.company_id = ? AND s.is_active = 1
            ORDER BY s.name ASC
            LIMIT 3
        ")->bind(1, $service['category_id'])
          ->bind(2, $service['id'])
          ->bind(3, $company['id'])
          ->resultSet();

        // Get service variants/add-ons if any
        $variants = $this->db->query("
            SELECT * FROM service_variants
            WHERE service_id = ? AND is_active = 1
            ORDER BY price ASC
        ")->bind(1, $service['id'])->resultSet();

        // Get branches that offer this service
        $branches = $this->db->query("
            SELECT DISTINCT b.* FROM branches b
            JOIN staff_services ss ON ss.user_id IN (
                SELECT user_id FROM staff_services WHERE service_id = ?
            )
            WHERE b.company_id = ? AND b.status = 'active'
        ")->bind(1, $service['id'])->bind(2, $company['id'])->resultSet();

        // Mock gallery images (in real app, these would come from database)
        $gallery = [
            $service['image'],
            '/assets/images/services/gallery-1.jpg',
            '/assets/images/services/gallery-2.jpg',
            '/assets/images/services/gallery-3.jpg'
        ];

        // Mock reviews/testimonials (in real app, these would come from database)
        $reviews = [
            [
                'name' => 'Sarah Johnson',
                'rating' => 5,
                'comment' => 'Amazing service! The results exceeded my expectations.',
                'date' => '2024-01-15'
            ],
            [
                'name' => 'Mike Chen',
                'rating' => 5,
                'comment' => 'Professional and skilled. Highly recommend!',
                'date' => '2024-01-10'
            ]
        ];

        // SEO data
        $seoTitle = $service['name'] . ' - ' . $company['name'];
        $seoDescription = substr($service['short_description'] ?? $service['description'], 0, 155) . '...';
        $seoImage = $service['image'];

        $this->view('services/show', [
            'company' => $company,
            'service' => $service,
            'relatedServices' => $relatedServices,
            'variants' => $variants,
            'branches' => $branches,
            'gallery' => $gallery,
            'reviews' => $reviews,
            'seoTitle' => $seoTitle,
            'seoDescription' => $seoDescription,
            'seoImage' => $seoImage
        ]);
    }
}