<?php
/**
 * Booking Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class BookingController extends Controller {
    public function create() {
        // Get company info
        $company = $this->db->query("SELECT * FROM companies WHERE status = 'active' LIMIT 1")->single();

        if (!$company) {
            $this->view('setup');
            return;
        }

        // Get branches
        $branches = $this->db->query("
            SELECT * FROM branches
            WHERE company_id = ? AND status = 'active'
            ORDER BY is_main_branch DESC, name ASC
        ")->bind(1, $company['id'])->resultSet();

        // Get services
        $services = $this->db->query("
            SELECT s.*, c.name as category_name
            FROM services s
            LEFT JOIN service_categories c ON s.category_id = c.id
            WHERE s.company_id = ? AND s.is_active = 1
            ORDER BY c.display_order ASC, s.name ASC
        ")->bind(1, $company['id'])->resultSet();

        // Check if there's a pre-selected service from URL
        $selectedService = null;
        if (isset($_GET['service'])) {
            $selectedService = $this->db->query("
                SELECT s.*, c.name as category_name
                FROM services s
                LEFT JOIN service_categories c ON s.category_id = c.id
                WHERE s.slug = ? AND s.company_id = ? AND s.is_active = 1
            ")->bind(1, $_GET['service'])->bind(2, $company['id'])->single();
        }

        $this->view('booking/create', [
            'company' => $company,
            'branches' => $branches,
            'services' => $services,
            'selectedService' => $selectedService
        ]);
    }

    public function store() {
        // Validate CSRF token
        if (!isset($_POST[CSRF_TOKEN_NAME]) || !verifyCsrfToken($_POST[CSRF_TOKEN_NAME])) {
            $this->json(['error' => 'Invalid CSRF token'], 403);
            return;
        }

        // Validate required fields
        $required = ['branch_id', 'service_id', 'booking_date', 'start_time', 'first_name', 'last_name', 'email', 'phone'];
        foreach ($required as $field) {
            if (empty($_POST[$field])) {
                $this->json(['error' => "Missing required field: {$field}"], 400);
                return;
            }
        }

        // Get service details
        $service = $this->db->query("SELECT * FROM services WHERE id = ? AND is_active = 1")
                          ->bind(1, $_POST['service_id'])->single();

        if (!$service) {
            $this->json(['error' => 'Service not found'], 404);
            return;
        }

        // Get branch details
        $branch = $this->db->query("SELECT * FROM branches WHERE id = ? AND status = 'active'")
                          ->bind(1, $_POST['branch_id'])->single();

        if (!$branch) {
            $this->json(['error' => 'Branch not found'], 404);
            return;
        }

        // Parse date and time
        $bookingDate = $_POST['booking_date'];
        $startTime = $_POST['start_time'];
        $startDateTime = new DateTime("{$bookingDate} {$startTime}");
        $endDateTime = clone $startDateTime;
        $endDateTime->add(new DateInterval("PT{$service['duration_minutes']}M"));

        // Check for booking conflicts
        $conflict = $this->db->query("
            SELECT COUNT(*) as count FROM bookings
            WHERE branch_id = ?
            AND status IN ('pending', 'confirmed')
            AND (
                (start_at <= ? AND end_at > ?) OR
                (start_at < ? AND end_at >= ?) OR
                (start_at >= ? AND end_at <= ?)
            )
        ")->bind(1, $branch['id'])
          ->bind(2, $startDateTime->format('Y-m-d H:i:s'))
          ->bind(3, $startDateTime->format('Y-m-d H:i:s'))
          ->bind(4, $endDateTime->format('Y-m-d H:i:s'))
          ->bind(5, $endDateTime->format('Y-m-d H:i:s'))
          ->bind(6, $startDateTime->format('Y-m-d H:i:s'))
          ->bind(7, $endDateTime->format('Y-m-d H:i:s'))
          ->single();

        if ($conflict['count'] > 0) {
            $this->json(['error' => 'This time slot is no longer available'], 409);
            return;
        }

        // Check staff availability if staff is selected
        if (!empty($_POST['staff_id'])) {
            $staffConflict = $this->db->query("
                SELECT COUNT(*) as count FROM bookings
                WHERE staff_id = ?
                AND status IN ('pending', 'confirmed')
                AND (
                    (start_at <= ? AND end_at > ?) OR
                    (start_at < ? AND end_at >= ?) OR
                    (start_at >= ? AND end_at <= ?)
                )
            ")->bind(1, $_POST['staff_id'])
              ->bind(2, $startDateTime->format('Y-m-d H:i:s'))
              ->bind(3, $startDateTime->format('Y-m-d H:i:s'))
              ->bind(4, $endDateTime->format('Y-m-d H:i:s'))
              ->bind(5, $endDateTime->format('Y-m-d H:i:s'))
              ->bind(6, $startDateTime->format('Y-m-d H:i:s'))
              ->bind(7, $endDateTime->format('Y-m-d H:i:s'))
              ->single();

            if ($staffConflict['count'] > 0) {
                $this->json(['error' => 'Selected staff is not available at this time'], 409);
                return;
            }
        }

        // Start transaction
        $this->db->beginTransaction();

        try {
            // Check if customer exists or create guest customer
            $customerId = null;
            if (Auth::check()) {
                $customerId = Auth::user()['id'];
            } else {
                // Create guest customer record
                $this->db->query("
                    INSERT INTO guest_customers (first_name, last_name, email, phone, created_at)
                    VALUES (?, ?, ?, ?, NOW())
                ")->bind(1, $_POST['first_name'])
                  ->bind(2, $_POST['last_name'])
                  ->bind(3, $_POST['email'])
                  ->bind(4, $_POST['phone'])
                  ->execute();
            }

            // Generate booking number
            $bookingNumber = 'BK' . date('Ymd') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);

            // Create booking
            $this->db->query("
                INSERT INTO bookings (
                    booking_number, company_id, branch_id, service_id, customer_id,
                    staff_id, guest_first_name, guest_last_name, guest_email, guest_phone,
                    start_at, end_at, duration_minutes, total_amount, status, notes, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', ?, NOW())
            ")->bind(1, $bookingNumber)
              ->bind(2, $service['company_id'])
              ->bind(3, $branch['id'])
              ->bind(4, $service['id'])
              ->bind(5, $customerId)
              ->bind(6, $_POST['staff_id'] ?? null)
              ->bind(7, $_POST['first_name'])
              ->bind(8, $_POST['last_name'])
              ->bind(9, $_POST['email'])
              ->bind(10, $_POST['phone'])
              ->bind(11, $startDateTime->format('Y-m-d H:i:s'))
              ->bind(12, $endDateTime->format('Y-m-d H:i:s'))
              ->bind(13, $service['duration_minutes'])
              ->bind(14, $service['price'])
              ->bind(15, $_POST['notes'] ?? null)
              ->execute();

            $bookingId = $this->db->lastInsertId();

            // Commit transaction
            $this->db->commit();

            // Send confirmation email/WhatsApp (would be implemented)
            // $this->sendBookingConfirmation($bookingId);

            $this->json([
                'success' => true,
                'booking_id' => $bookingId,
                'booking_number' => $bookingNumber,
                'message' => 'Booking created successfully'
            ]);

        } catch (Exception $e) {
            $this->db->rollback();
            error_log("Booking creation failed: " . $e->getMessage());
            $this->json(['error' => 'Failed to create booking'], 500);
        }
    }

    public function show($id) {
        // Get booking details
        $booking = $this->db->query("
            SELECT b.*, s.name as service_name, s.price, br.name as branch_name,
                   br.address, u.first_name as staff_first_name, u.last_name as staff_last_name
            FROM bookings b
            JOIN services s ON b.service_id = s.id
            JOIN branches br ON b.branch_id = br.id
            LEFT JOIN users u ON b.staff_id = u.id
            WHERE b.id = ?
        ")->bind(1, $id)->single();

        if (!$booking) {
            http_response_code(404);
            $this->view('errors/404');
            return;
        }

        // Check permissions (customer can only view their own bookings, staff/admin can view more)
        if (!Auth::check()) {
            // Guest booking - check if email matches
            if ($booking['customer_id'] !== null || $booking['guest_email'] !== $_GET['email'] ?? '') {
                http_response_code(403);
                $this->view('errors/403');
                return;
            }
        } elseif (Auth::user()['role'] === 'customer' && $booking['customer_id'] !== Auth::user()['id']) {
            http_response_code(403);
            $this->view('errors/403');
            return;
        }

        $this->view('booking/show', ['booking' => $booking]);
    }

    private function json($data, $statusCode = 200) {
        http_response_code($statusCode);
        header('Content-Type: application/json');
        echo json_encode($data);
        exit;
    }
}