<?php
/**
 * Authentication Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class AuthController extends Controller {
    public function login() {
        // If already logged in, redirect to dashboard
        if ($this->auth->check()) {
            $this->redirectToDashboard();
        }

        $this->view('auth/login');
    }

    public function doLogin() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/login');
        }

        $email = $this->getPost('email');
        $password = $this->getPost('password');
        $remember = $this->getPost('remember');

        // Validate input
        $errors = $this->validateRequired($_POST, ['email', 'password']);

        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/login');
        }

        // Attempt login
        if ($this->auth->attempt($email, $password)) {
            // Regenerate session for security
            session_regenerate_id(true);

            // Set remember me cookie if requested
            if ($remember) {
                $token = bin2hex(random_bytes(32));
                setcookie('remember_token', $token, time() + (30 * 24 * 60 * 60), '/', '', true, true);
                // In a real app, you'd store this token in the database
            }

            $this->redirectToDashboard();
        } else {
            $_SESSION['form_errors'] = ['general' => 'Invalid email or password'];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/login');
        }
    }

    public function register() {
        // If already logged in, redirect to dashboard
        if ($this->auth->check()) {
            $this->redirectToDashboard();
        }

        $this->view('auth/register');
    }

    public function doRegister() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/register');
        }

        $data = [
            'first_name' => $this->getPost('first_name'),
            'last_name' => $this->getPost('last_name'),
            'email' => $this->getPost('email'),
            'phone' => $this->getPost('phone'),
            'password' => $this->getPost('password'),
            'password_confirm' => $this->getPost('password_confirm')
        ];

        // Validate input
        $errors = $this->validateRequired($data, ['first_name', 'last_name', 'email', 'password']);

        if (empty($errors)) {
            // Validate email format
            if (!$this->validateEmail($data['email'])) {
                $errors['email'] = 'Please enter a valid email address';
            }

            // Check password length
            if (strlen($data['password']) < 8) {
                $errors['password'] = 'Password must be at least 8 characters long';
            }

            // Check password confirmation
            if ($data['password'] !== $data['password_confirm']) {
                $errors['password_confirm'] = 'Passwords do not match';
            }

            // Check if email already exists
            $userModel = new User();
            if ($userModel->emailExists($data['email'])) {
                $errors['email'] = 'Email address already registered';
            }
        }

        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $data;
            $this->redirect(BASE_URL . '/register');
        }

        // Create user
        $userData = [
            'first_name' => $data['first_name'],
            'last_name' => $data['last_name'],
            'email' => $data['email'],
            'phone' => $data['phone'],
            'status' => 'active'
        ];

        $userId = $this->auth->register($userData);

        if ($userId) {
            // Auto-login after registration
            $this->auth->attempt($data['email'], $data['password']);
            $this->redirect(BASE_URL . '/dashboard', 'Account created successfully! Welcome to our salon.');
        } else {
            $_SESSION['form_errors'] = ['general' => 'Registration failed. Please try again.'];
            $_SESSION['form_data'] = $data;
            $this->redirect(BASE_URL . '/register');
        }
    }

    public function logout() {
        $this->auth->logout();

        // Clear remember me cookie
        if (isset($_COOKIE['remember_token'])) {
            setcookie('remember_token', '', time() - 3600, '/', '', true, true);
        }

        $this->redirect(BASE_URL . '/', 'You have been logged out successfully.');
    }

    private function redirectToDashboard() {
        $user = $this->auth->user();

        switch ($user['role']) {
            case 'super_admin':
            case 'company_admin':
            case 'manager':
                $this->redirect(BASE_URL . '/admin/dashboard');
                break;
            case 'staff':
                $this->redirect(BASE_URL . '/staff/dashboard');
                break;
            default:
                $this->redirect(BASE_URL . '/dashboard');
        }
    }
}
