<?php
/**
 * Booking API Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class BookingController extends Controller {
    public function store() {
        try {
            // Get JSON input
            $input = json_decode(file_get_contents('php://input'), true);

            if (!$input) {
                return $this->json(['error' => 'Invalid JSON input'], 400);
            }

            // Validate required fields
            $requiredFields = ['service_id', 'branch_id', 'start_at'];
            $validationErrors = $this->validateRequired($input, $requiredFields);

            if (!empty($validationErrors)) {
                return $this->json(['error' => 'Validation failed', 'details' => $validationErrors], 422);
            }

            // Additional validation
            $errors = $this->validateBookingData($input);
            if (!empty($errors)) {
                return $this->json(['error' => 'Validation failed', 'details' => $errors], 422);
            }

            // Check if user is authenticated (for registered users)
            $customerId = $this->auth->check() ? $this->auth->id() : null;

            // Prepare booking data
            $bookingData = [
                'service_id' => $input['service_id'],
                'branch_id' => $input['branch_id'],
                'customer_id' => $customerId,
                'staff_id' => $input['staff_id'] ?? null,
                'start_at' => $input['start_at'],
                'status' => 'pending',
                'notes' => $input['notes'] ?? '',
                'customer_notes' => $input['customer_notes'] ?? ''
            ];

            // Handle guest booking
            if (!$customerId && isset($input['guest_details'])) {
                $bookingData['guest_details'] = $input['guest_details'];
            }

            // Create booking
            $bookingModel = new Booking();
            $bookingId = $bookingModel->createBooking($bookingData);

            // Get created booking details
            $booking = $bookingModel->findWithDetails($bookingId);

            // Send confirmation notifications
            $this->sendBookingConfirmation($booking);

            $this->json([
                'success' => true,
                'message' => 'Booking created successfully',
                'data' => [
                    'booking_id' => $bookingId,
                    'booking_number' => $booking['booking_number'],
                    'status' => $booking['status'],
                    'start_at' => $booking['start_at'],
                    'service' => $booking['service_name'],
                    'branch' => $booking['branch_name'],
                    'staff' => $booking['staff_name'] ?? null,
                    'total_amount' => (float) $booking['total_amount']
                ]
            ], 201);

        } catch (Exception $e) {
            error_log("Booking API Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to create booking: ' . $e->getMessage()], 500);
        }
    }

    public function show($id) {
        try {
            $bookingModel = new Booking();

            // Check permissions
            $booking = $bookingModel->findWithDetails($id);

            if (!$booking) {
                return $this->json(['error' => 'Booking not found'], 404);
            }

            // Check if user can view this booking
            if (!$this->canViewBooking($booking)) {
                return $this->json(['error' => 'Access denied'], 403);
            }

            $this->json([
                'success' => true,
                'data' => $this->formatBookingResponse($booking)
            ]);

        } catch (Exception $e) {
            error_log("Booking API Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to fetch booking'], 500);
        }
    }

    public function update($id) {
        try {
            $input = json_decode(file_get_contents('php://input'), true);

            if (!$input) {
                return $this->json(['error' => 'Invalid JSON input'], 400);
            }

            $bookingModel = new Booking();
            $booking = $bookingModel->findWithDetails($id);

            if (!$booking) {
                return $this->json(['error' => 'Booking not found'], 404);
            }

            // Check permissions
            if (!$this->canModifyBooking($booking)) {
                return $this->json(['error' => 'Access denied'], 403);
            }

            // Handle different update types
            $updateType = $input['action'] ?? 'update';

            switch ($updateType) {
                case 'reschedule':
                    $result = $this->rescheduleBooking($id, $input);
                    break;
                case 'cancel':
                    $result = $this->cancelBooking($id, $input);
                    break;
                case 'status':
                    $result = $this->updateBookingStatus($id, $input);
                    break;
                default:
                    $result = $this->updateBookingDetails($id, $input);
            }

            if ($result) {
                // Get updated booking
                $updatedBooking = $bookingModel->findWithDetails($id);

                // Send notifications if needed
                if ($updateType === 'reschedule' || $updateType === 'cancel') {
                    $this->sendBookingUpdateNotification($updatedBooking, $updateType);
                }

                $this->json([
                    'success' => true,
                    'message' => 'Booking updated successfully',
                    'data' => $this->formatBookingResponse($updatedBooking)
                ]);
            } else {
                $this->json(['error' => 'Failed to update booking'], 500);
            }

        } catch (Exception $e) {
            error_log("Booking Update API Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to update booking'], 500);
        }
    }

    public function destroy($id) {
        try {
            $bookingModel = new Booking();
            $booking = $bookingModel->findWithDetails($id);

            if (!$booking) {
                return $this->json(['error' => 'Booking not found'], 404);
            }

            // Check permissions
            if (!$this->canModifyBooking($booking)) {
                return $this->json(['error' => 'Access denied'], 403);
            }

            // Only allow deletion of pending bookings
            if ($booking['status'] !== 'pending') {
                return $this->json(['error' => 'Cannot delete confirmed bookings'], 400);
            }

            $result = $bookingModel->delete($id);

            if ($result) {
                $this->json([
                    'success' => true,
                    'message' => 'Booking deleted successfully'
                ]);
            } else {
                $this->json(['error' => 'Failed to delete booking'], 500);
            }

        } catch (Exception $e) {
            error_log("Booking Delete API Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to delete booking'], 500);
        }
    }

    /**
     * Validate booking data
     */
    private function validateBookingData($data) {
        $errors = [];

        // Validate service exists and is active
        $service = $this->db->query("SELECT * FROM services WHERE id = ? AND is_active = 1")
                           ->bind(1, $data['service_id'])
                           ->single();
        if (!$service) {
            $errors['service_id'] = 'Invalid or inactive service';
        }

        // Validate branch exists and is active
        $branch = $this->db->query("SELECT * FROM branches WHERE id = ? AND status = 'active'")
                          ->bind(1, $data['branch_id'])
                          ->single();
        if (!$branch) {
            $errors['branch_id'] = 'Invalid or inactive branch';
        }

        // Validate staff (if provided)
        if (!empty($data['staff_id'])) {
            $staff = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'staff' AND status = 'active'")
                             ->bind(1, $data['staff_id'])
                             ->single();
            if (!$staff) {
                $errors['staff_id'] = 'Invalid or inactive staff member';
            }
        }

        // Validate datetime
        if (!strtotime($data['start_at'])) {
            $errors['start_at'] = 'Invalid date and time format';
        } else {
            $bookingTime = strtotime($data['start_at']);
            $now = time();
            $minAdvanceTime = MIN_ADVANCE_BOOKING_HOURS * 3600; // hours to seconds

            if ($bookingTime <= $now + $minAdvanceTime) {
                $errors['start_at'] = 'Booking must be made at least ' . MIN_ADVANCE_BOOKING_HOURS . ' hours in advance';
            }

            if ($bookingTime > strtotime('+' . MAX_ADVANCE_BOOKING_DAYS . ' days')) {
                $errors['start_at'] = 'Cannot book more than ' . MAX_ADVANCE_BOOKING_DAYS . ' days in advance';
            }
        }

        // Validate guest details (if no customer_id)
        if (empty($data['customer_id']) && isset($data['guest_details'])) {
            $guestErrors = $this->validateGuestDetails($data['guest_details']);
            if (!empty($guestErrors)) {
                $errors['guest_details'] = $guestErrors;
            }
        }

        return $errors;
    }

    /**
     * Validate guest details
     */
    private function validateGuestDetails($guest) {
        $errors = [];

        if (empty($guest['first_name']) || strlen($guest['first_name']) < 2) {
            $errors['first_name'] = 'First name must be at least 2 characters';
        }

        if (empty($guest['last_name']) || strlen($guest['last_name']) < 2) {
            $errors['last_name'] = 'Last name must be at least 2 characters';
        }

        if (empty($guest['email']) || !filter_var($guest['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Valid email address is required';
        }

        if (empty($guest['phone']) || !preg_match('/^[\+]?[1-9][\d]{0,15}$/', $guest['phone'])) {
            $errors['phone'] = 'Valid phone number is required';
        }

        return $errors;
    }

    /**
     * Check if user can view booking
     */
    private function canViewBooking($booking) {
        if ($this->auth->isAdmin()) {
            return true;
        }

        if ($this->auth->isStaff()) {
            // Staff can view their assigned bookings or bookings at their branch
            return $booking['staff_id'] == $this->auth->id() ||
                   $booking['branch_id'] == $this->auth->branchId();
        }

        // Customers can view their own bookings
        return $booking['customer_id'] == $this->auth->id() ||
               ($booking['guest_email'] && $booking['guest_email'] === $this->auth->user()['email']);
    }

    /**
     * Check if user can modify booking
     */
    private function canModifyBooking($booking) {
        if ($this->auth->isAdmin()) {
            return true;
        }

        if ($this->auth->isStaff()) {
            // Staff can modify bookings at their branch
            return $booking['branch_id'] == $this->auth->branchId();
        }

        // Customers can modify their pending bookings
        return $booking['customer_id'] == $this->auth->id() &&
               $booking['status'] === 'pending';
    }

    /**
     * Reschedule booking
     */
    private function rescheduleBooking($id, $data) {
        if (empty($data['new_start_at'])) {
            throw new Exception('New start time is required for rescheduling');
        }

        $bookingModel = new Booking();
        $booking = $bookingModel->find($id);

        // Update booking time
        return $bookingModel->update($id, [
            'start_at' => $data['new_start_at'],
            'status' => 'pending' // Reset status for rescheduled bookings
        ]);
    }

    /**
     * Cancel booking
     */
    private function cancelBooking($id, $data) {
        $bookingModel = new Booking();

        return $bookingModel->updateStatus($id, 'cancelled', $this->auth->id());
    }

    /**
     * Update booking status
     */
    private function updateBookingStatus($id, $data) {
        if (empty($data['status'])) {
            throw new Exception('Status is required');
        }

        $bookingModel = new Booking();
        return $bookingModel->updateStatus($id, $data['status'], $this->auth->id());
    }

    /**
     * Update booking details
     */
    private function updateBookingDetails($id, $data) {
        $updateData = [];

        if (isset($data['notes'])) {
            $updateData['notes'] = $data['notes'];
        }

        if (isset($data['customer_notes'])) {
            $updateData['customer_notes'] = $data['customer_notes'];
        }

        if (!empty($updateData)) {
            $bookingModel = new Booking();
            return $bookingModel->update($id, $updateData);
        }

        return true;
    }

    /**
     * Format booking response
     */
    private function formatBookingResponse($booking) {
        return [
            'id' => $booking['id'],
            'booking_number' => $booking['booking_number'],
            'service' => [
                'id' => $booking['service_id'],
                'name' => $booking['service_name'],
                'duration_minutes' => $booking['duration_minutes']
            ],
            'branch' => [
                'id' => $booking['branch_id'],
                'name' => $booking['branch_name'],
                'address' => $booking['branch_address']
            ],
            'staff' => $booking['staff_name'] ? [
                'id' => $booking['staff_id'],
                'name' => $booking['staff_first_name'] . ' ' . $booking['staff_last_name']
            ] : null,
            'customer' => $booking['customer_id'] ? [
                'id' => $booking['customer_id'],
                'name' => $booking['first_name'] . ' ' . $booking['last_name'],
                'email' => $booking['email'],
                'phone' => $booking['phone']
            ] : [
                'name' => $booking['guest_first_name'] . ' ' . $booking['guest_last_name'],
                'email' => $booking['guest_email'],
                'phone' => $booking['guest_phone']
            ],
            'start_at' => $booking['start_at'],
            'end_at' => $booking['end_at'],
            'status' => $booking['status'],
            'total_amount' => (float) $booking['total_amount'],
            'notes' => $booking['notes'],
            'customer_notes' => $booking['customer_notes'],
            'created_at' => $booking['created_at'],
            'updated_at' => $booking['updated_at']
        ];
    }

    /**
     * Send booking confirmation notifications
     */
    private function sendBookingConfirmation($booking) {
        // This would integrate with notification system
        // For now, just log the action
        error_log("Booking confirmation sent for booking #" . $booking['booking_number']);
    }

    /**
     * Send booking update notifications
     */
    private function sendBookingUpdateNotification($booking, $action) {
        // This would integrate with notification system
        error_log("Booking {$action} notification sent for booking #" . $booking['booking_number']);
    }
}
