<?php
/**
 * Availability API Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class AvailabilityController extends Controller {
    public function check() {
        try {
            $serviceId = $this->getQuery('service_id');
            $branchId = $this->getQuery('branch_id');
            $staffId = $this->getQuery('staff_id');
            $date = $this->getQuery('date');
            $duration = (int) $this->getQuery('duration', 60); // minutes

            // Validate required parameters
            if (!$serviceId || !$branchId || !$date) {
                return $this->json([
                    'error' => 'Missing required parameters: service_id, branch_id, date'
                ], 400);
            }

            // Validate date format
            if (!strtotime($date)) {
                return $this->json(['error' => 'Invalid date format'], 400);
            }

            // Check if date is not in the past
            if (strtotime($date) < strtotime(date('Y-m-d'))) {
                return $this->json(['error' => 'Cannot book appointments in the past'], 400);
            }

            $availability = $this->getServiceAvailability($serviceId, $branchId, $staffId, $date, $duration);

            $this->json([
                'success' => true,
                'data' => $availability
            ]);

        } catch (Exception $e) {
            error_log("Availability API Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to check availability'], 500);
        }
    }

    /**
     * Get service availability for a specific date
     */
    private function getServiceAvailability($serviceId, $branchId, $staffId, $date, $duration) {
        $bookingModel = new Booking();
        $branchModel = new Branch();

        // Get service details
        $service = $this->db->query("SELECT * FROM services WHERE id = ? AND is_active = 1")
                           ->bind(1, $serviceId)
                           ->single();

        if (!$service) {
            throw new Exception('Service not found or inactive');
        }

        // Get branch availability
        $branchAvailability = $branchModel->getAvailability($branchId, $date);

        if (!$branchAvailability || !$branchAvailability['is_open']) {
            return [
                'date' => $date,
                'is_available' => false,
                'reason' => 'Branch is closed on this date',
                'branch_hours' => $branchAvailability,
                'available_slots' => []
            ];
        }

        $openTime = $branchAvailability['open_time'];
        $closeTime = $branchAvailability['close_time'];

        // Generate time slots
        $slots = $this->generateTimeSlots($openTime, $closeTime, $service['buffer_before_minutes'], $service['buffer_after_minutes'], $duration);

        // Check each slot for conflicts
        $availableSlots = [];
        foreach ($slots as $slot) {
            $slotStart = $date . ' ' . $slot['start'];
            $slotEnd = $date . ' ' . $slot['end'];

            $conflict = $bookingModel->checkAvailability($serviceId, $staffId, $slotStart, $slotEnd);

            if ($conflict['available']) {
                $availableSlots[] = [
                    'start_time' => $slotStart,
                    'end_time' => $slotEnd,
                    'formatted_time' => $slot['formatted'],
                    'available' => true
                ];
            }
        }

        return [
            'date' => $date,
            'is_available' => !empty($availableSlots),
            'branch_hours' => $branchAvailability,
            'service' => [
                'id' => $service['id'],
                'name' => $service['name'],
                'duration_minutes' => $service['duration_minutes'],
                'buffer_before_minutes' => $service['buffer_before_minutes'],
                'buffer_after_minutes' => $service['buffer_after_minutes']
            ],
            'available_slots' => $availableSlots,
            'total_slots' => count($availableSlots)
        ];
    }

    /**
     * Generate time slots based on branch hours and service requirements
     */
    private function generateTimeSlots($openTime, $closeTime, $bufferBefore, $bufferAfter, $serviceDuration) {
        $slots = [];

        // Convert times to minutes since midnight
        $openMinutes = $this->timeToMinutes($openTime);
        $closeMinutes = $this->timeToMinutes($closeTime);

        // Add buffer to opening time
        $effectiveOpenMinutes = $openMinutes + $bufferBefore;

        // Subtract buffer and service duration from closing time
        $effectiveCloseMinutes = $closeMinutes - $bufferAfter - $serviceDuration;

        if ($effectiveOpenMinutes >= $effectiveCloseMinutes) {
            return $slots; // No available slots
        }

        // Generate slots every 15 minutes (configurable)
        $slotInterval = BOOKING_TIME_SLOT; // minutes
        $currentMinutes = $effectiveOpenMinutes;

        while ($currentMinutes + $serviceDuration <= $effectiveCloseMinutes) {
            $startTime = $this->minutesToTime($currentMinutes);
            $endTime = $this->minutesToTime($currentMinutes + $serviceDuration);

            $slots[] = [
                'start' => $startTime,
                'end' => $endTime,
                'formatted' => formatTime($startTime) . ' - ' . formatTime($endTime)
            ];

            $currentMinutes += $slotInterval;
        }

        return $slots;
    }

    /**
     * Convert time string to minutes since midnight
     */
    private function timeToMinutes($time) {
        list($hours, $minutes) = explode(':', $time);
        return ($hours * 60) + $minutes;
    }

    /**
     * Convert minutes since midnight to time string
     */
    private function minutesToTime($minutes) {
        $hours = floor($minutes / 60);
        $mins = $minutes % 60;
        return sprintf('%02d:%02d:00', $hours, $mins);
    }

    /**
     * Get multiple dates availability
     */
    public function checkMultiple() {
        try {
            $serviceId = $this->getQuery('service_id');
            $branchId = $this->getQuery('branch_id');
            $staffId = $this->getQuery('staff_id');
            $startDate = $this->getQuery('start_date');
            $endDate = $this->getQuery('end_date', date('Y-m-d', strtotime('+30 days')));
            $duration = (int) $this->getQuery('duration', 60);

            if (!$serviceId || !$branchId || !$startDate) {
                return $this->json([
                    'error' => 'Missing required parameters: service_id, branch_id, start_date'
                ], 400);
            }

            $availabilities = [];
            $currentDate = strtotime($startDate);
            $endDateTime = strtotime($endDate);

            while ($currentDate <= $endDateTime) {
                $date = date('Y-m-d', $currentDate);
                $availability = $this->getServiceAvailability($serviceId, $branchId, $staffId, $date, $duration);
                $availabilities[] = $availability;
                $currentDate = strtotime('+1 day', $currentDate);
            }

            $this->json([
                'success' => true,
                'data' => $availabilities,
                'meta' => [
                    'total_dates' => count($availabilities),
                    'date_range' => [
                        'start' => $startDate,
                        'end' => $endDate
                    ]
                ]
            ]);

        } catch (Exception $e) {
            error_log("Multiple Availability API Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to check availability'], 500);
        }
    }
}
