<?php
/**
 * Admin Staff Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class StaffController extends Controller {
    public function index() {
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();

        // Get filter parameters
        $branch = $this->getQuery('branch');
        $status = $this->getQuery('status', 'active');
        $search = $this->getQuery('search');
        $page = (int) $this->getQuery('page', 1);

        // Build where clause
        $where = ["u.company_id = ?", "u.role = 'staff'"];
        $params = [$companyId];

        if ($branch) {
            $where[] = "u.branch_id = ?";
            $params[] = $branch;
        } elseif ($branchId) {
            $where[] = "u.branch_id = ?";
            $params[] = $branchId;
        }

        if ($status === 'active') {
            $where[] = "u.status = 'active'";
        } elseif ($status === 'inactive') {
            $where[] = "u.status = 'inactive'";
        }

        if ($search) {
            $where[] = "(u.first_name LIKE ? OR u.last_name LIKE ? OR u.email LIKE ?)";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
        }

        $whereClause = implode(' AND ', $where);

        // Get staff with pagination
        $offset = ($page - 1) * ADMIN_ITEMS_PER_PAGE;

        $staff = $this->db->query("
            SELECT u.*, b.name as branch_name,
                   COUNT(sb.id) as services_count,
                   COUNT(bk.id) as total_bookings,
                   COUNT(CASE WHEN bk.status = 'completed' THEN 1 END) as completed_bookings,
                   COALESCE(SUM(CASE WHEN bk.status = 'completed' THEN bk.total_amount END), 0) as total_revenue
            FROM users u
            LEFT JOIN branches b ON u.branch_id = b.id
            LEFT JOIN staff_services sb ON u.id = sb.user_id
            LEFT JOIN bookings bk ON u.id = bk.staff_id
            WHERE {$whereClause}
            GROUP BY u.id
            ORDER BY u.first_name, u.last_name
            LIMIT ? OFFSET ?
        ", array_merge($params, [ADMIN_ITEMS_PER_PAGE, $offset]))->resultSet();

        // Get total count
        $totalStaff = $this->db->query("
            SELECT COUNT(*) as total FROM users u WHERE {$whereClause}
        ", $params)->single()['total'];

        // Get branches for filter
        $branches = $this->db->query("
            SELECT id, name FROM branches
            WHERE company_id = ? AND status = 'active'
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        $this->adminView('staff/index', [
            'staff' => $staff,
            'branches' => $branches,
            'filters' => [
                'branch' => $branch,
                'status' => $status,
                'search' => $search,
                'page' => $page
            ],
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalStaff / ADMIN_ITEMS_PER_PAGE),
                'total_items' => $totalStaff
            ],
            'pageTitle' => 'Staff Management'
        ]);
    }

    public function create() {
        $companyId = $this->auth->companyId();

        // Get branches
        $branches = $this->db->query("
            SELECT id, name FROM branches
            WHERE company_id = ? AND status = 'active'
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        // Get services for staff assignment
        $services = $this->db->query("
            SELECT id, name FROM services
            WHERE company_id = ? AND is_active = 1
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        $this->adminView('staff/create', [
            'branches' => $branches,
            'services' => $services,
            'pageTitle' => 'Add New Staff Member'
        ]);
    }

    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/admin/staff/create');
        }

        $data = [
            'company_id' => $this->auth->companyId(),
            'first_name' => $this->getPost('first_name'),
            'last_name' => $this->getPost('last_name'),
            'email' => $this->getPost('email'),
            'phone' => $this->getPost('phone'),
            'branch_id' => $this->getPost('branch_id'),
            'role' => 'staff',
            'status' => 'active'
        ];

        // Get services and commission rates
        $services = $this->getPost('services', []);
        $commissionRates = $this->getPost('commission_rates', []);

        // Validate input
        $errors = $this->validateStaffInput($data);

        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/staff/create');
        }

        try {
            // Check if email already exists
            $existingUser = $this->db->query("SELECT id FROM users WHERE email = ?")
                                   ->bind(1, $data['email'])
                                   ->single();

            if ($existingUser) {
                $_SESSION['form_errors'] = ['email' => 'Email address already exists'];
                $_SESSION['form_data'] = $_POST;
                $this->redirect(BASE_URL . '/admin/staff/create');
            }

            // Generate temporary password
            $tempPassword = $this->generateTempPassword();
            $data['password_hash'] = password_hash($tempPassword, PASSWORD_DEFAULT);

            // Create staff member
            $staffId = $this->db->insert('users', $data);

            // Assign services to staff
            if (!empty($services)) {
                foreach ($services as $index => $serviceId) {
                    $commissionRate = isset($commissionRates[$index]) ? (float) $commissionRates[$index] : 0;

                    $this->db->insert('staff_services', [
                        'user_id' => $staffId,
                        'service_id' => $serviceId,
                        'commission_rate' => $commissionRate,
                        'is_primary' => ($index === 0) // First service is primary
                    ]);
                }
            }

            // Create default availability (Monday-Friday, 9 AM - 5 PM)
            $this->createDefaultAvailability($staffId, $data['branch_id']);

            // Send welcome email with temporary password
            $this->sendWelcomeEmail($data['email'], $data['first_name'], $tempPassword);

            $this->redirect(BASE_URL . '/admin/staff', 'Staff member added successfully!');

        } catch (Exception $e) {
            error_log("Staff creation error: " . $e->getMessage());
            $_SESSION['form_errors'] = ['general' => 'Failed to create staff member'];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/staff/create');
        }
    }

    public function edit($id) {
        $staff = $this->db->query("
            SELECT u.*, b.name as branch_name
            FROM users u
            LEFT JOIN branches b ON u.branch_id = b.id
            WHERE u.id = ? AND u.role = 'staff'
        ")->bind(1, $id)->single();

        if (!$staff || !$this->canAccessStaff($staff)) {
            $this->adminView('errors/404');
            return;
        }

        $companyId = $this->auth->companyId();

        // Get branches
        $branches = $this->db->query("
            SELECT id, name FROM branches
            WHERE company_id = ? AND status = 'active'
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        // Get services
        $services = $this->db->query("
            SELECT id, name FROM services
            WHERE company_id = ? AND is_active = 1
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        // Get assigned services
        $assignedServices = $this->db->query("
            SELECT ss.service_id, ss.commission_rate, ss.is_primary, s.name
            FROM staff_services ss
            JOIN services s ON ss.service_id = s.id
            WHERE ss.user_id = ?
            ORDER BY ss.is_primary DESC, s.name ASC
        ")->bind(1, $id)->resultSet();

        $this->adminView('staff/edit', [
            'staff' => $staff,
            'branches' => $branches,
            'services' => $services,
            'assignedServices' => $assignedServices,
            'pageTitle' => 'Edit Staff Member - ' . $staff['first_name'] . ' ' . $staff['last_name']
        ]);
    }

    public function update($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/admin/staff/' . $id . '/edit');
        }

        $staff = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'staff'")
                         ->bind(1, $id)
                         ->single();

        if (!$staff || !$this->canAccessStaff($staff)) {
            $this->adminView('errors/404');
            return;
        }

        $data = [
            'first_name' => $this->getPost('first_name'),
            'last_name' => $this->getPost('last_name'),
            'email' => $this->getPost('email'),
            'phone' => $this->getPost('phone'),
            'branch_id' => $this->getPost('branch_id'),
            'status' => $this->getPost('status')
        ];

        // Get services and commission rates
        $services = $this->getPost('services', []);
        $commissionRates = $this->getPost('commission_rates', []);

        // Validate input
        $errors = $this->validateStaffInput($data, $id);

        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/staff/' . $id . '/edit');
        }

        try {
            // Check if email already exists (excluding current user)
            $existingUser = $this->db->query("SELECT id FROM users WHERE email = ? AND id != ?")
                                   ->bind(1, $data['email'])
                                   ->bind(2, $id)
                                   ->single();

            if ($existingUser) {
                $_SESSION['form_errors'] = ['email' => 'Email address already exists'];
                $_SESSION['form_data'] = $_POST;
                $this->redirect(BASE_URL . '/admin/staff/' . $id . '/edit');
            }

            // Update staff member
            $this->db->update('users', $data, ['id' => $id]);

            // Update services assignment
            $this->db->query("DELETE FROM staff_services WHERE user_id = ?")->bind(1, $id)->execute();

            if (!empty($services)) {
                foreach ($services as $index => $serviceId) {
                    $commissionRate = isset($commissionRates[$index]) ? (float) $commissionRates[$index] : 0;

                    $this->db->insert('staff_services', [
                        'user_id' => $id,
                        'service_id' => $serviceId,
                        'commission_rate' => $commissionRate,
                        'is_primary' => ($index === 0) // First service is primary
                    ]);
                }
            }

            $this->redirect(BASE_URL . '/admin/staff', 'Staff member updated successfully!');

        } catch (Exception $e) {
            error_log("Staff update error: " . $e->getMessage());
            $_SESSION['form_errors'] = ['general' => 'Failed to update staff member'];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/staff/' . $id . '/edit');
        }
    }

    public function destroy($id) {
        $staff = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'staff'")
                         ->bind(1, $id)
                         ->single();

        if (!$staff || !$this->canAccessStaff($staff)) {
            $this->json(['error' => 'Staff member not found'], 404);
        }

        // Check if staff has upcoming bookings
        $upcomingBookings = $this->db->query("
            SELECT COUNT(*) as count FROM bookings
            WHERE staff_id = ? AND start_at > NOW() AND status IN ('confirmed', 'pending')
        ")->bind(1, $id)->single()['count'];

        if ($upcomingBookings > 0) {
            $this->json(['error' => 'Cannot delete staff member with upcoming bookings'], 400);
        }

        try {
            // Delete staff services
            $this->db->query("DELETE FROM staff_services WHERE user_id = ?")->bind(1, $id)->execute();

            // Delete availability
            $this->db->query("DELETE FROM staff_availability WHERE user_id = ?")->bind(1, $id)->execute();

            // Delete staff member
            $this->db->delete('users', ['id' => $id]);

            $this->json([
                'success' => true,
                'message' => 'Staff member deleted successfully'
            ]);

        } catch (Exception $e) {
            error_log("Staff deletion error: " . $e->getMessage());
            $this->json(['error' => 'Failed to delete staff member'], 500);
        }
    }

    public function toggleStatus($id) {
        $staff = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'staff'")
                         ->bind(1, $id)
                         ->single();

        if (!$staff || !$this->canAccessStaff($staff)) {
            $this->json(['error' => 'Staff member not found'], 404);
        }

        $newStatus = $staff['status'] === 'active' ? 'inactive' : 'active';

        try {
            $this->db->update('users', ['status' => $newStatus], ['id' => $id]);

            $this->json([
                'success' => true,
                'message' => 'Staff status updated successfully',
                'new_status' => $newStatus
            ]);

        } catch (Exception $e) {
            error_log("Staff status toggle error: " . $e->getMessage());
            $this->json(['error' => 'Failed to update staff status'], 500);
        }
    }

    public function schedule($id) {
        $staff = $this->db->query("
            SELECT u.*, b.name as branch_name
            FROM users u
            LEFT JOIN branches b ON u.branch_id = b.id
            WHERE u.id = ? AND u.role = 'staff'
        ")->bind(1, $id)->single();

        if (!$staff || !$this->canAccessStaff($staff)) {
            $this->adminView('errors/404');
            return;
        }

        // Get current availability
        $availability = $this->db->query("
            SELECT * FROM staff_availability
            WHERE user_id = ?
            ORDER BY day_of_week ASC, start_time ASC
        ")->bind(1, $id)->resultSet();

        $this->adminView('staff/schedule', [
            'staff' => $staff,
            'availability' => $availability,
            'pageTitle' => 'Staff Schedule - ' . $staff['first_name'] . ' ' . $staff['last_name']
        ]);
    }

    public function updateSchedule($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['error' => 'Method not allowed'], 405);
        }

        $staff = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'staff'")
                         ->bind(1, $id)
                         ->single();

        if (!$staff || !$this->canAccessStaff($staff)) {
            $this->json(['error' => 'Staff member not found'], 404);
        }

        try {
            // Delete existing availability
            $this->db->query("DELETE FROM staff_availability WHERE user_id = ?")->bind(1, $id)->execute();

            // Add new availability
            $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];

            foreach ($days as $dayIndex => $dayName) {
                $enabled = $this->getPost("{$dayName}_enabled");
                $startTime = $this->getPost("{$dayName}_start");
                $endTime = $this->getPost("{$dayName}_end");
                $breakStart = $this->getPost("{$dayName}_break_start");
                $breakEnd = $this->getPost("{$dayName}_break_end");

                if ($enabled && $startTime && $endTime) {
                    $this->db->insert('staff_availability', [
                        'user_id' => $id,
                        'branch_id' => $staff['branch_id'],
                        'day_of_week' => $dayIndex,
                        'start_time' => $startTime,
                        'end_time' => $endTime,
                        'break_start' => $breakStart ?: null,
                        'break_end' => $breakEnd ?: null,
                        'is_available' => true
                    ]);
                }
            }

            $this->json([
                'success' => true,
                'message' => 'Staff schedule updated successfully'
            ]);

        } catch (Exception $e) {
            error_log("Staff schedule update error: " . $e->getMessage());
            $this->json(['error' => 'Failed to update staff schedule'], 500);
        }
    }

    /**
     * Validate staff input
     */
    private function validateStaffInput($data, $excludeId = null) {
        $errors = [];

        // Required fields
        if (empty($data['first_name'])) {
            $errors['first_name'] = 'First name is required';
        }

        if (empty($data['last_name'])) {
            $errors['last_name'] = 'Last name is required';
        }

        if (empty($data['email'])) {
            $errors['email'] = 'Email is required';
        } elseif (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Invalid email format';
        }

        // Branch validation
        if (empty($data['branch_id'])) {
            $errors['branch_id'] = 'Branch assignment is required';
        } else {
            $branch = $this->db->query("SELECT id FROM branches WHERE id = ? AND status = 'active'")
                              ->bind(1, $data['branch_id'])
                              ->single();
            if (!$branch) {
                $errors['branch_id'] = 'Invalid branch selected';
            }
        }

        return $errors;
    }

    /**
     * Generate temporary password
     */
    private function generateTempPassword() {
        return bin2hex(random_bytes(4)); // 8 character password
    }

    /**
     * Create default availability
     */
    private function createDefaultAvailability($staffId, $branchId) {
        // Default: Monday-Friday 9 AM - 5 PM
        for ($day = 1; $day <= 5; $day++) {
            $this->db->insert('staff_availability', [
                'user_id' => $staffId,
                'branch_id' => $branchId,
                'day_of_week' => $day,
                'start_time' => '09:00:00',
                'end_time' => '17:00:00',
                'is_available' => true
            ]);
        }
    }

    /**
     * Check if user can access staff member
     */
    private function canAccessStaff($staff) {
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();

        if ($branchId) {
            return $staff['branch_id'] == $branchId;
        } elseif ($companyId) {
            return $staff['company_id'] == $companyId;
        }

        return false;
    }

    /**
     * Send welcome email
     */
    private function sendWelcomeEmail($email, $firstName, $tempPassword) {
        $emailAdapter = new EmailAdapter();
        $result = $emailAdapter->send($email, 'Welcome to Luxury Beauty Spa', "
            Dear {$firstName},

            Welcome to Luxury Beauty Spa! Your staff account has been created.

            Your temporary password is: {$tempPassword}

            Please log in and change your password immediately.

            Login URL: " . BASE_URL . "/login

            Best regards,
            Luxury Beauty Spa Team
        ");

        if (!$result['success']) {
            error_log("Failed to send welcome email to {$email}: " . $result['message']);
        }
    }
}
