<?php
/**
 * Admin Settings Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class SettingController extends Controller {
    public function index() {
        $companyId = $this->auth->companyId();

        // Get current settings
        $settings = $this->getSettings($companyId);

        // Get branches for branch-specific settings
        $branches = $this->db->query("
            SELECT id, name FROM branches
            WHERE company_id = ? AND status = 'active'
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        $this->adminView('settings/index', [
            'settings' => $settings,
            'branches' => $branches,
            'pageTitle' => 'Settings'
        ]);
    }

    public function update() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/admin/settings');
        }

        $companyId = $this->auth->companyId();
        $branchId = $this->getPost('branch_id'); // For branch-specific settings

        $settingsData = [
            // Company Settings
            'company_name' => $this->getPost('company_name'),
            'company_email' => $this->getPost('company_email'),
            'company_phone' => $this->getPost('company_phone'),
            'company_address' => $this->getPost('company_address'),
            'company_description' => $this->getPost('company_description'),

            // Branding
            'primary_color' => $this->getPost('primary_color'),
            'secondary_color' => $this->getPost('secondary_color'),
            'logo_url' => $this->getPost('logo_url'),
            'favicon_url' => $this->getPost('favicon_url'),

            // Business Hours
            'business_hours' => json_encode($this->getPost('business_hours', [])),

            // Booking Settings
            'booking_cancellation_policy' => $this->getPost('booking_cancellation_policy'),
            'booking_minimum_advance_hours' => (int) $this->getPost('booking_minimum_advance_hours', 2),
            'booking_maximum_advance_days' => (int) $this->getPost('booking_maximum_advance_days', 90),
            'booking_buffer_before_minutes' => (int) $this->getPost('booking_buffer_before_minutes', 0),
            'booking_buffer_after_minutes' => (int) $this->getPost('booking_buffer_after_minutes', 0),
            'booking_reminder_24h' => $this->getPost('booking_reminder_24h') === 'on',
            'booking_reminder_2h' => $this->getPost('booking_reminder_2h') === 'on',
            'booking_reminder_enabled' => $this->getPost('booking_reminder_enabled') === 'on',

            // Payment Settings
            'currency' => $this->getPost('currency', 'USD'),
            'payment_methods' => json_encode($this->getPost('payment_methods', [])),
            'tax_rate' => (float) $this->getPost('tax_rate', 0),

            // Notification Settings
            'email_notifications' => $this->getPost('email_notifications') === 'on',
            'sms_notifications' => $this->getPost('sms_notifications') === 'on',
            'whatsapp_notifications' => $this->getPost('whatsapp_notifications') === 'on',

            // SEO Settings
            'seo_title' => $this->getPost('seo_title'),
            'seo_description' => $this->getPost('seo_description'),
            'seo_keywords' => $this->getPost('seo_keywords'),
            'google_analytics_id' => $this->getPost('google_analytics_id'),

            // Social Media
            'facebook_url' => $this->getPost('facebook_url'),
            'instagram_url' => $this->getPost('instagram_url'),
            'twitter_url' => $this->getPost('twitter_url'),

            // Advanced Settings
            'timezone' => $this->getPost('timezone', 'UTC'),
            'language' => $this->getPost('language', 'en'),
            'maintenance_mode' => $this->getPost('maintenance_mode') === 'on',
            'debug_mode' => $this->getPost('debug_mode') === 'on'
        ];

        // Validate input
        $errors = $this->validateSettings($settingsData);
        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/settings');
        }

        try {
            // Save settings
            $this->saveSettings($companyId, $branchId, $settingsData);

            // Update company record if company settings changed
            if (!$branchId) {
                $this->updateCompanyRecord($companyId, $settingsData);
            }

            $this->redirect(BASE_URL . '/admin/settings', 'Settings updated successfully!');

        } catch (Exception $e) {
            error_log("Settings update error: " . $e->getMessage());
            $_SESSION['form_errors'] = ['general' => 'Failed to update settings'];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/settings');
        }
    }

    public function uploadLogo() {
        try {
            if (!isset($_FILES['logo']) || $_FILES['logo']['error'] !== UPLOAD_ERR_OK) {
                $this->json(['error' => 'No file uploaded'], 400);
            }

            $file = $_FILES['logo'];

            // Validate file
            if (!in_array($file['type'], ALLOWED_IMAGE_TYPES)) {
                $this->json(['error' => 'Invalid file type. Only JPEG, PNG, GIF, and WebP are allowed.'], 400);
            }

            if ($file['size'] > MAX_FILE_SIZE) {
                $this->json(['error' => 'File too large. Maximum size is ' . (MAX_FILE_SIZE / 1024 / 1024) . 'MB.'], 400);
            }

            // Create upload directory
            $uploadDir = UPLOADS_PATH . '/logos/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }

            // Generate unique filename
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'logo_' . $this->auth->companyId() . '_' . time() . '.' . $extension;
            $filepath = $uploadDir . $filename;

            if (move_uploaded_file($file['tmp_name'], $filepath)) {
                $url = BASE_URL . '/uploads/logos/' . $filename;

                // Save logo URL to settings
                $this->saveSetting($this->auth->companyId(), null, 'logo_url', $url);

                $this->json([
                    'success' => true,
                    'message' => 'Logo uploaded successfully',
                    'url' => $url
                ]);
            } else {
                $this->json(['error' => 'Failed to save file'], 500);
            }

        } catch (Exception $e) {
            error_log("Logo upload error: " . $e->getMessage());
            $this->json(['error' => 'Upload failed'], 500);
        }
    }

    public function uploadFavicon() {
        try {
            if (!isset($_FILES['favicon']) || $_FILES['favicon']['error'] !== UPLOAD_ERR_OK) {
                $this->json(['error' => 'No file uploaded'], 400);
            }

            $file = $_FILES['favicon'];

            // Validate file (favicon should be small)
            if (!in_array($file['type'], ALLOWED_IMAGE_TYPES)) {
                $this->json(['error' => 'Invalid file type. Only JPEG, PNG, GIF, and WebP are allowed.'], 400);
            }

            if ($file['size'] > 102400) { // 100KB limit for favicon
                $this->json(['error' => 'File too large. Maximum size is 100KB.'], 400);
            }

            // Create upload directory
            $uploadDir = UPLOADS_PATH . '/favicons/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }

            // Generate unique filename
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'favicon_' . $this->auth->companyId() . '_' . time() . '.' . $extension;
            $filepath = $uploadDir . $filename;

            if (move_uploaded_file($file['tmp_name'], $filepath)) {
                $url = BASE_URL . '/uploads/favicons/' . $filename;

                // Save favicon URL to settings
                $this->saveSetting($this->auth->companyId(), null, 'favicon_url', $url);

                $this->json([
                    'success' => true,
                    'message' => 'Favicon uploaded successfully',
                    'url' => $url
                ]);
            } else {
                $this->json(['error' => 'Failed to save file'], 500);
            }

        } catch (Exception $e) {
            error_log("Favicon upload error: " . $e->getMessage());
            $this->json(['error' => 'Upload failed'], 500);
        }
    }

    public function testEmail() {
        try {
            $testEmail = $this->getPost('test_email');

            if (!$testEmail || !filter_var($testEmail, FILTER_VALIDATE_EMAIL)) {
                $this->json(['error' => 'Valid email address required'], 400);
            }

            $emailAdapter = new EmailAdapter();
            $result = $emailAdapter->send($testEmail, 'Email Test', 'This is a test email from your salon system.');

            if ($result['success']) {
                $this->json(['success' => true, 'message' => 'Test email sent successfully']);
            } else {
                $this->json(['error' => $result['message']], 500);
            }

        } catch (Exception $e) {
            error_log("Email test error: " . $e->getMessage());
            $this->json(['error' => 'Email test failed'], 500);
        }
    }

    public function testWhatsApp() {
        try {
            $testPhone = $this->getPost('test_phone');

            if (!$testPhone) {
                $this->json(['error' => 'Phone number required'], 400);
            }

            $whatsappAdapter = new WhatsAppAdapter();
            $result = $whatsappAdapter->send($testPhone, 'This is a test WhatsApp message from your salon system.');

            if ($result['success']) {
                $this->json(['success' => true, 'message' => 'Test WhatsApp message sent successfully']);
            } else {
                $this->json(['error' => $result['message']], 500);
            }

        } catch (Exception $e) {
            error_log("WhatsApp test error: " . $e->getMessage());
            $this->json(['error' => 'WhatsApp test failed'], 500);
        }
    }

    public function testSMS() {
        try {
            $testPhone = $this->getPost('test_phone');

            if (!$testPhone) {
                $this->json(['error' => 'Phone number required'], 400);
            }

            $smsAdapter = new SMSAdapter();
            $result = $smsAdapter->send($testPhone, 'This is a test SMS from your salon system.');

            if ($result['success']) {
                $this->json(['success' => true, 'message' => 'Test SMS sent successfully']);
            } else {
                $this->json(['error' => $result['message']], 500);
            }

        } catch (Exception $e) {
            error_log("SMS test error: " . $e->getMessage());
            $this->json(['error' => 'SMS test failed'], 500);
        }
    }

    /**
     * Get settings for company/branch
     */
    private function getSettings($companyId, $branchId = null) {
        $settings = [];

        // Get company settings
        $companySettings = $this->db->query("
            SELECT setting_key, setting_value, setting_type
            FROM settings
            WHERE company_id = ? AND branch_id IS NULL
        ")->bind(1, $companyId)->resultSet();

        foreach ($companySettings as $setting) {
            $settings[$setting['setting_key']] = $this->castSettingValue($setting['setting_value'], $setting['setting_type']);
        }

        // Get company info
        $company = $this->db->query("SELECT * FROM companies WHERE id = ?")->bind(1, $companyId)->single();
        if ($company) {
            $settings = array_merge($settings, [
                'company_name' => $company['name'],
                'company_email' => $company['email'],
                'company_phone' => $company['phone'],
                'company_address' => $company['address'],
                'company_description' => $company['description'],
                'primary_color' => $company['primary_color'],
                'secondary_color' => $company['secondary_color'],
                'currency' => $company['currency'],
                'timezone' => $company['timezone']
            ]);
        }

        return $settings;
    }

    /**
     * Save settings
     */
    private function saveSettings($companyId, $branchId, $settingsData) {
        foreach ($settingsData as $key => $value) {
            $this->saveSetting($companyId, $branchId, $key, $value);
        }
    }

    /**
     * Save individual setting
     */
    private function saveSetting($companyId, $branchId, $key, $value) {
        $existing = $this->db->query("
            SELECT id FROM settings
            WHERE company_id = ? AND branch_id <=> ? AND setting_key = ?
        ")->bind(1, $companyId)->bind(2, $branchId)->bind(3, $key)->single();

        $settingType = $this->detectSettingType($value);

        if ($existing) {
            $this->db->update('settings', [
                'setting_value' => (string) $value,
                'setting_type' => $settingType,
                'updated_at' => date('Y-m-d H:i:s')
            ], ['id' => $existing['id']]);
        } else {
            $this->db->insert('settings', [
                'company_id' => $companyId,
                'branch_id' => $branchId,
                'setting_key' => $key,
                'setting_value' => (string) $value,
                'setting_type' => $settingType,
                'is_public' => false
            ]);
        }
    }

    /**
     * Update company record
     */
    private function updateCompanyRecord($companyId, $data) {
        $updateData = [];

        if (isset($data['company_name'])) $updateData['name'] = $data['company_name'];
        if (isset($data['company_email'])) $updateData['email'] = $data['company_email'];
        if (isset($data['company_phone'])) $updateData['phone'] = $data['company_phone'];
        if (isset($data['company_address'])) $updateData['address'] = $data['company_address'];
        if (isset($data['company_description'])) $updateData['description'] = $data['company_description'];
        if (isset($data['primary_color'])) $updateData['primary_color'] = $data['primary_color'];
        if (isset($data['secondary_color'])) $updateData['secondary_color'] = $data['secondary_color'];
        if (isset($data['currency'])) $updateData['currency'] = $data['currency'];
        if (isset($data['timezone'])) $updateData['timezone'] = $data['timezone'];

        if (!empty($updateData)) {
            $this->db->update('companies', $updateData, ['id' => $companyId]);
        }
    }

    /**
     * Cast setting value based on type
     */
    private function castSettingValue($value, $type) {
        switch ($type) {
            case 'boolean':
                return filter_var($value, FILTER_VALIDATE_BOOLEAN);
            case 'number':
                return is_numeric($value) ? (float) $value : $value;
            case 'json':
                $decoded = json_decode($value, true);
                return $decoded !== null ? $decoded : $value;
            default:
                return $value;
        }
    }

    /**
     * Detect setting type
     */
    private function detectSettingType($value) {
        if (is_bool($value)) return 'boolean';
        if (is_numeric($value)) return 'number';
        if (is_array($value) || (is_string($value) && json_decode($value) !== null)) return 'json';
        return 'string';
    }

    /**
     * Validate settings input
     */
    private function validateSettings($data) {
        $errors = [];

        // Email validation
        if (!empty($data['company_email']) && !filter_var($data['company_email'], FILTER_VALIDATE_EMAIL)) {
            $errors['company_email'] = 'Invalid email format';
        }

        // Phone validation
        if (!empty($data['company_phone']) && !preg_match('/^[\+]?[1-9][\d]{0,15}$/', $data['company_phone'])) {
            $errors['company_phone'] = 'Invalid phone number format';
        }

        // Color validation (hex)
        if (!empty($data['primary_color']) && !preg_match('/^#[a-fA-F0-9]{6}$/', $data['primary_color'])) {
            $errors['primary_color'] = 'Invalid color format (use #RRGGBB)';
        }

        if (!empty($data['secondary_color']) && !preg_match('/^#[a-fA-F0-9]{6}$/', $data['secondary_color'])) {
            $errors['secondary_color'] = 'Invalid color format (use #RRGGBB)';
        }

        // Numeric validations
        if (isset($data['booking_minimum_advance_hours']) && $data['booking_minimum_advance_hours'] < 0) {
            $errors['booking_minimum_advance_hours'] = 'Must be a positive number';
        }

        if (isset($data['booking_maximum_advance_days']) && $data['booking_maximum_advance_days'] < 1) {
            $errors['booking_maximum_advance_days'] = 'Must be at least 1 day';
        }

        if (isset($data['tax_rate']) && ($data['tax_rate'] < 0 || $data['tax_rate'] > 100)) {
            $errors['tax_rate'] = 'Tax rate must be between 0 and 100';
        }

        return $errors;
    }
}
