<?php
/**
 * Admin Service Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class ServiceController extends Controller {
    public function index() {
        $companyId = $this->auth->companyId();

        // Get filter parameters
        $category = $this->getQuery('category');
        $status = $this->getQuery('status', 'all');
        $search = $this->getQuery('search');
        $page = (int) $this->getQuery('page', 1);

        // Get categories for filter
        $categories = $this->db->query("
            SELECT id, name FROM service_categories
            WHERE company_id = ? OR company_id IS NULL
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        // Build where clause
        $where = ["s.company_id = ?"];
        $params = [$companyId];

        if ($category) {
            $where[] = "s.category_id = ?";
            $params[] = $category;
        }

        if ($status === 'active') {
            $where[] = "s.is_active = 1";
        } elseif ($status === 'inactive') {
            $where[] = "s.is_active = 0";
        }

        if ($search) {
            $where[] = "(s.name LIKE ? OR s.description LIKE ?)";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
        }

        $whereClause = implode(' AND ', $where);

        // Get services with pagination
        $offset = ($page - 1) * ADMIN_ITEMS_PER_PAGE;

        $services = $this->db->query("
            SELECT s.*, sc.name as category_name,
                   COUNT(b.id) as total_bookings,
                   COUNT(CASE WHEN b.status = 'completed' THEN 1 END) as completed_bookings,
                   COALESCE(AVG(b.total_amount), 0) as avg_booking_value
            FROM services s
            LEFT JOIN service_categories sc ON s.category_id = sc.id
            LEFT JOIN bookings b ON s.id = b.service_id
            WHERE {$whereClause}
            GROUP BY s.id
            ORDER BY s.name ASC
            LIMIT ? OFFSET ?
        ", array_merge($params, [ADMIN_ITEMS_PER_PAGE, $offset]))->resultSet();

        // Get total count
        $totalServices = $this->db->query("
            SELECT COUNT(*) as total FROM services s WHERE {$whereClause}
        ", $params)->single()['total'];

        $this->adminView('services/index', [
            'services' => $services,
            'categories' => $categories,
            'filters' => [
                'category' => $category,
                'status' => $status,
                'search' => $search,
                'page' => $page
            ],
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalServices / ADMIN_ITEMS_PER_PAGE),
                'total_items' => $totalServices
            ],
            'pageTitle' => 'Services Management'
        ]);
    }

    public function create() {
        $companyId = $this->auth->companyId();

        // Get categories
        $categories = $this->db->query("
            SELECT id, name FROM service_categories
            WHERE company_id = ? OR company_id IS NULL
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        $this->adminView('services/create', [
            'categories' => $categories,
            'pageTitle' => 'Add New Service'
        ]);
    }

    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/admin/services/create');
        }

        $data = [
            'company_id' => $this->auth->companyId(),
            'category_id' => $this->getPost('category_id'),
            'name' => $this->getPost('name'),
            'slug' => generateSlug($this->getPost('name')),
            'description' => $this->getPost('description'),
            'short_description' => $this->getPost('short_description'),
            'price' => (float) $this->getPost('price'),
            'duration_minutes' => (int) $this->getPost('duration_minutes'),
            'preparation_time' => (int) $this->getPost('preparation_time', 0),
            'cleanup_time' => (int) $this->getPost('cleanup_time', 0),
            'is_active' => $this->getPost('is_active') ? 1 : 0,
            'is_featured' => $this->getPost('is_featured') ? 1 : 0,
            'max_capacity' => (int) $this->getPost('max_capacity', 1),
            'requires_staff' => $this->getPost('requires_staff') ? 1 : 0
        ];

        // Validate input
        $errors = $this->validateServiceInput($data);

        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/services/create');
        }

        try {
            $serviceId = $this->db->insert('services', $data);

            // Handle image upload if provided
            if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
                $imageUrl = $this->uploadServiceImage($serviceId, $_FILES['image']);
                if ($imageUrl) {
                    $this->db->update('services', ['image' => $imageUrl], ['id' => $serviceId]);
                }
            }

            $this->redirect(BASE_URL . '/admin/services', 'Service created successfully!');

        } catch (Exception $e) {
            error_log("Service creation error: " . $e->getMessage());
            $_SESSION['form_errors'] = ['general' => 'Failed to create service'];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/services/create');
        }
    }

    public function show($id) {
        $service = $this->db->query("
            SELECT s.*, sc.name as category_name
            FROM services s
            LEFT JOIN service_categories sc ON s.category_id = sc.id
            WHERE s.id = ? AND s.company_id = ?
        ")->bind(1, $id)->bind(2, $this->auth->companyId())->single();

        if (!$service) {
            $this->adminView('errors/404');
            return;
        }

        // Get service statistics
        $stats = $this->db->query("
            SELECT
                COUNT(*) as total_bookings,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_bookings,
                COUNT(CASE WHEN status = 'cancelled' THEN 1 END) as cancelled_bookings,
                COALESCE(SUM(CASE WHEN status = 'completed' THEN total_amount END), 0) as total_revenue,
                COALESCE(AVG(CASE WHEN status = 'completed' THEN total_amount END), 0) as avg_booking_value
            FROM bookings
            WHERE service_id = ?
        ")->bind(1, $id)->single();

        // Get recent bookings for this service
        $recentBookings = $this->db->query("
            SELECT b.*, u.first_name, u.last_name, st.first_name as staff_first_name, st.last_name as staff_last_name
            FROM bookings b
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN users st ON b.staff_id = st.id
            WHERE b.service_id = ?
            ORDER BY b.created_at DESC
            LIMIT 10
        ")->bind(1, $id)->resultSet();

        $this->adminView('services/show', [
            'service' => $service,
            'stats' => $stats,
            'recentBookings' => $recentBookings,
            'pageTitle' => 'Service Details - ' . $service['name']
        ]);
    }

    public function edit($id) {
        $service = $this->db->query("
            SELECT s.*, sc.name as category_name
            FROM services s
            LEFT JOIN service_categories sc ON s.category_id = sc.id
            WHERE s.id = ? AND s.company_id = ?
        ")->bind(1, $id)->bind(2, $this->auth->companyId())->single();

        if (!$service) {
            $this->adminView('errors/404');
            return;
        }

        $companyId = $this->auth->companyId();

        // Get categories
        $categories = $this->db->query("
            SELECT id, name FROM service_categories
            WHERE company_id = ? OR company_id IS NULL
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        $this->adminView('services/edit', [
            'service' => $service,
            'categories' => $categories,
            'pageTitle' => 'Edit Service - ' . $service['name']
        ]);
    }

    public function update($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/admin/services/' . $id . '/edit');
        }

        $service = $this->db->query("SELECT * FROM services WHERE id = ? AND company_id = ?")
                           ->bind(1, $id)->bind(2, $this->auth->companyId())->single();

        if (!$service) {
            $this->adminView('errors/404');
            return;
        }

        $data = [
            'category_id' => $this->getPost('category_id'),
            'name' => $this->getPost('name'),
            'slug' => generateSlug($this->getPost('name')),
            'description' => $this->getPost('description'),
            'short_description' => $this->getPost('short_description'),
            'price' => (float) $this->getPost('price'),
            'duration_minutes' => (int) $this->getPost('duration_minutes'),
            'preparation_time' => (int) $this->getPost('preparation_time', 0),
            'cleanup_time' => (int) $this->getPost('cleanup_time', 0),
            'is_active' => $this->getPost('is_active') ? 1 : 0,
            'is_featured' => $this->getPost('is_featured') ? 1 : 0,
            'max_capacity' => (int) $this->getPost('max_capacity', 1),
            'requires_staff' => $this->getPost('requires_staff') ? 1 : 0,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        // Validate input
        $errors = $this->validateServiceInput($data, $id);

        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/services/' . $id . '/edit');
        }

        try {
            $this->db->update('services', $data, ['id' => $id]);

            // Handle image upload if provided
            if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
                $imageUrl = $this->uploadServiceImage($id, $_FILES['image']);
                if ($imageUrl) {
                    $this->db->update('services', ['image' => $imageUrl], ['id' => $id]);
                }
            }

            $this->redirect(BASE_URL . '/admin/services', 'Service updated successfully!');

        } catch (Exception $e) {
            error_log("Service update error: " . $e->getMessage());
            $_SESSION['form_errors'] = ['general' => 'Failed to update service'];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/services/' . $id . '/edit');
        }
    }

    public function destroy($id) {
        $service = $this->db->query("SELECT * FROM services WHERE id = ? AND company_id = ?")
                           ->bind(1, $id)->bind(2, $this->auth->companyId())->single();

        if (!$service) {
            $this->json(['error' => 'Service not found'], 404);
        }

        // Check if service has bookings
        $bookingCount = $this->db->query("SELECT COUNT(*) as count FROM bookings WHERE service_id = ?")
                                ->bind(1, $id)->single()['count'];

        if ($bookingCount > 0) {
            $this->json(['error' => 'Cannot delete service with existing bookings'], 400);
        }

        try {
            $this->db->delete('services', ['id' => $id]);
            $this->json(['success' => true, 'message' => 'Service deleted successfully']);

        } catch (Exception $e) {
            error_log("Service deletion error: " . $e->getMessage());
            $this->json(['error' => 'Failed to delete service'], 500);
        }
    }

    public function toggleStatus($id) {
        $service = $this->db->query("SELECT * FROM services WHERE id = ? AND company_id = ?")
                           ->bind(1, $id)->bind(2, $this->auth->companyId())->single();

        if (!$service) {
            $this->json(['error' => 'Service not found'], 404);
        }

        $newStatus = $service['is_active'] ? 0 : 1;

        try {
            $this->db->update('services', ['is_active' => $newStatus], ['id' => $id]);
            $this->json([
                'success' => true,
                'message' => 'Service status updated successfully',
                'new_status' => $newStatus
            ]);

        } catch (Exception $e) {
            error_log("Service status toggle error: " . $e->getMessage());
            $this->json(['error' => 'Failed to update service status'], 500);
        }
    }

    /**
     * Validate service input
     */
    private function validateServiceInput($data, $excludeId = null) {
        $errors = [];

        if (empty($data['name'])) {
            $errors['name'] = 'Service name is required';
        }

        if (empty($data['price']) || $data['price'] <= 0) {
            $errors['price'] = 'Valid price is required';
        }

        if (empty($data['duration_minutes']) || $data['duration_minutes'] <= 0) {
            $errors['duration_minutes'] = 'Valid duration is required';
        }

        // Check for duplicate name
        $query = "SELECT COUNT(*) as count FROM services WHERE name = ? AND company_id = ?";
        $params = [$data['name'], $this->auth->companyId()];

        if ($excludeId) {
            $query .= " AND id != ?";
            $params[] = $excludeId;
        }

        $duplicate = $this->db->query($query, $params)->single();
        if ($duplicate['count'] > 0) {
            $errors['name'] = 'Service name already exists';
        }

        return $errors;
    }

    /**
     * Upload service image
     */
    private function uploadServiceImage($serviceId, $file) {
        try {
            if (!in_array($file['type'], ALLOWED_IMAGE_TYPES)) {
                return false;
            }

            if ($file['size'] > MAX_FILE_SIZE) {
                return false;
            }

            // Create upload directory
            $uploadDir = UPLOADS_PATH . '/services/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }

            // Generate unique filename
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'service_' . $serviceId . '_' . time() . '.' . $extension;
            $filepath = $uploadDir . $filename;

            if (move_uploaded_file($file['tmp_name'], $filepath)) {
                return BASE_URL . '/uploads/services/' . $filename;
            }

            return false;

        } catch (Exception $e) {
            error_log("Service image upload error: " . $e->getMessage());
            return false;
        }
    }
}