<?php
/**
 * Admin Dashboard Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class DashboardController extends Controller {
    public function index() {
        // Get company/branch context
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();

        // Get today's stats
        $todayStats = $this->getTodayStats($companyId, $branchId);

        // Get recent bookings
        $recentBookings = $this->getRecentBookings($companyId, $branchId, 10);

        // Get upcoming bookings
        $upcomingBookings = $this->getUpcomingBookings($companyId, $branchId, 5);

        // Get monthly revenue chart data
        $revenueData = $this->getMonthlyRevenue($companyId, $branchId);

        // Get popular services
        $popularServices = $this->getPopularServices($companyId, $branchId, 5);

        // Get staff performance
        $staffPerformance = $this->getStaffPerformance($companyId, $branchId);

        $this->adminView('dashboard/index', [
            'todayStats' => $todayStats,
            'recentBookings' => $recentBookings,
            'upcomingBookings' => $upcomingBookings,
            'revenueData' => $revenueData,
            'popularServices' => $popularServices,
            'staffPerformance' => $staffPerformance,
            'pageTitle' => 'Dashboard'
        ]);
    }

    /**
     * Get today's statistics
     */
    private function getTodayStats($companyId, $branchId) {
        $today = date('Y-m-d');
        $whereClause = "DATE(b.created_at) = ?";
        $params = [$today];

        if ($branchId) {
            $whereClause .= " AND b.branch_id = ?";
            $params[] = $branchId;
        } elseif ($companyId) {
            $whereClause .= " AND b.company_id = ?";
            $params[] = $companyId;
        }

        $stats = $this->db->query("
            SELECT
                COUNT(*) as total_bookings,
                COUNT(CASE WHEN b.status = 'confirmed' THEN 1 END) as confirmed_bookings,
                COUNT(CASE WHEN b.status = 'pending' THEN 1 END) as pending_bookings,
                COUNT(CASE WHEN b.status = 'completed' THEN 1 END) as completed_bookings,
                COUNT(CASE WHEN b.status = 'cancelled' THEN 1 END) as cancelled_bookings,
                SUM(CASE WHEN b.status = 'completed' THEN b.total_amount ELSE 0 END) as today_revenue
            FROM bookings b
            WHERE {$whereClause}
        ", $params)->single();

        // Get new customers today
        $newCustomers = $this->db->query("
            SELECT COUNT(*) as new_customers
            FROM users u
            WHERE DATE(u.created_at) = ? AND u.role = 'customer'
            " . ($companyId ? " AND u.company_id = ?" : "")
        ", $companyId ? [$today, $companyId] : [$today])->single();

        return array_merge($stats, $newCustomers);
    }

    /**
     * Get recent bookings
     */
    private function getRecentBookings($companyId, $branchId, $limit = 10) {
        $whereClause = "";
        $params = [];

        if ($branchId) {
            $whereClause = "b.branch_id = ?";
            $params[] = $branchId;
        } elseif ($companyId) {
            $whereClause = "b.company_id = ?";
            $params[] = $companyId;
        }

        $sql = "
            SELECT b.*, s.name as service_name, u.first_name, u.last_name,
                   br.name as branch_name, st.first_name as staff_first_name,
                   st.last_name as staff_last_name
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN branches br ON b.branch_id = br.id
            LEFT JOIN users st ON b.staff_id = st.id
            " . ($whereClause ? "WHERE {$whereClause}" : "") . "
            ORDER BY b.created_at DESC
            LIMIT ?
        ";

        $params[] = $limit;

        return $this->db->query($sql, $params)->resultSet();
    }

    /**
     * Get upcoming bookings
     */
    private function getUpcomingBookings($companyId, $branchId, $limit = 5) {
        $whereClause = "b.start_at > NOW() AND b.status IN ('confirmed', 'pending')";
        $params = [];

        if ($branchId) {
            $whereClause .= " AND b.branch_id = ?";
            $params[] = $branchId;
        } elseif ($companyId) {
            $whereClause .= " AND b.company_id = ?";
            $params[] = $companyId;
        }

        $sql = "
            SELECT b.*, s.name as service_name, s.duration_minutes,
                   u.first_name, u.last_name, br.name as branch_name,
                   st.first_name as staff_first_name, st.last_name as staff_last_name
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN branches br ON b.branch_id = br.id
            LEFT JOIN users st ON b.staff_id = st.id
            WHERE {$whereClause}
            ORDER BY b.start_at ASC
            LIMIT ?
        ";

        $params[] = $limit;

        return $this->db->query($sql, $params)->resultSet();
    }

    /**
     * Get monthly revenue data for charts
     */
    private function getMonthlyRevenue($companyId, $branchId) {
        $whereClause = "b.status = 'completed'";
        $params = [];

        if ($branchId) {
            $whereClause .= " AND b.branch_id = ?";
            $params[] = $branchId;
        } elseif ($companyId) {
            $whereClause .= " AND b.company_id = ?";
            $params[] = $companyId;
        }

        // Get last 12 months
        $revenueData = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = date('Y-m-01', strtotime("-{$i} months"));
            $monthName = date('M Y', strtotime($date));

            $monthRevenue = $this->db->query("
                SELECT COALESCE(SUM(b.total_amount), 0) as revenue
                FROM bookings b
                WHERE {$whereClause} AND DATE_FORMAT(b.start_at, '%Y-%m') = DATE_FORMAT(?, '%Y-%m')
            ", array_merge($params, [$date]))->single();

            $revenueData[] = [
                'month' => $monthName,
                'revenue' => (float) $monthRevenue['revenue']
            ];
        }

        return $revenueData;
    }

    /**
     * Get popular services
     */
    private function getPopularServices($companyId, $branchId, $limit = 5) {
        $whereClause = "";
        $params = [];

        if ($branchId) {
            $whereClause = "b.branch_id = ?";
            $params[] = $branchId;
        } elseif ($companyId) {
            $whereClause = "b.company_id = ?";
            $params[] = $companyId;
        }

        $sql = "
            SELECT s.name, s.id, COUNT(b.id) as booking_count,
                   SUM(b.total_amount) as total_revenue
            FROM services s
            LEFT JOIN bookings b ON s.id = b.service_id AND b.status = 'completed'
            " . ($whereClause ? "WHERE {$whereClause}" : "") . "
            GROUP BY s.id, s.name
            ORDER BY booking_count DESC
            LIMIT ?
        ";

        $params[] = $limit;

        return $this->db->query($sql, $params)->resultSet();
    }

    /**
     * Get staff performance
     */
    private function getStaffPerformance($companyId, $branchId) {
        $whereClause = "";
        $params = [];

        if ($branchId) {
            $whereClause = "u.branch_id = ?";
            $params[] = $branchId;
        } elseif ($companyId) {
            $whereClause = "u.company_id = ?";
            $params[] = $companyId;
        }

        $sql = "
            SELECT u.first_name, u.last_name, u.id,
                   COUNT(b.id) as total_bookings,
                   COUNT(CASE WHEN b.status = 'completed' THEN 1 END) as completed_bookings,
                   COALESCE(SUM(b.total_amount), 0) as total_revenue
            FROM users u
            LEFT JOIN bookings b ON u.id = b.staff_id AND b.status = 'completed'
            WHERE u.role = 'staff' AND u.status = 'active'
            " . ($whereClause ? "AND {$whereClause}" : "") . "
            GROUP BY u.id, u.first_name, u.last_name
            ORDER BY total_bookings DESC
            LIMIT 10
        ";

        return $this->db->query($sql, $params)->resultSet();
    }
}
