<?php
/**
 * Sitemap Generator Script
 * Salon/Spa/Beauty Parlor Booking System
 *
 * This script generates sitemap.xml for SEO
 * Should be run via CRON job weekly
 * Example CRON: 0 0 * * 0 php /path/to/salon-system/scripts/generate-sitemap.php
 */

// Include configuration
require_once __DIR__ . '/../config/config.php';

echo "Starting sitemap generation at " . date('Y-m-d H:i:s') . "\n";

try {
    // Initialize database
    $db = Database::getInstance();

    // Base URL for the sitemap
    $baseUrl = rtrim(BASE_URL, '/');

    // Start building sitemap XML
    $sitemap = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $sitemap .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"' . "\n";
    $sitemap .= '        xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"' . "\n";
    $sitemap .= '        xsi:schemaLocation="http://www.sitemaps.org/schemas/sitemap/0.9' . "\n";
    $sitemap .= '        http://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd">' . "\n";

    // Add static pages
    $staticPages = [
        '/' => ['priority' => '1.0', 'changefreq' => 'daily'],
        '/services' => ['priority' => '0.9', 'changefreq' => 'weekly'],
        '/about' => ['priority' => '0.7', 'changefreq' => 'monthly'],
        '/contact' => ['priority' => '0.8', 'changefreq' => 'monthly'],
        '/gallery' => ['priority' => '0.6', 'changefreq' => 'weekly'],
        '/book' => ['priority' => '0.9', 'changefreq' => 'daily']
    ];

    foreach ($staticPages as $url => $options) {
        $sitemap .= "  <url>\n";
        $sitemap .= "    <loc>{$baseUrl}{$url}</loc>\n";
        $sitemap .= "    <lastmod>" . date('Y-m-d') . "</lastmod>\n";
        $sitemap .= "    <changefreq>{$options['changefreq']}</changefreq>\n";
        $sitemap .= "    <priority>{$options['priority']}</priority>\n";
        $sitemap .= "  </url>\n";
    }

    // Add service pages
    $services = $db->query("
        SELECT slug, updated_at
        FROM services
        WHERE is_active = 1 AND slug IS NOT NULL
        ORDER BY updated_at DESC
    ")->resultSet();

    foreach ($services as $service) {
        $sitemap .= "  <url>\n";
        $sitemap .= "    <loc>{$baseUrl}/services/{$service['slug']}</loc>\n";
        $sitemap .= "    <lastmod>" . date('Y-m-d', strtotime($service['updated_at'])) . "</lastmod>\n";
        $sitemap .= "    <changefreq>weekly</changefreq>\n";
        $sitemap .= "    <priority>0.8</priority>\n";
        $sitemap .= "  </url>\n";
    }

    // Add branch pages
    $branches = $db->query("
        SELECT slug, updated_at
        FROM branches
        WHERE status = 'active' AND slug IS NOT NULL
        ORDER BY updated_at DESC
    ")->resultSet();

    foreach ($branches as $branch) {
        $sitemap .= "  <url>\n";
        $sitemap .= "    <loc>{$baseUrl}/branches/{$branch['slug']}</loc>\n";
        $sitemap .= "    <lastmod>" . date('Y-m-d', strtotime($branch['updated_at'])) . "</lastmod>\n";
        $sitemap .= "    <changefreq>monthly</changefreq>\n";
        $sitemap .= "    <priority>0.7</priority>\n";
        $sitemap .= "  </url>\n";
    }

    // Add blog/news pages (if implemented)
    // Add product pages (if shop is active)

    // Close sitemap
    $sitemap .= '</urlset>';

    // Save sitemap to public directory
    $sitemapPath = PUBLIC_PATH . '/sitemap.xml';
    if (file_put_contents($sitemapPath, $sitemap)) {
        echo "Sitemap generated successfully at {$sitemapPath}\n";
        echo "Total URLs: " . (count($staticPages) + count($services) + count($branches)) . "\n";
    } else {
        throw new Exception("Failed to write sitemap to {$sitemapPath}");
    }

    // Generate robots.txt if it doesn't exist
    $this->generateRobotsTxt();

} catch (Exception $e) {
    error_log("Sitemap generation error: " . $e->getMessage());
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}

echo "Sitemap generation completed at " . date('Y-m-d H:i:s') . "\n";

/**
 * Generate robots.txt file
 */
function generateRobotsTxt() {
    $baseUrl = rtrim(BASE_URL, '/');
    $domain = parse_url($baseUrl, PHP_URL_HOST);

    $robotsTxt = "User-agent: *\n";
    $robotsTxt .= "Allow: /\n\n";

    // Disallow admin areas
    $robotsTxt .= "Disallow: /admin/\n";
    $robotsTxt .= "Disallow: /staff/\n";
    $robotsTxt .= "Disallow: /api/\n";
    $robotsTxt .= "Disallow: /uploads/private/\n";
    $robotsTxt .= "Disallow: /*?*\n\n"; // Disallow URLs with query parameters

    // Sitemap reference
    $robotsTxt .= "Sitemap: {$baseUrl}/sitemap.xml\n";

    // Save robots.txt
    $robotsPath = PUBLIC_PATH . '/robots.txt';
    if (file_put_contents($robotsPath, $robotsTxt)) {
        echo "robots.txt generated successfully at {$robotsPath}\n";
    } else {
        echo "Warning: Failed to generate robots.txt\n";
    }
}
