<?php
/**
 * WhatsApp Business API Adapter
 * Salon/Spa/Beauty Parlor Booking System
 */

class WhatsAppAdapter {
    private $accessToken;
    private $phoneNumberId;
    private $businessAccountId;
    private $apiVersion = 'v18.0';

    public function __construct() {
        $this->accessToken = WHATSAPP_ACCESS_TOKEN ?: null;
        $this->phoneNumberId = WHATSAPP_PHONE_NUMBER_ID ?: null;
        $this->businessAccountId = WHATSAPP_BUSINESS_ACCOUNT_ID ?: null;
    }

    /**
     * Send WhatsApp message
     */
    public function send($to, $message, $templateId = null, $templateData = []) {
        try {
            // Remove any non-numeric characters from phone number
            $to = preg_replace('/\D/', '', $to);

            // Add country code if not present (assuming international format)
            if (strlen($to) === 9 && !str_starts_with($to, '254')) {
                // For Kenya, add 254 prefix if not present
                $to = '254' . $to;
            }

            // Check if WhatsApp API is configured
            if (!$this->accessToken || !$this->phoneNumberId) {
                // Log the message instead of sending
                error_log("WhatsApp message not sent (API not configured): To: {$to}, Message: " . substr($message, 0, 100) . "...");
                return [
                    'success' => true,
                    'message' => 'Message logged (WhatsApp API not configured)',
                    'message_id' => 'logged_' . time()
                ];
            }

            // Use template if provided
            if ($templateId) {
                return $this->sendTemplateMessage($to, $templateId, $templateData);
            }

            // Send text message
            $result = $this->sendTextMessage($to, $message);

            return [
                'success' => $result['success'] ?? false,
                'message' => $result['message'] ?? 'Message sent',
                'message_id' => $result['message_id'] ?? null
            ];

        } catch (Exception $e) {
            error_log("WhatsApp sending error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'WhatsApp message failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Send text message
     */
    private function sendTextMessage($to, $message) {
        $url = "https://graph.facebook.com/{$this->apiVersion}/{$this->phoneNumberId}/messages";

        $data = [
            'messaging_product' => 'whatsapp',
            'to' => $to,
            'type' => 'text',
            'text' => [
                'body' => $message
            ]
        ];

        return $this->makeAPIRequest($url, $data);
    }

    /**
     * Send template message
     */
    private function sendTemplateMessage($to, $templateName, $templateData) {
        $url = "https://graph.facebook.com/{$this->apiVersion}/{$this->phoneNumberId}/messages";

        $data = [
            'messaging_product' => 'whatsapp',
            'to' => $to,
            'type' => 'template',
            'template' => [
                'name' => $templateName,
                'language' => [
                    'code' => 'en_US'
                ]
            ]
        ];

        // Add template components if data is provided
        if (!empty($templateData)) {
            $components = [];

            // Add body parameters
            if (isset($templateData['body'])) {
                $components[] = [
                    'type' => 'body',
                    'parameters' => array_map(function($param) {
                        return ['type' => 'text', 'text' => $param];
                    }, $templateData['body'])
                ];
            }

            if (!empty($components)) {
                $data['template']['components'] = $components;
            }
        }

        return $this->makeAPIRequest($url, $data);
    }

    /**
     * Make API request to WhatsApp Business API
     */
    private function makeAPIRequest($url, $data) {
        // For development/testing, log the request instead of sending
        if (IS_DEVELOPMENT) {
            error_log("WHATSAPP API CALL: {$url}");
            error_log("DATA: " . json_encode($data));
            return [
                'success' => true,
                'message' => 'WhatsApp message logged (development mode)',
                'message_id' => 'dev_' . time()
            ];
        }

        $ch = curl_init($url);

        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->accessToken,
                'Content-Type: application/json'
            ],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 30
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);

        curl_close($ch);

        if ($error) {
            throw new Exception("cURL Error: {$error}");
        }

        $responseData = json_decode($response, true);

        if ($httpCode >= 200 && $httpCode < 300) {
            return [
                'success' => true,
                'message' => 'Message sent successfully',
                'message_id' => $responseData['messages'][0]['id'] ?? null,
                'response' => $responseData
            ];
        } else {
            $errorMessage = $responseData['error']['message'] ?? 'Unknown error';
            return [
                'success' => false,
                'message' => "WhatsApp API Error: {$errorMessage}",
                'response' => $responseData
            ];
        }
    }

    /**
     * Check if phone number is valid for WhatsApp
     */
    public function validatePhoneNumber($phoneNumber) {
        // Remove any non-numeric characters
        $phoneNumber = preg_replace('/\D/', '', $phoneNumber);

        // Basic validation for international format
        if (strlen($phoneNumber) < 10 || strlen($phoneNumber) > 15) {
            return false;
        }

        // Check if it starts with a valid country code
        $validPrefixes = ['1', '7', '20', '27', '30', '31', '32', '33', '34', '36', '39', '40', '41', '43', '44', '45', '46', '47', '48', '49', '51', '52', '53', '54', '55', '56', '57', '58', '60', '61', '62', '63', '64', '65', '66', '81', '82', '84', '86', '90', '91', '92', '93', '94', '95', '98', '212', '213', '216', '218', '220', '221', '222', '223', '224', '225', '226', '227', '228', '229', '230', '231', '232', '233', '234', '235', '236', '237', '238', '239', '240', '241', '242', '243', '244', '245', '246', '247', '248', '249', '250', '251', '252', '253', '254', '255', '256', '257', '258', '260', '261', '262', '263', '264', '265', '266', '267', '268', '269', '290', '291', '297', '298', '299', '350', '351', '352', '353', '354', '355', '356', '357', '358', '359', '370', '371', '372', '373', '374', '375', '376', '377', '378', '380', '381', '382', '383', '385', '386', '387', '389', '420', '421', '423', '500', '501', '502', '503', '504', '505', '506', '507', '508', '509', '590', '591', '592', '593', '594', '595', '596', '597', '598', '599', '670', '672', '673', '674', '675', '676', '677', '678', '679', '680', '681', '682', '683', '684', '685', '686', '687', '688', '689', '690', '691', '692', '850', '852', '853', '855', '856', '880', '886', '960', '961', '962', '963', '964', '965', '966', '967', '968', '970', '971', '972', '973', '974', '975', '976', '977', '992', '993', '994', '995', '996', '998'];

        foreach ($validPrefixes as $prefix) {
            if (str_starts_with($phoneNumber, $prefix)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get WhatsApp templates for booking confirmations
     */
    public static function getBookingConfirmationTemplate() {
        return [
            'name' => 'booking_confirmation',
            'body' => [
                'Hi {{customer_name}}! 🎉',
                'Your booking #{{booking_number}} has been confirmed.',
                '{{service_name}} on {{booking_date}} at {{booking_time}}',
                'Staff: {{staff_name}} | Branch: {{branch_name}}',
                'Total: ${{total_amount}}',
                'See you soon! 💅'
            ]
        ];
    }

    /**
     * Get WhatsApp templates for booking reminders
     */
    public static function getBookingReminderTemplate() {
        return [
            'name' => 'booking_reminder',
            'body' => [
                'Hi {{customer_name}}! ⏰',
                'Friendly reminder of your appointment tomorrow.',
                '{{service_name}} at {{booking_time}}',
                'Staff: {{staff_name}} | Branch: {{branch_name}}',
                'Booking #{{booking_number}}',
                'See you soon! 💅'
            ]
        ];
    }

    /**
     * Send bulk WhatsApp messages
     */
    public function sendBulk($recipients, $message, $templateId = null, $templateData = []) {
        $results = [];

        foreach ($recipients as $recipient) {
            if (is_array($recipient)) {
                $phone = $recipient['phone'];
                $data = array_merge($templateData, $recipient);
            } else {
                $phone = $recipient;
                $data = $templateData;
            }

            $results[] = [
                'phone' => $phone,
                'result' => $this->send($phone, $message, $templateId, $data)
            ];
        }

        return $results;
    }
}
