<?php
/**
 * SMS Notification Adapter
 * Salon/Spa/Beauty Parlor Booking System
 * Supports multiple SMS providers (Twilio, SMPP, etc.)
 */

class SMSAdapter {
    private $provider;
    private $config;

    public function __construct() {
        // Default to Twilio, but can be extended for other providers
        $this->provider = getenv('SMS_PROVIDER') ?: 'twilio';
        $this->config = $this->loadProviderConfig();
    }

    /**
     * Send SMS message
     */
    public function send($to, $message, $templateId = null, $templateData = []) {
        try {
            // Remove any non-numeric characters from phone number
            $to = preg_replace('/\D/', '', $to);

            // Check if SMS provider is configured
            if (!$this->config) {
                // Log the message instead of sending
                error_log("SMS not sent (provider not configured): To: {$to}, Message: " . substr($message, 0, 100) . "...");
                return [
                    'success' => true,
                    'message' => 'SMS logged (provider not configured)',
                    'message_id' => 'logged_' . time()
                ];
            }

            // Use template if provided
            if ($templateId) {
                $template = $this->loadTemplate($templateId);
                if ($template) {
                    $message = $this->replacePlaceholders($template, $templateData);
                }
            }

            // Send SMS using configured provider
            $result = $this->sendSMS($to, $message);

            return [
                'success' => $result['success'] ?? false,
                'message' => $result['message'] ?? 'SMS sent',
                'message_id' => $result['message_id'] ?? null
            ];

        } catch (Exception $e) {
            error_log("SMS sending error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'SMS sending failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Send SMS using Twilio
     */
    private function sendTwilioSMS($to, $message) {
        $accountSid = $this->config['account_sid'] ?? '';
        $authToken = $this->config['auth_token'] ?? '';
        $fromNumber = $this->config['from_number'] ?? '';

        if (!$accountSid || !$authToken || !$fromNumber) {
            throw new Exception('Twilio credentials not configured');
        }

        $url = "https://api.twilio.com/2010-04-01/Accounts/{$accountSid}/Messages.json";

        // For development/testing, log instead of sending
        if (IS_DEVELOPMENT) {
            error_log("TWILIO SMS: To: {$to}, From: {$fromNumber}, Message: " . substr($message, 0, 100) . "...");
            return [
                'success' => true,
                'message' => 'SMS logged (development mode)',
                'message_id' => 'dev_' . time()
            ];
        }

        $ch = curl_init($url);

        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query([
                'From' => $fromNumber,
                'To' => $to,
                'Body' => $message
            ]),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_USERPWD => $accountSid . ':' . $authToken,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded'
            ],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 30
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);

        curl_close($ch);

        if ($error) {
            throw new Exception("Twilio API Error: {$error}");
        }

        $responseData = json_decode($response, true);

        if ($httpCode >= 200 && $httpCode < 300) {
            return [
                'success' => true,
                'message' => 'SMS sent successfully',
                'message_id' => $responseData['sid'] ?? null,
                'response' => $responseData
            ];
        } else {
            $errorMessage = $responseData['message'] ?? 'Unknown error';
            return [
                'success' => false,
                'message' => "Twilio Error: {$errorMessage}",
                'response' => $responseData
            ];
        }
    }

    /**
     * Send SMS using SMPP (Simplified implementation)
     */
    private function sendSMPPSMS($to, $message) {
        // SMPP implementation would go here
        // This is a placeholder for SMPP integration
        error_log("SMPP SMS: To: {$to}, Message: " . substr($message, 0, 100) . "...");

        return [
            'success' => true,
            'message' => 'SMS sent via SMPP (placeholder)',
            'message_id' => 'smpp_' . time()
        ];
    }

    /**
     * Send SMS using configured provider
     */
    private function sendSMS($to, $message) {
        switch ($this->provider) {
            case 'twilio':
                return $this->sendTwilioSMS($to, $message);
            case 'smpp':
                return $this->sendSMPPSMS($to, $message);
            default:
                throw new Exception("Unsupported SMS provider: {$this->provider}");
        }
    }

    /**
     * Load provider configuration
     */
    private function loadProviderConfig() {
        switch ($this->provider) {
            case 'twilio':
                return [
                    'account_sid' => getenv('TWILIO_ACCOUNT_SID'),
                    'auth_token' => getenv('TWILIO_AUTH_TOKEN'),
                    'from_number' => getenv('TWILIO_FROM_NUMBER')
                ];
            case 'smpp':
                return [
                    'host' => getenv('SMPP_HOST'),
                    'port' => getenv('SMPP_PORT'),
                    'system_id' => getenv('SMPP_SYSTEM_ID'),
                    'password' => getenv('SMPP_PASSWORD'),
                    'from_number' => getenv('SMPP_FROM_NUMBER')
                ];
            default:
                return null;
        }
    }

    /**
     * Load SMS template
     */
    private function loadTemplate($templateId) {
        // This would load template from database
        // For now, return predefined templates
        $templates = [
            'booking_confirmation' => "Hi {{customer_name}}! Your booking #{{booking_number}} for {{service_name}} on {{booking_date}} at {{booking_time}} is confirmed. Total: ${{total_amount}}. See you soon!",
            'booking_reminder' => "Hi {{customer_name}}! Reminder: Your {{service_name}} appointment is tomorrow at {{booking_time}}. Booking #{{booking_number}}.",
            'booking_cancelled' => "Hi {{customer_name}}! Your booking #{{booking_number}} has been cancelled. Contact us to reschedule."
        ];

        return $templates[$templateId] ?? null;
    }

    /**
     * Replace placeholders in template
     */
    private function replacePlaceholders($template, $data) {
        foreach ($data as $key => $value) {
            $template = str_replace("{{{$key}}}", $value, $template);
        }
        return $template;
    }

    /**
     * Send bulk SMS messages
     */
    public function sendBulk($recipients, $message, $templateId = null, $templateData = []) {
        $results = [];

        foreach ($recipients as $recipient) {
            if (is_array($recipient)) {
                $phone = $recipient['phone'];
                $data = array_merge($templateData, $recipient);
            } else {
                $phone = $recipient;
                $data = $templateData;
            }

            $results[] = [
                'phone' => $phone,
                'result' => $this->send($phone, $message, $templateId, $data)
            ];
        }

        return $results;
    }

    /**
     * Check SMS credit balance (provider-specific)
     */
    public function checkBalance() {
        switch ($this->provider) {
            case 'twilio':
                return $this->checkTwilioBalance();
            default:
                return ['balance' => 'N/A', 'currency' => 'N/A'];
        }
    }

    /**
     * Check Twilio balance
     */
    private function checkTwilioBalance() {
        $accountSid = $this->config['account_sid'] ?? '';
        $authToken = $this->config['auth_token'] ?? '';

        if (!$accountSid || !$authToken) {
            return ['balance' => 'N/A', 'currency' => 'N/A'];
        }

        $url = "https://api.twilio.com/2010-04-01/Accounts/{$accountSid}/Balance.json";

        $ch = curl_init($url);

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_USERPWD => $accountSid . ':' . $authToken,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 30
        ]);

        $response = curl_exec($ch);
        $error = curl_error($ch);

        curl_close($ch);

        if ($error) {
            return ['balance' => 'Error', 'currency' => 'N/A'];
        }

        $responseData = json_decode($response, true);

        return [
            'balance' => $responseData['balance'] ?? 'N/A',
            'currency' => $responseData['currency'] ?? 'N/A'
        ];
    }

    /**
     * Validate phone number for SMS
     */
    public function validatePhoneNumber($phoneNumber) {
        // Remove any non-numeric characters
        $phoneNumber = preg_replace('/\D/', '', $phoneNumber);

        // Basic validation for international format
        if (strlen($phoneNumber) < 10 || strlen($phoneNumber) > 15) {
            return false;
        }

        return true;
    }
}
