# Luxury Beauty Spa - Booking & Management System

A comprehensive, production-ready Salon/Spa/Beauty Parlor booking and management system built with PHP, Tailwind CSS, and vanilla JavaScript. Features multi-tenant architecture, PWA capabilities, and mobile-first responsive design.

## 🚀 Features

### Core Features
- **Multi-tenant System**: Support for multiple companies, branches, and agents
- **Online Booking**: Elegant booking wizard with availability management
- **Staff Management**: Schedule management, commission tracking, performance reports
- **Service Management**: Categories, variants, pricing, and image galleries
- **Payment Processing**: M-Pesa integration, manual payment recording
- **E-commerce Shop**: Product catalog with cart and checkout functionality
- **Notification System**: Email, WhatsApp, and SMS notifications
- **Support Chat**: Internal ticketing system with customer support
- **Analytics & Reports**: Comprehensive reporting and business intelligence

### Technical Features
- **PWA Ready**: Progressive Web App with offline capabilities
- **Mobile-First**: Responsive design optimized for all devices
- **SEO Optimized**: Schema.org markup, meta tags, sitemap generation
- **Security**: CSRF protection, input validation, secure session management
- **Performance**: Lazy loading, image optimization, caching
- **Accessibility**: WCAG compliant with proper ARIA labels

## 📋 Requirements

- **PHP**: 7.4 or higher (8.0+ recommended)
- **MySQL**: 5.7 or higher (8.0+ recommended)
- **Web Server**: Apache/Nginx with mod_rewrite
- **Node.js & npm**: For building Tailwind CSS (optional for development)
- **SSL Certificate**: Required for production PWA features

## 🛠️ Installation

### 1. Clone the Repository
```bash
git clone https://github.com/your-repo/salon-system.git
cd salon-system
```

### 2. Install Dependencies
```bash
# Install PHP dependencies (if using Composer)
composer install

# Install Node.js dependencies (for Tailwind CSS)
npm install
```

### 3. Database Setup

#### Option A: If you have CREATE DATABASE privileges
```bash
# Run the database creation script
mysql -u your_username -p < create-database.sql

# Import the main schema
mysql -u your_username -p salon_system < schema.sql
```

#### Option B: Manual Database Creation (Recommended for shared hosting)
1. **Create database manually** through your hosting control panel (cPanel, Plesk, etc.) or phpMyAdmin:
   - Database name: `salon_system`
   - Character set: `utf8mb4`
   - Collation: `utf8mb4_unicode_ci`

2. **Import the schema**:
   ```bash
   # Import schema into existing database
   mysql -u your_username -p salon_system < schema.sql
   ```

#### Option C: Using phpMyAdmin
1. Open phpMyAdmin in your browser
2. Create a new database named `salon_system` with `utf8mb4_unicode_ci` collation
3. Select the database and click "Import"
4. Upload the `schema.sql` file
5. Click "Go" to import

#### Troubleshooting Database Permission Issues
If you get "#1044 - Access denied" error:

1. **Contact your hosting provider** to grant CREATE DATABASE privileges
2. **Use Option B** (manual database creation) above
3. **Ask your database administrator** to create the database for you
4. **Use a different MySQL user** that has the required privileges

**Note**: Most shared hosting providers don't give CREATE DATABASE privileges for security reasons. Use Option B for shared hosting.

### 4. Environment Configuration
```bash
# Copy environment file
cp .env.example .env

# Edit configuration
nano .env
```

Required environment variables:
```env
APP_ENV=production
APP_URL=https://yourdomain.com
DB_HOST=localhost
DB_NAME=salon_system
DB_USER=your_db_user
DB_PASS=your_db_password
```

### 5. Build Assets
```bash
# Build Tailwind CSS
npm run build

# For development with watch mode
npm run dev
```

### 6. Web Server Configuration

#### Apache (.htaccess)
```apache
RewriteEngine On
RewriteRule ^(.*)$ public/index.php [QSA,L]
```

#### Nginx
```nginx
server {
    listen 80;
    server_name yourdomain.com;
    root /path/to/salon-system/public;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        include fastcgi_params;
        fastcgi_pass unix:/var/run/php/php8.0-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
    }
}
```

### 7. Set Permissions
```bash
# Set proper permissions
chown -R www-data:www-data .
chmod -R 755 storage/
chmod -R 755 public/uploads/
```

### 8. CRON Jobs Setup
```bash
# Add to crontab for automated reminders
* * * * * php /path/to/salon-system/scripts/reminders.php

# Generate sitemap weekly
0 0 * * 0 php /path/to/salon-system/scripts/generate-sitemap.php
```

## 🔧 Configuration

### Payment Integration

#### M-Pesa Setup
1. Register for M-Pesa API access at [Safaricom Developer Portal](https://developer.safaricom.co.ke/)
2. Add credentials to `.env`:
```env
MPESA_CONSUMER_KEY=your-consumer-key
MPESA_CONSUMER_SECRET=your-consumer-secret
MPESA_SHORTCODE=your-shortcode
MPESA_PASSKEY=your-passkey
MPESA_ENVIRONMENT=production
```

#### Manual Payments
The system supports manual payment recording for cash and other payment methods.

### Notification Setup

#### SMTP Email
```env
SMTP_HOST=smtp.gmail.com
SMTP_PORT=587
SMTP_USER=your-email@gmail.com
SMTP_PASS=your-app-password
SMTP_ENCRYPTION=tls
```

#### WhatsApp Business API
1. Set up WhatsApp Business API
2. Add credentials to `.env`:
```env
WHATSAPP_ACCESS_TOKEN=your-access-token
WHATSAPP_PHONE_NUMBER_ID=your-phone-number-id
```

### File Uploads
- Maximum file size: 5MB (configurable in `config/config.php`)
- Supported formats: JPEG, PNG, GIF, WebP
- Images are stored in `public/uploads/` with randomized filenames

## 🚀 Usage

### Admin Access
- Default admin login: `admin@luxurybeautyspa.com` / `password`
- Access admin panel at: `https://yourdomain.com/admin`

### Staff Access
- Staff login: `sarah@luxurybeautyspa.com` / `password`
- Access staff panel at: `https://yourdomain.com/staff`

### Customer Features
- Browse services and book appointments
- View booking history and receipts
- Contact support through chat system
- Shop for products online

## 📱 PWA Features

The application includes Progressive Web App features:

- **Installable**: Add to home screen on mobile devices
- **Offline Support**: Basic functionality works offline
- **Push Notifications**: Booking reminders and updates
- **Background Sync**: Sync data when connection is restored

### PWA Installation
1. Ensure SSL certificate is installed
2. Service worker is automatically registered
3. Users will see "Add to Home Screen" prompt

## 🔒 Security

### Best Practices Implemented
- CSRF protection on all forms
- Input validation and sanitization
- Prepared statements for database queries
- Secure session management
- File upload validation
- Rate limiting on sensitive endpoints

### Security Checklist
- [ ] Change default admin passwords
- [ ] Use strong, unique passwords
- [ ] Enable SSL/HTTPS
- [ ] Keep software updated
- [ ] Regular security audits
- [ ] Monitor access logs

## 📊 API Endpoints

### Public API
```
GET  /api/services          # List services
GET  /api/services/{slug}   # Service details
GET  /api/branches          # List branches
GET  /api/availability      # Check availability
POST /api/bookings          # Create booking
POST /api/auth/login        # User login
POST /api/auth/register     # User registration
```

### Admin API
```
GET  /admin/bookings        # List bookings
POST /admin/bookings/{id}/status  # Update booking status
GET  /admin/services        # List services
POST /admin/services        # Create service
PUT  /admin/services/{id}   # Update service
DELETE /admin/services/{id} # Delete service
```

## 🧪 Testing

### Running Tests
```bash
# Run PHP unit tests
php tests/run.php

# Run booking conflict tests
php tests/booking-conflicts.php

# Run availability algorithm tests
php tests/availability-test.php
```

### Sample Data
The system includes sample data for testing:
- 1 Company (Luxury Beauty Spa)
- 2 Branches (Downtown & Uptown)
- Sample services across categories
- Staff members with schedules
- Customer accounts for testing

## 🚀 Deployment

### Production Checklist
- [ ] Environment variables configured
- [ ] SSL certificate installed
- [ ] Database optimized
- [ ] File permissions set correctly
- [ ] CRON jobs configured
- [ ] Backup system in place
- [ ] Monitoring tools configured

### Performance Optimization
- Enable opcode caching (OPcache)
- Use CDN for static assets
- Implement database query caching
- Enable gzip compression
- Optimize images automatically

### Backup Strategy
```bash
# Database backup
mysqldump salon_system > backup_$(date +%Y%m%d_%H%M%S).sql

# File backup
tar -czf uploads_backup_$(date +%Y%m%d_%H%M%S).tar.gz public/uploads/
```

## 🤝 Support

### Documentation
- [User Guide](docs/user-guide.md)
- [API Documentation](docs/api.md)
- [Customization Guide](docs/customization.md)

### Community
- [GitHub Issues](https://github.com/your-repo/issues)
- [Discussion Forum](https://forum.yourdomain.com)

### Professional Support
For enterprise support and custom development:
- Email: support@luxurybeautyspa.com
- Phone: +1 (234) 567-8900

## 📄 License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## 🙏 Acknowledgments

- Built with modern web technologies
- Inspired by leading salon management systems
- Community contributions and feedback

---

**Important Notes:**
- Never store production credentials in code
- Keep backups of your database regularly
- Monitor system performance and security
- Stay updated with the latest security patches
