# 🚀 Salon/Spa/Beauty Parlor Booking System - Deployment Guide

## 📋 Table of Contents
- [System Requirements](#system-requirements)
- [Quick Start (Local Development)](#quick-start-local-development)
- [Database Setup](#database-setup)
- [Environment Configuration](#environment-configuration)
- [Web Server Configuration](#web-server-configuration)
- [Asset Compilation](#asset-compilation)
- [SSL Certificate Setup](#ssl-certificate-setup)
- [Production Deployment](#production-deployment)
- [Scheduled Tasks (CRON Jobs)](#scheduled-tasks-cron-jobs)
- [Third-Party Integrations](#third-party-integrations)
- [Security Checklist](#security-checklist)
- [Monitoring & Maintenance](#monitoring--maintenance)
- [Troubleshooting](#troubleshooting)

---

## 🖥️ System Requirements

### Minimum Requirements
- **PHP**: 7.4+ or 8.0+ (8.1+ recommended)
- **Database**: MySQL 5.7+ or MariaDB 10.3+
- **Web Server**: Apache 2.4+ or Nginx 1.18+
- **RAM**: 512MB minimum, 1GB recommended
- **Storage**: 500MB for application + database

### Recommended Stack
- **PHP**: 8.1+ with extensions: `pdo`, `pdo_mysql`, `mbstring`, `openssl`, `gd`, `curl`, `zip`
- **Database**: MySQL 8.0+ or MariaDB 10.5+
- **Web Server**: Nginx 1.20+ with PHP-FPM
- **SSL**: Let's Encrypt (free) or commercial certificate
- **Domain**: Custom domain (subdomains for multi-tenant)

---

## 🏃‍♂️ Quick Start (Local Development)

### 1. Clone & Install Dependencies
```bash
# Clone the repository
git clone https://github.com/your-repo/salon-booking-system.git
cd salon-booking-system

# Install PHP dependencies (if using Composer)
composer install

# Install Node.js dependencies
npm install
```

### 2. Setup Environment
```bash
# Copy environment template
cp .env.example .env

# Generate application key
php -r "echo bin2hex(random_bytes(32)) . PHP_EOL;" > APP_KEY.txt
# Add the key to your .env file
```

### 3. Database Setup
```bash
# Create database
mysql -u root -p
CREATE DATABASE salon_system CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
exit

# Import schema
mysql -u root -p salon_system < schema.sql
```

### 4. Configure Web Server

#### Apache (.htaccess)
Create `.htaccess` in the `public/` directory:
```apache
RewriteEngine On
RewriteBase /

# Force HTTPS
RewriteCond %{HTTPS} off
RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]

# Handle PHP files
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^ index.php [QSA,L]

# Security headers
<FilesMatch "\.(php|html|js|css)$">
    Header set X-Frame-Options "SAMEORIGIN"
    Header set X-Content-Type-Options "nosniff"
    Header set X-XSS-Protection "1; mode=block"
</FilesMatch>

# Cache static assets
<FilesMatch "\.(css|js|png|jpg|jpeg|gif|ico|svg)$">
    ExpiresActive on
    ExpiresDefault "access plus 1 month"
</FilesMatch>
```

#### Nginx Configuration
```nginx
server {
    listen 80;
    server_name yourdomain.com www.yourdomain.com;
    root /path/to/salon-system/public;
    index index.php;

    # Security headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;

    # Handle PHP
    location ~ \.php$ {
        include fastcgi_params;
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
    }

    # Handle static files
    location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg)$ {
        expires 1M;
        add_header Cache-Control "public, immutable";
    }

    # Main rewrite rule
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    # Deny access to sensitive files
    location ~ /\. {
        deny all;
    }
}
```

### 5. Build Assets
```bash
# Build CSS and JS
npm run build

# Or for development with hot reload
npm run dev
```

### 6. Set Permissions
```bash
# Set proper permissions
chown -R www-data:www-data /path/to/salon-system
chmod -R 755 /path/to/salon-system
chmod -R 775 /path/to/salon-system/storage  # If you add storage directory
```

### 7. Access Application
Visit `http://localhost` or your configured domain.

---

## 🗄️ Database Setup

### Option 1: Using phpMyAdmin
1. Log into phpMyAdmin
2. Create database: `salon_system` (utf8mb4_unicode_ci)
3. Import `schema.sql`

### Option 2: Command Line
```bash
# Create database
mysql -u username -p -e "CREATE DATABASE salon_system CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"

# Import schema
mysql -u username -p salon_system < schema.sql
```

### Option 3: Using Adminer (Lightweight alternative)
```bash
# Download and serve Adminer
wget https://www.adminer.org/latest.php -O adminer.php
php -S localhost:8080 adminer.php
```

### Seed Data
The schema includes sample data for:
- 2 companies (Luxury Beauty Spa, Elite Hair Studio)
- Multiple branches
- Sample services and categories
- Staff accounts
- Customer accounts

**Default Admin Login:**
- Email: `admin@luxurybeauty.com`
- Password: `password123`

---

## ⚙️ Environment Configuration

### Required Environment Variables (.env)
```env
# Application
APP_NAME="Luxury Beauty Spa"
APP_URL="https://yourdomain.com"
APP_KEY="your-32-character-random-key"
APP_ENV="production"
APP_DEBUG=false

# Database
DB_HOST="localhost"
DB_PORT=3306
DB_NAME="salon_system"
DB_USER="your_db_user"
DB_PASS="your_secure_password"

# SMTP (Email)
SMTP_HOST="smtp.gmail.com"
SMTP_PORT=587
SMTP_USER="your-email@gmail.com"
SMTP_PASS="your-app-password"
SMTP_ENCRYPTION="tls"

# WhatsApp Business API
WHATSAPP_ACCESS_TOKEN="your-whatsapp-token"
WHATSAPP_PHONE_NUMBER_ID="your-phone-number-id"
WHATSAPP_BUSINESS_ACCOUNT_ID="your-business-account-id"

# M-Pesa Integration
MPESA_CONSUMER_KEY="your-consumer-key"
MPESA_CONSUMER_SECRET="your-consumer-secret"
MPESA_SHORTCODE="your-shortcode"
MPESA_PASSKEY="your-passkey"
MPESA_ENVIRONMENT="sandbox"  # Change to "production" for live

# File Uploads
MAX_FILE_SIZE="10MB"
ALLOWED_EXTENSIONS="jpg,jpeg,png,gif,pdf"

# Security
CSRF_TOKEN_NAME="salon_csrf_token"
SESSION_LIFETIME=7200  # 2 hours

# Timezone
APP_TIMEZONE="UTC"
```

### Environment File Security
```bash
# Ensure .env is not committed to git
echo ".env" >> .gitignore

# Set proper permissions
chmod 600 .env
```

---

## 🌐 Web Server Configuration

### Apache Virtual Host
```apache
<VirtualHost *:80>
    ServerName yourdomain.com
    ServerAlias www.yourdomain.com
    DocumentRoot /path/to/salon-system/public

    <Directory /path/to/salon-system/public>
        AllowOverride All
        Require all granted
    </Directory>

    ErrorLog ${APACHE_LOG_DIR}/salon_error.log
    CustomLog ${APACHE_LOG_DIR}/salon_access.log combined
</VirtualHost>
```

### Nginx with SSL
```nginx
server {
    listen 443 ssl http2;
    server_name yourdomain.com www.yourdomain.com;

    ssl_certificate /path/to/ssl/cert.pem;
    ssl_certificate_key /path/to/ssl/private.key;
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers ECDHE-RSA-AES128-GCM-SHA256:ECDHE-RSA-AES256-GCM-SHA384;

    root /path/to/salon-system/public;
    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        include fastcgi_params;
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
    }

    location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot)$ {
        expires 1y;
        add_header Cache-Control "public, immutable";
    }
}

# Redirect HTTP to HTTPS
server {
    listen 80;
    server_name yourdomain.com www.yourdomain.com;
    return 301 https://$server_name$request_uri;
}
```

---

## 🎨 Asset Compilation

### Development Mode
```bash
# Watch for changes and auto-recompile
npm run dev
```

### Production Build
```bash
# Build optimized assets
npm run build
```

### Tailwind Configuration
```javascript
// tailwind.config.js
module.exports = {
  content: [
    "./src/Views/**/*.php",
    "./public/**/*.html",
    "./assets/js/**/*.js"
  ],
  theme: {
    extend: {
      colors: {
        primary: 'var(--primary-color)',
        secondary: 'var(--secondary-color)',
      }
    }
  }
}
```

---

## 🔒 SSL Certificate Setup

### Let's Encrypt (Free)
```bash
# Install Certbot
sudo apt install certbot python3-certbot-nginx

# Get certificate
sudo certbot --nginx -d yourdomain.com -d www.yourdomain.com

# Auto-renewal
sudo crontab -e
# Add: 0 12 * * * /usr/bin/certbot renew --quiet
```

### Manual SSL Setup
```bash
# Generate self-signed certificate (for testing only)
openssl req -x509 -newkey rsa:4096 -keyout key.pem -out cert.pem -days 365 -nodes
```

---

## 🚀 Production Deployment

### 1. Server Preparation
```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install required packages
sudo apt install -y nginx mysql-server php8.1-fpm php8.1-mysql php8.1-mbstring php8.1-xml php8.1-curl php8.1-gd php8.1-zip unzip curl

# Install Node.js for asset compilation
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt install -y nodejs

# Install Composer
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer
```

### 2. Application Deployment
```bash
# Create application directory
sudo mkdir -p /var/www/salon-system
sudo chown -R $USER:www-data /var/www/salon-system

# Clone or upload application
cd /var/www/salon-system
# Upload your files here

# Install dependencies
composer install --optimize-autoloader --no-dev
npm install && npm run build
```

### 3. Database Setup
```bash
# Secure MySQL installation
sudo mysql_secure_installation

# Create database and user
sudo mysql -e "
CREATE DATABASE salon_system CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'salon_user'@'localhost' IDENTIFIED BY 'secure_password_here';
GRANT ALL PRIVILEGES ON salon_system.* TO 'salon_user'@'localhost';
FLUSH PRIVILEGES;
"

# Import schema
mysql -u salon_user -p salon_system < schema.sql
```

### 4. Environment Configuration
```bash
# Copy and configure environment
cp .env.example .env
# Edit .env with production values

# Generate secure app key
php -r "echo 'APP_KEY=' . bin2hex(random_bytes(32)) . PHP_EOL;" >> .env
```

### 5. Web Server Setup
```bash
# Copy Nginx configuration
sudo cp deployment/nginx.conf /etc/nginx/sites-available/salon-system
sudo ln -s /etc/nginx/sites-available/salon-system /etc/nginx/sites-enabled/

# Test configuration
sudo nginx -t

# Restart services
sudo systemctl restart nginx php8.1-fpm
```

### 6. SSL Setup
```bash
# Install SSL certificate
sudo certbot --nginx -d yourdomain.com -d www.yourdomain.com
```

### 7. File Permissions
```bash
# Set proper ownership
sudo chown -R www-data:www-data /var/www/salon-system
sudo chmod -R 755 /var/www/salon-system
sudo chmod 600 /var/www/salon-system/.env
```

---

## ⏰ Scheduled Tasks (CRON Jobs)

### Reminder System
```bash
# Edit crontab
crontab -e

# Add these lines for automated reminders
*/5 * * * * php /path/to/salon-system/scripts/reminders.php  # Every 5 minutes
0 9 * * * php /path/to/salon-system/scripts/daily-reports.php  # Daily reports at 9 AM
0 2 * * * php /path/to/salon-system/scripts/backup.php  # Daily backup at 2 AM
```

### Reminder Script Content
```php
<?php
// scripts/reminders.php
require_once __DIR__ . '/../config/config.php';

// Send reminders for upcoming bookings
$reminders = $db->query("
    SELECT b.*, c.name as company_name, s.name as service_name
    FROM bookings b
    JOIN companies c ON b.company_id = c.id
    JOIN services s ON b.service_id = s.id
    WHERE b.status = 'confirmed'
    AND b.start_at BETWEEN NOW() AND DATE_ADD(NOW(), INTERVAL 24 HOUR)
    AND b.reminder_sent = 0
")->resultSet();

foreach ($reminders as $booking) {
    // Send email reminder
    $emailAdapter = new EmailAdapter();
    $emailAdapter->sendReminder($booking);

    // Send WhatsApp reminder
    $whatsappAdapter = new WhatsAppAdapter();
    $whatsappAdapter->sendReminder($booking);

    // Mark as sent
    $db->query("UPDATE bookings SET reminder_sent = 1 WHERE id = ?")
        ->bind(1, $booking['id'])->execute();
}
```

---

## 🔗 Third-Party Integrations

### M-Pesa Integration
1. Register at [Safaricom Developer Portal](https://developer.safaricom.co.ke/)
2. Get Consumer Key & Secret
3. Configure in `.env`
4. Test with sandbox environment first

### WhatsApp Business API
1. Apply for WhatsApp Business API
2. Get access token and phone number ID
3. Configure webhook endpoints
4. Test message sending

### Email Service (SMTP)
```env
# Gmail SMTP
SMTP_HOST=smtp.gmail.com
SMTP_PORT=587
SMTP_USER=your-email@gmail.com
SMTP_PASS=your-app-password
SMTP_ENCRYPTION=tls

# SendGrid
SMTP_HOST=smtp.sendgrid.net
SMTP_PORT=587
SMTP_USER=apikey
SMTP_PASS=your-sendgrid-api-key
```

---

## 🔐 Security Checklist

### Pre-Deployment
- [ ] Change default database password
- [ ] Generate unique APP_KEY (32 characters)
- [ ] Configure proper file permissions (644 for files, 755 for directories)
- [ ] Disable directory listing
- [ ] Configure firewall (allow only 80, 443, 22)

### Application Security
- [ ] Enable CSRF protection
- [ ] Implement rate limiting
- [ ] Use prepared statements (already implemented)
- [ ] Validate and sanitize all inputs
- [ ] Implement proper session management
- [ ] Use HTTPS everywhere

### Server Security
```bash
# Install fail2ban for SSH protection
sudo apt install fail2ban

# Configure UFW firewall
sudo ufw allow ssh
sudo ufw allow 'Nginx Full'
sudo ufw enable

# Disable root login
sudo sed -i 's/#PermitRootLogin yes/PermitRootLogin no/' /etc/ssh/sshd_config
sudo systemctl restart ssh
```

---

## 📊 Monitoring & Maintenance

### Log Files
```bash
# Application logs
tail -f /var/log/nginx/error.log
tail -f /var/log/nginx/access.log

# PHP logs
tail -f /var/log/php8.1-fpm.log
```

### Database Maintenance
```bash
# Create backup script
mysqldump -u salon_user -p salon_system > backup_$(date +%Y%m%d_%H%M%S).sql

# Optimize tables weekly
mysql -u salon_user -p salon_system -e "OPTIMIZE TABLE bookings, users, services;"
```

### Performance Monitoring
```bash
# Check PHP-FPM status
sudo systemctl status php8.1-fpm

# Monitor resource usage
htop
df -h
free -h

# Database connections
mysql -u salon_user -p -e "SHOW PROCESSLIST;"
```

---

## 🔧 Troubleshooting

### Common Issues

#### 1. 500 Internal Server Error
```bash
# Check PHP error logs
tail -f /var/log/php8.1-fpm.log

# Check file permissions
ls -la /var/www/salon-system

# Check .env file
cat /var/www/salon-system/.env
```

#### 2. Database Connection Error
```bash
# Test database connection
mysql -u salon_user -p salon_system -e "SELECT 1;"

# Check database credentials in .env
grep "DB_" /var/www/salon-system/.env
```

#### 3. Assets Not Loading
```bash
# Clear browser cache
# Check if assets were built
ls -la /var/www/salon-system/public/assets/

# Rebuild assets
cd /var/www/salon-system
npm run build
```

#### 4. SSL Certificate Issues
```bash
# Check certificate validity
openssl x509 -in /etc/letsencrypt/live/yourdomain.com/cert.pem -text -noout

# Renew certificate
sudo certbot renew
```

#### 5. Permission Denied Errors
```bash
# Fix ownership
sudo chown -R www-data:www-data /var/www/salon-system

# Fix permissions
sudo find /var/www/salon-system -type f -exec chmod 644 {} \;
sudo find /var/www/salon-system -type d -exec chmod 755 {} \;
```

### Getting Help
1. Check application logs first
2. Verify server configuration
3. Test database connectivity
4. Check file permissions
5. Review environment variables
6. Consult [PHP documentation](https://php.net) and [MySQL documentation](https://dev.mysql.com/doc/)

---

## 📞 Support

For additional support:
- 📧 Email: support@salonsystem.com
- 📖 Documentation: [docs.salonsystem.com](https://docs.salonsystem.com)
- 🐛 Bug Reports: [GitHub Issues](https://github.com/your-repo/issues)

---

## 🎉 You're All Set!

Your Salon/Spa/Beauty Parlor Booking System is now ready for production use. Remember to:
- Keep backups regular
- Monitor system performance
- Update dependencies regularly
- Renew SSL certificates before expiration
- Stay updated with security patches

Happy booking! 💅✨
