<?php
/**
 * System Integration Test Script
 * Salon/Spa/Beauty Parlor Booking System
 *
 * This script tests the complete system integration
 */

// Include configuration
require_once __DIR__ . '/../config/config.php';

echo "=== Salon System Integration Test ===\n\n";

$testsPassed = 0;
$testsTotal = 0;

function test($name, $callback) {
    global $testsPassed, $testsTotal;
    $testsTotal++;

    echo "Testing: {$name}... ";

    try {
        $result = $callback();
        if ($result === true) {
            echo "✓ PASS\n";
            $testsPassed++;
        } else {
            echo "✗ FAIL: {$result}\n";
        }
    } catch (Exception $e) {
        echo "✗ ERROR: " . $e->getMessage() . "\n";
    }
}

try {
    // Initialize database
    $db = Database::getInstance();

    // Test 1: Database Connection
    test("Database Connection", function() use ($db) {
        $result = $db->query("SELECT 1 as test")->single();
        return $result['test'] === 1;
    });

    // Test 2: Company Model
    test("Company Model", function() {
        $companyModel = new Company();
        $companies = $companyModel->getActive();
        return !empty($companies);
    });

    // Test 3: Branch Model
    test("Branch Model", function() {
        $branchModel = new Branch();
        $branches = $branchModel->getByCompany(1, true);
        return !empty($branches);
    });

    // Test 4: Service Model
    test("Service Model", function() {
        $serviceModel = new Service();
        $services = $serviceModel->getByCompany(1, true);
        return !empty($services);
    });

    // Test 5: User Model
    test("User Model", function() {
        $userModel = new User();
        $users = $userModel->all('created_at DESC', 5);
        return !empty($users);
    });

    // Test 6: Booking Model
    test("Booking Model", function() {
        $bookingModel = new Booking();
        $bookings = $bookingModel->getByCompany(1, [], 5);
        return is_array($bookings);
    });

    // Test 7: Payment Model
    test("Payment Model", function() {
        $paymentModel = new Payment();
        $stats = $paymentModel->getStats(1);
        return is_array($stats);
    });

    // Test 8: Authentication
    test("Authentication System", function() {
        $auth = Auth::getInstance();
        return is_object($auth);
    });

    // Test 9: Router
    test("Router System", function() {
        $router = new Router();
        return is_object($router);
    });

    // Test 10: Email Adapter
    test("Email Adapter", function() {
        $emailAdapter = new EmailAdapter();
        return is_object($emailAdapter);
    });

    // Test 11: WhatsApp Adapter
    test("WhatsApp Adapter", function() {
        $whatsappAdapter = new WhatsAppAdapter();
        return is_object($whatsappAdapter);
    });

    // Test 12: SMS Adapter
    test("SMS Adapter", function() {
        $smsAdapter = new SMSAdapter();
        return is_object($smsAdapter);
    });

    // Test 13: File Upload Validation
    test("File Upload Validation", function() {
        return function_exists('isAllowedFileType') && function_exists('sanitize');
    });

    // Test 14: CSRF Protection
    test("CSRF Protection", function() {
        $token1 = generateCsrfToken();
        $token2 = generateCsrfToken();
        return !empty($token1) && !empty($token2);
    });

    // Test 15: Password Hashing
    test("Password Security", function() {
        $password = "testpassword123";
        $hash = password_hash($password, PASSWORD_DEFAULT);
        return password_verify($password, $hash);
    });

    // Test 16: Booking Conflict Prevention
    test("Booking Conflict Prevention", function() {
        $bookingModel = new Booking();

        // Get test data
        $service = $db->query("SELECT id FROM services WHERE is_active = 1 LIMIT 1")->single();
        $staff = $db->query("SELECT id FROM users WHERE role = 'staff' AND status = 'active' LIMIT 1")->single();
        $branch = $db->query("SELECT id FROM branches WHERE status = 'active' LIMIT 1")->single();

        if (!$service || !$staff || !$branch) {
            return "Missing test data";
        }

        $startTime = date('Y-m-d H:i:s', strtotime('+7 days 10:00:00'));
        $endTime = date('Y-m-d H:i:s', strtotime('+7 days 11:00:00'));

        $availability = $bookingModel->checkAvailability(
            $service['id'],
            $staff['id'],
            $startTime,
            $endTime
        );

        return is_array($availability) && isset($availability['available']);
    });

    // Test 17: API Response Format
    test("API Response Format", function() {
        // Simulate API response
        $response = [
            'success' => true,
            'data' => ['test' => 'data'],
            'meta' => ['total' => 1]
        ];

        return isset($response['success']) && isset($response['data']);
    });

    // Test 18: Template Rendering
    test("Template System", function() {
        return function_exists('view') && function_exists('sanitize');
    });

    // Test 19: Cache System (if enabled)
    test("Cache System", function() {
        return defined('CACHE_ENABLED');
    });

    // Test 20: Environment Configuration
    test("Environment Configuration", function() {
        return defined('DB_HOST') && defined('BASE_URL');
    });

} catch (Exception $e) {
    echo "CRITICAL ERROR: " . $e->getMessage() . "\n";
    exit(1);
}

// Test Results
echo "\n=== Test Results ===\n";
echo "Passed: {$testsPassed}/{$testsTotal} tests\n";
$percentage = round(($testsPassed / $testsTotal) * 100, 1);
echo "Success Rate: {$percentage}%\n\n";

if ($testsPassed === $testsTotal) {
    echo "🎉 ALL TESTS PASSED! The system is fully functional.\n";
    echo "You can now proceed with production deployment.\n";
} elseif ($percentage >= 90) {
    echo "✅ Most tests passed. The system is ready for production with minor fixes.\n";
} elseif ($percentage >= 75) {
    echo "⚠️  Good test results. Some features may need attention before production.\n";
} else {
    echo "❌ Several tests failed. Please review and fix issues before deployment.\n";
}

echo "\n=== System Information ===\n";
echo "PHP Version: " . PHP_VERSION . "\n";
echo "Database: " . (DB_NAME ?: 'Not configured') . "\n";
echo "Environment: " . ENVIRONMENT . "\n";
echo "Base URL: " . BASE_URL . "\n";
echo "Cache Enabled: " . (CACHE_ENABLED ? 'Yes' : 'No') . "\n";

echo "\n=== Next Steps ===\n";
echo "1. Configure environment variables in .env file\n";
echo "2. Set up web server (Apache/Nginx)\n";
echo "3. Configure SMTP for email notifications\n";
echo "4. Set up CRON jobs for reminders and sitemap\n";
echo "5. Test PWA installation on mobile devices\n";
echo "6. Monitor system performance and security\n";

echo "\n=== Salon System Integration Test Completed ===\n";
