<?php
/**
 * Booking Conflict Test Script
 * Salon/Spa/Beauty Parlor Booking System
 *
 * This script tests the booking conflict prevention algorithm
 */

// Include configuration
require_once __DIR__ . '/../config/config.php';

echo "=== Booking Conflict Prevention Test ===\n\n";

try {
    // Initialize database
    $db = Database::getInstance();

    // Create test data
    echo "Setting up test data...\n";

    // Get existing service and staff for testing
    $service = $db->query("SELECT id, name, duration_minutes FROM services WHERE is_active = 1 LIMIT 1")->single();
    $staff = $db->query("SELECT id, first_name, last_name FROM users WHERE role = 'staff' AND status = 'active' LIMIT 1")->single();
    $branch = $db->query("SELECT id, name FROM branches WHERE status = 'active' LIMIT 1")->single();
    $company = $db->query("SELECT id, name FROM companies WHERE status = 'active' LIMIT 1")->single();

    if (!$service || !$staff || !$branch || !$company) {
        echo "ERROR: Missing test data. Please run the SQL seed data first.\n";
        exit(1);
    }

    echo "Test Service: {$service['name']} ({$service['duration_minutes']} min)\n";
    echo "Test Staff: {$staff['first_name']} {$staff['last_name']}\n";
    echo "Test Branch: {$branch['name']}\n\n";

    // Test 1: Create initial booking
    echo "Test 1: Creating initial booking...\n";
    $bookingModel = new Booking();

    $startTime = date('Y-m-d H:i:s', strtotime('+1 day 10:00:00')); // Tomorrow at 10 AM

    $bookingData = [
        'service_id' => $service['id'],
        'branch_id' => $branch['id'],
        'staff_id' => $staff['id'],
        'start_at' => $startTime,
        'status' => 'confirmed',
        'notes' => 'Test booking for conflict prevention'
    ];

    $bookingId1 = $bookingModel->createBooking($bookingData);
    echo "✓ Created booking #{$bookingId1} at {$startTime}\n\n";

    // Test 2: Try to create conflicting booking (same staff, overlapping time)
    echo "Test 2: Testing conflict prevention (same staff, overlapping time)...\n";

    $conflictingStart = date('Y-m-d H:i:s', strtotime('+1 day 10:15:00')); // 15 min later (should conflict)

    $conflictData = [
        'service_id' => $service['id'],
        'branch_id' => $branch['id'],
        'staff_id' => $staff['id'],
        'start_at' => $conflictingStart,
        'status' => 'confirmed'
    ];

    try {
        $bookingId2 = $bookingModel->createBooking($conflictData);
        echo "✗ ERROR: Conflicting booking was created! This should not happen.\n";
    } catch (Exception $e) {
        echo "✓ Conflict correctly prevented: " . $e->getMessage() . "\n";
    }

    // Test 3: Create non-conflicting booking (different time)
    echo "\nTest 3: Testing non-conflicting booking (different time)...\n";

    $nonConflictingStart = date('Y-m-d H:i:s', strtotime('+1 day 14:00:00')); // Same day, different time

    $nonConflictData = [
        'service_id' => $service['id'],
        'branch_id' => $branch['id'],
        'staff_id' => $staff['id'],
        'start_at' => $nonConflictingStart,
        'status' => 'confirmed'
    ];

    try {
        $bookingId3 = $bookingModel->createBooking($nonConflictData);
        echo "✓ Non-conflicting booking created successfully #{$bookingId3} at {$nonConflictingStart}\n";
    } catch (Exception $e) {
        echo "✗ ERROR: Non-conflicting booking failed: " . $e->getMessage() . "\n";
    }

    // Test 4: Test availability check
    echo "\nTest 4: Testing availability check method...\n";

    $availability = $bookingModel->checkAvailability(
        $service['id'],
        $staff['id'],
        date('Y-m-d H:i:s', strtotime('+1 day 16:00:00')), // Should be available
        date('Y-m-d H:i:s', strtotime('+1 day 17:00:00'))
    );

    if ($availability['available']) {
        echo "✓ Availability check: Time slot is available\n";
    } else {
        echo "✗ Availability check: Time slot not available - {$availability['reason']}\n";
    }

    // Test 5: Test buffer time conflict
    echo "\nTest 5: Testing buffer time conflict...\n";

    $bufferConflictStart = date('Y-m-d H:i:s', strtotime('+1 day 10:50:00')); // Should conflict with buffer

    $bufferData = [
        'service_id' => $service['id'],
        'branch_id' => $branch['id'],
        'staff_id' => $staff['id'],
        'start_at' => $bufferConflictStart,
        'status' => 'confirmed'
    ];

    try {
        $bookingId4 = $bookingModel->createBooking($bufferData);
        echo "✗ ERROR: Buffer time conflict was allowed! This should not happen.\n";
    } catch (Exception $e) {
        echo "✓ Buffer time conflict correctly prevented: " . $e->getMessage() . "\n";
    }

    // Cleanup: Delete test bookings
    echo "\nCleaning up test data...\n";
    $db->delete('bookings', ['id' => $bookingId1]);
    if (isset($bookingId3)) {
        $db->delete('bookings', ['id' => $bookingId3]);
    }
    echo "✓ Test bookings deleted\n";

} catch (Exception $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\n=== Booking Conflict Prevention Test Completed ===\n";
echo "All conflict prevention mechanisms are working correctly!\n";
