<?php
/**
 * Router Class
 * Salon/Spa/Beauty Parlor Booking System
 */

class Router {
    private $routes = [];
    private $middlewares = [];

    public function __construct() {
        // Set default routes
        $this->setDefaultRoutes();
    }

    /**
     * Add a route
     */
    public function add($method, $path, $handler, $middleware = []) {
        $this->routes[] = [
            'method' => strtoupper($method),
            'path' => $this->normalizePath($path),
            'handler' => $handler,
            'middleware' => $middleware,
            'params' => []
        ];
    }

    /**
     * Add GET route
     */
    public function get($path, $handler, $middleware = []) {
        $this->add('GET', $path, $handler, $middleware);
    }

    /**
     * Add POST route
     */
    public function post($path, $handler, $middleware = []) {
        $this->add('POST', $path, $handler, $middleware);
    }

    /**
     * Add PUT route
     */
    public function put($path, $handler, $middleware = []) {
        $this->add('PUT', $path, $handler, $middleware);
    }

    /**
     * Add DELETE route
     */
    public function delete($path, $handler, $middleware = []) {
        $this->add('DELETE', $path, $handler, $middleware);
    }

    /**
     * Add middleware
     */
    public function middleware($middleware) {
        $this->middlewares[] = $middleware;
        return $this;
    }

    /**
     * Group routes with prefix and middleware
     */
    public function group($prefix, $callback, $middleware = []) {
        $previousMiddlewares = $this->middlewares;
        $this->middlewares = array_merge($this->middlewares, $middleware);

        $callback($this);

        $this->middlewares = $previousMiddlewares;
    }

    /**
     * Dispatch the request
     */
    public function dispatch() {
        $method = getRequestMethod();
        $uri = $this->normalizePath(getRequestUri());

        foreach ($this->routes as $route) {
            if ($route['method'] === $method || $route['method'] === 'ANY') {
                $params = $this->matchRoute($route['path'], $uri);

                if ($params !== false) {
                    // Run global middlewares
                    foreach ($this->middlewares as $middleware) {
                        $this->runMiddleware($middleware);
                    }

                    // Run route middlewares
                    foreach ($route['middleware'] as $middleware) {
                        $this->runMiddleware($middleware);
                    }

                    // Set route parameters
                    $_GET = array_merge($_GET, $params);

                    // Call handler
                    return $this->callHandler($route['handler'], $params);
                }
            }
        }

        // No route found
        $this->handle404();
    }

    /**
     * Match route pattern with URI
     */
    private function matchRoute($pattern, $uri) {
        // Convert route parameters to regex
        $pattern = preg_replace('/\{(\w+)\}/', '(?P<$1>[^/]+)', $pattern);
        $pattern = "#^{$pattern}$#";

        if (preg_match($pattern, $uri, $matches)) {
            $params = [];
            foreach ($matches as $key => $value) {
                if (!is_numeric($key)) {
                    $params[$key] = $value;
                }
            }
            return $params;
        }

        return false;
    }

    /**
     * Normalize path
     */
    private function normalizePath($path) {
        return '/' . trim($path, '/');
    }

    /**
     * Run middleware
     */
    private function runMiddleware($middleware) {
        if (is_callable($middleware)) {
            call_user_func($middleware);
        } elseif (is_string($middleware)) {
            // Assume it's a class method like 'AuthMiddleware@handle'
            list($class, $method) = explode('@', $middleware);
            $instance = new $class();
            call_user_func([$instance, $method]);
        }
    }

    /**
     * Call route handler
     */
    private function callHandler($handler, $params = []) {
        if (is_callable($handler)) {
            return call_user_func_array($handler, $params);
        } elseif (is_string($handler)) {
            // Assume it's a controller method like 'HomeController@index'
            list($controller, $method) = explode('@', $handler);
            $controllerInstance = new $controller();
            return call_user_func_array([$controllerInstance, $method], $params);
        }
    }

    /**
     * Handle 404
     */
    private function handle404() {
        http_response_code(404);
        if (file_exists(PUBLIC_PATH . '/404.php')) {
            include PUBLIC_PATH . '/404.php';
        } else {
            echo '<h1>404 - Page Not Found</h1>';
        }
        exit;
    }

    /**
     * Set default routes
     */
    private function setDefaultRoutes() {
        // Public routes
        $this->get('/', 'HomeController@index');
        $this->get('/services', 'ServiceController@index');
        $this->get('/services/{slug}', 'ServiceController@show');
        $this->get('/book', 'BookingController@create');
        $this->post('/book', 'BookingController@store');
        $this->get('/login', 'AuthController@login');
        $this->post('/login', 'AuthController@doLogin');
        $this->get('/register', 'AuthController@register');
        $this->post('/register', 'AuthController@doRegister');
        $this->post('/logout', 'AuthController@logout');

        // API routes
        $this->group('/api', function($router) {
            $router->get('/services', 'Api\ServiceController@index');
            $router->get('/services/{slug}', 'Api\ServiceController@show');
            $router->get('/branches', 'Api\BranchController@index');
            $router->get('/availability', 'Api\AvailabilityController@check');
            $router->post('/bookings', 'Api\BookingController@store');
            $router->post('/auth/login', 'Api\AuthController@login');
            $router->post('/auth/register', 'Api\AuthController@register');
        });

        // Admin routes
        $this->group('/admin', function($router) {
            $router->get('/', 'Admin\DashboardController@index');
            $router->get('/dashboard', 'Admin\DashboardController@index');
            $router->get('/bookings', 'Admin\BookingController@index');
            $router->get('/services', 'Admin\ServiceController@index');
            $router->get('/staff', 'Admin\StaffController@index');
            $router->get('/customers', 'Admin\CustomerController@index');
            $router->get('/reports', 'Admin\ReportController@index');
            $router->get('/settings', 'Admin\SettingController@index');
        }, ['AuthMiddleware@admin']);

        // Staff routes
        $this->group('/staff', function($router) {
            $router->get('/', 'Staff\DashboardController@index');
            $router->get('/dashboard', 'Staff\DashboardController@index');
            $router->get('/schedule', 'Staff\ScheduleController@index');
            $router->get('/bookings', 'Staff\BookingController@index');
        }, ['AuthMiddleware@staff']);

        // Shop routes
        $this->group('/shop', function($router) {
            $router->get('/', 'Shop\CatalogController@index');
            $router->get('/products/{slug}', 'Shop\CatalogController@show');
            $router->get('/cart', 'Shop\CartController@index');
            $router->post('/cart/add', 'Shop\CartController@add');
            $router->get('/checkout', 'Shop\CheckoutController@index');
            $router->post('/checkout', 'Shop\CheckoutController@process');
        });
    }
}
