<?php
/**
 * Service Model
 * Salon/Spa/Beauty Parlor Booking System
 */

class Service extends Model {
    protected $table = 'services';

    /**
     * Get services by category
     */
    public function getByCategory($categoryId, $companyId = null, $activeOnly = true) {
        $where = ["category_id = ?"];
        $params = [$categoryId];

        if ($companyId) {
            $where[] = "company_id = ?";
            $params[] = $companyId;
        }

        if ($activeOnly) {
            $where[] = "is_active = 1";
        }

        $whereClause = implode(' AND ', $where);

        return $this->db->query("SELECT * FROM {$this->table} WHERE {$whereClause} ORDER BY name ASC", $params)->resultSet();
    }

    /**
     * Get featured services
     */
    public function getFeatured($companyId, $limit = 6) {
        return $this->db->query("SELECT * FROM {$this->table}
                                WHERE company_id = ? AND is_featured = 1 AND is_active = 1
                                ORDER BY created_at DESC LIMIT ?")
                       ->bind(1, $companyId)->bind(2, $limit)->resultSet();
    }

    /**
     * Get services with categories
     */
    public function getWithCategories($companyId, $activeOnly = true) {
        $where = ["s.company_id = ?"];
        $params = [$companyId];

        if ($activeOnly) {
            $where[] = "s.is_active = 1";
        }

        $whereClause = implode(' AND ', $where);

        return $this->db->query("SELECT s.*, sc.name as category_name, sc.slug as category_slug
                                FROM {$this->table} s
                                LEFT JOIN service_categories sc ON s.category_id = sc.id
                                WHERE {$whereClause}
                                ORDER BY s.name ASC", $params)->resultSet();
    }

    /**
     * Search services
     */
    public function search($query, $companyId, $activeOnly = true) {
        $where = ["s.company_id = ?", "MATCH(s.name, s.description, s.short_description) AGAINST(? IN NATURAL LANGUAGE MODE)"];
        $params = [$companyId, $query];

        if ($activeOnly) {
            $where[] = "s.is_active = 1";
        }

        $whereClause = implode(' AND ', $where);

        return $this->db->query("SELECT s.*, sc.name as category_name,
            MATCH(s.name, s.description, s.short_description) AGAINST(? IN NATURAL LANGUAGE MODE) as relevance
            FROM {$this->table} s
            LEFT JOIN service_categories sc ON s.category_id = sc.id
            WHERE {$whereClause}
            ORDER BY relevance DESC, s.name ASC", $params)->resultSet();
    }

    /**
     * Get service by slug
     */
    public function getBySlug($slug, $companyId) {
        return $this->db->query("SELECT s.*, sc.name as category_name, sc.slug as category_slug
                                FROM {$this->table} s
                                LEFT JOIN service_categories sc ON s.category_id = sc.id
                                WHERE s.slug = ? AND s.company_id = ? AND s.is_active = 1")
                       ->bind(1, $slug)->bind(2, $companyId)->single();
    }

    /**
     * Get service statistics
     */
    public function getStats($serviceId) {
        return $this->db->query("SELECT
            COUNT(*) as total_bookings,
            COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_bookings,
            COUNT(CASE WHEN status = 'cancelled' THEN 1 END) as cancelled_bookings,
            COALESCE(SUM(CASE WHEN status = 'completed' THEN total_amount END), 0) as total_revenue,
            COALESCE(AVG(CASE WHEN status = 'completed' THEN total_amount END), 0) as avg_booking_value
            FROM bookings WHERE service_id = ?")
                       ->bind(1, $serviceId)->single();
    }

    /**
     * Check if service can be deleted
     */
    public function canDelete($serviceId) {
        $bookingCount = $this->db->query("SELECT COUNT(*) as count FROM bookings WHERE service_id = ?")
                                ->bind(1, $serviceId)->single()['count'];
        return $bookingCount === 0;
    }

    /**
     * Get available services for booking
     */
    public function getAvailableForBooking($companyId, $branchId = null) {
        $where = ["s.company_id = ?", "s.is_active = 1", "s.requires_staff = 1"];
        $params = [$companyId];

        if ($branchId) {
            $where[] = "EXISTS (SELECT 1 FROM staff_services ss JOIN users u ON ss.user_id = u.id
                              WHERE ss.service_id = s.id AND u.branch_id = ? AND u.status = 'active')";
            $params[] = $branchId;
        }

        $whereClause = implode(' AND ', $where);

        return $this->db->query("SELECT s.*, sc.name as category_name
                                FROM {$this->table} s
                                LEFT JOIN service_categories sc ON s.category_id = sc.id
                                WHERE {$whereClause}
                                ORDER BY s.name ASC", $params)->resultSet();
    }

    /**
     * Update service status
     */
    public function updateStatus($serviceId, $status) {
        return $this->db->update($this->table, [
            'is_active' => $status,
            'updated_at' => date('Y-m-d H:i:s')
        ], ['id' => $serviceId]);
    }
}