<?php
/**
 * Product Category Model
 * Salon/Spa/Beauty Parlor Booking System
 */

class ProductCategory extends Model {
    protected $table = 'product_categories';

    /**
     * Get categories by company
     */
    public function getByCompany($companyId) {
        return $this->db->query("SELECT * FROM {$this->table}
                                WHERE company_id = ? OR company_id IS NULL
                                ORDER BY name ASC")
                       ->bind(1, $companyId)->resultSet();
    }

    /**
     * Get category with product count
     */
    public function getWithProductCount($companyId) {
        return $this->db->query("SELECT pc.*,
            COUNT(p.id) as product_count,
            COUNT(CASE WHEN p.is_active = 1 THEN 1 END) as active_product_count
            FROM {$this->table} pc
            LEFT JOIN products p ON pc.id = p.category_id
            WHERE pc.company_id = ? OR pc.company_id IS NULL
            GROUP BY pc.id
            ORDER BY pc.name ASC")
                       ->bind(1, $companyId)->resultSet();
    }

    /**
     * Get category by slug
     */
    public function getBySlug($slug, $companyId = null) {
        $query = "SELECT * FROM {$this->table} WHERE slug = ?";
        $params = [$slug];

        if ($companyId) {
            $query .= " AND (company_id = ? OR company_id IS NULL)";
            $params[] = $companyId;
        }

        return $this->db->query($query, $params)->single();
    }

    /**
     * Create category with slug
     */
    public function create($data) {
        // Generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = $this->createSlug($data['name']);
        }

        return $this->db->insert($this->table, $data);
    }

    /**
     * Update category with slug
     */
    public function update($id, $data) {
        // Generate slug if name changed and slug not provided
        if (isset($data['name']) && !isset($data['slug'])) {
            $data['slug'] = $this->createSlug($data['name'], $id);
        }

        return $this->db->update($this->table, $data, ['id' => $id]);
    }

    /**
     * Delete category (with products reassignment)
     */
    public function deleteCategory($id, $reassignTo = null) {
        // Start transaction
        $this->db->beginTransaction();

        try {
            if ($reassignTo) {
                // Reassign products to another category
                $this->db->query("UPDATE products SET category_id = ? WHERE category_id = ?")
                        ->bind(1, $reassignTo)->bind(2, $id)->execute();
            } else {
                // Set products to no category
                $this->db->query("UPDATE products SET category_id = NULL WHERE category_id = ?")
                        ->bind(1, $id)->execute();
            }

            // Delete category
            $this->db->delete($this->table, ['id' => $id]);

            $this->db->commit();
            return true;

        } catch (Exception $e) {
            $this->db->rollback();
            return false;
        }
    }

    /**
     * Check if category has products
     */
    public function hasProducts($categoryId) {
        $count = $this->db->query("SELECT COUNT(*) as count FROM products WHERE category_id = ?")
                         ->bind(1, $categoryId)->single()['count'];
        return $count > 0;
    }

    /**
     * Get category hierarchy (for future nested categories)
     */
    public function getHierarchy($companyId) {
        // For now, return flat hierarchy
        return $this->getByCompany($companyId);
    }

    /**
     * Create URL slug from name
     */
    private function createSlug($name, $excludeId = null) {
        $slug = strtolower(trim($name));
        $slug = preg_replace('/[^a-z0-9-]/', '-', $slug);
        $slug = preg_replace('/-+/', '-', $slug);
        $slug = trim($slug, '-');

        // Ensure uniqueness
        $originalSlug = $slug;
        $counter = 1;

        while ($this->slugExists($slug, $excludeId)) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    /**
     * Check if slug exists
     */
    private function slugExists($slug, $excludeId = null) {
        $query = "SELECT COUNT(*) as count FROM {$this->table} WHERE slug = ?";
        $params = [$slug];

        if ($excludeId) {
            $query .= " AND id != ?";
            $params[] = $excludeId;
        }

        return $this->db->query($query, $params)->single()['count'] > 0;
    }
}
