<?php
/**
 * Payment Model
 * Salon/Spa/Beauty Parlor Booking System
 */

class Payment extends Model {
    protected $table = 'payments';
    protected $fillable = ['booking_id', 'order_id', 'amount', 'payment_method_id', 'transaction_id', 'status', 'payment_date', 'notes', 'processed_by'];

    /**
     * Get payment with related data
     */
    public function findWithDetails($id) {
        $sql = "
            SELECT p.*, pm.name as payment_method_name, pm.type as payment_method_type,
                   b.booking_number, b.total_amount as booking_total,
                   u.first_name, u.last_name, u.email,
                   o.order_number, o.total_amount as order_total
            FROM {$this->table} p
            LEFT JOIN payment_methods pm ON p.payment_method_id = pm.id
            LEFT JOIN bookings b ON p.booking_id = b.id
            LEFT JOIN orders o ON p.order_id = o.id
            LEFT JOIN users u ON p.processed_by = u.id
            WHERE p.id = ?
        ";
        return $this->db->query($sql)->bind(1, $id)->single();
    }

    /**
     * Get payments by booking
     */
    public function getByBooking($bookingId) {
        return $this->findBy(['booking_id' => $bookingId], false);
    }

    /**
     * Get payments by order
     */
    public function getByOrder($orderId) {
        return $this->findBy(['order_id' => $orderId], false);
    }

    /**
     * Get payments by company
     */
    public function getByCompany($companyId, $filters = []) {
        $sql = "
            SELECT p.*, pm.name as payment_method_name, pm.type as payment_method_type,
                   b.booking_number, o.order_number, u.first_name, u.last_name
            FROM {$this->table} p
            LEFT JOIN payment_methods pm ON p.payment_method_id = pm.id
            LEFT JOIN bookings b ON p.booking_id = b.id
            LEFT JOIN orders o ON p.order_id = o.id
            LEFT JOIN users u ON p.processed_by = u.id
            WHERE (b.company_id = ? OR o.company_id = ?)
        ";
        $params = [$companyId, $companyId];

        // Apply filters
        if (!empty($filters['status'])) {
            $sql .= " AND p.status = ?";
            $params[] = $filters['status'];
        }

        if (!empty($filters['payment_method'])) {
            $sql .= " AND pm.type = ?";
            $params[] = $filters['payment_method'];
        }

        if (!empty($filters['date_from'])) {
            $sql .= " AND DATE(p.payment_date) >= ?";
            $params[] = $filters['date_from'];
        }

        if (!empty($filters['date_to'])) {
            $sql .= " AND DATE(p.payment_date) <= ?";
            $params[] = $filters['date_to'];
        }

        $sql .= " ORDER BY p.payment_date DESC";

        $query = $this->db->query($sql);
        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->resultSet();
    }

    /**
     * Create payment for booking
     */
    public function createForBooking($bookingId, $data) {
        // Verify booking exists and belongs to company
        $booking = $this->db->query("SELECT id, company_id, total_amount FROM bookings WHERE id = ?")
                           ->bind(1, $bookingId)
                           ->single();

        if (!$booking) {
            throw new Exception('Booking not found');
        }

        $data['booking_id'] = $bookingId;
        $data['amount'] = $booking['total_amount'];

        return $this->create($data);
    }

    /**
     * Create payment for order
     */
    public function createForOrder($orderId, $data) {
        // Verify order exists
        $order = $this->db->query("SELECT id, company_id, total_amount FROM orders WHERE id = ?")
                         ->bind(1, $orderId)
                         ->single();

        if (!$order) {
            throw new Exception('Order not found');
        }

        $data['order_id'] = $orderId;
        $data['amount'] = $order['total_amount'];

        return $this->create($data);
    }

    /**
     * Update payment status
     */
    public function updateStatus($paymentId, $status, $processedBy = null) {
        $data = ['status' => $status];

        if ($status === 'completed' && !$this->find($paymentId)['payment_date']) {
            $data['payment_date'] = date('Y-m-d H:i:s');
        }

        if ($processedBy) {
            $data['processed_by'] = $processedBy;
        }

        return $this->update($paymentId, $data);
    }

    /**
     * Get payment statistics
     */
    public function getStats($companyId, $dateFrom = null, $dateTo = null) {
        $sql = "
            SELECT
                COUNT(*) as total_payments,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_payments,
                COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_payments,
                COUNT(CASE WHEN status = 'failed' THEN 1 END) as failed_payments,
                SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END) as total_amount,
                AVG(CASE WHEN status = 'completed' THEN amount ELSE NULL END) as avg_payment_amount
            FROM {$this->table} p
            LEFT JOIN bookings b ON p.booking_id = b.id
            LEFT JOIN orders o ON p.order_id = o.id
            WHERE (b.company_id = ? OR o.company_id = ?)
        ";
        $params = [$companyId, $companyId];

        if ($dateFrom) {
            $sql .= " AND DATE(p.created_at) >= ?";
            $params[] = $dateFrom;
        }

        if ($dateTo) {
            $sql .= " AND DATE(p.created_at) <= ?";
            $params[] = $dateTo;
        }

        $query = $this->db->query($sql);
        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->single();
    }

    /**
     * Process M-Pesa payment (stub for integration)
     */
    public function processMpesaPayment($phoneNumber, $amount, $accountReference, $transactionDesc = '') {
        // This is a stub for M-Pesa integration
        // In production, this would make API calls to M-Pesa

        try {
            // Generate transaction ID
            $transactionId = 'MPESA' . date('YmdHis') . rand(1000, 9999);

            // Simulate API call delay
            usleep(2000000); // 2 seconds

            // Simulate success (90% success rate)
            $success = (rand(1, 10) <= 9);

            if ($success) {
                return [
                    'success' => true,
                    'transaction_id' => $transactionId,
                    'status' => 'completed',
                    'message' => 'Payment processed successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'transaction_id' => $transactionId,
                    'status' => 'failed',
                    'message' => 'Payment processing failed'
                ];
            }

        } catch (Exception $e) {
            return [
                'success' => false,
                'status' => 'failed',
                'message' => 'Payment processing error: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Verify M-Pesa transaction (stub)
     */
    public function verifyMpesaTransaction($transactionId) {
        // Stub for transaction verification
        // In production, this would query M-Pesa API

        return [
            'verified' => true,
            'amount' => 0,
            'status' => 'completed'
        ];
    }
}
