<?php
/**
 * Branch Model
 * Salon/Spa/Beauty Parlor Booking System
 */

class Branch extends Model {
    protected $table = 'branches';
    protected $fillable = ['company_id', 'name', 'slug', 'address', 'city', 'state', 'country', 'postal_code', 'phone', 'email', 'latitude', 'longitude', 'opening_hours', 'timezone', 'is_main', 'status'];

    /**
     * Get branch with company info
     */
    public function findWithCompany($id) {
        $sql = "
            SELECT b.*, c.name as company_name, c.slug as company_slug,
                   c.primary_color, c.secondary_color, c.timezone as company_timezone
            FROM {$this->table} b
            LEFT JOIN companies c ON b.company_id = c.id
            WHERE b.id = ?
        ";
        return $this->db->query($sql)->bind(1, $id)->single();
    }

    /**
     * Get branches by company
     */
    public function getByCompany($companyId, $activeOnly = true) {
        $sql = "SELECT * FROM {$this->table} WHERE company_id = ?";
        $params = [$companyId];

        if ($activeOnly) {
            $sql .= " AND status = 'active'";
        }

        $sql .= " ORDER BY is_main DESC, name ASC";

        return $this->db->query($sql)->bind(1, $companyId)->resultSet();
    }

    /**
     * Get branch services
     */
    public function getServices($branchId, $activeOnly = true) {
        $sql = "SELECT * FROM services WHERE branch_id = ?";
        $params = [$branchId];

        if ($activeOnly) {
            $sql .= " AND is_active = 1";
        }

        $sql .= " ORDER BY sort_order ASC, name ASC";

        return $this->db->query($sql)->bind(1, $branchId)->resultSet();
    }

    /**
     * Get branch staff
     */
    public function getStaff($branchId, $activeOnly = true) {
        $sql = "SELECT * FROM users WHERE branch_id = ?";
        $params = [$branchId];

        if ($activeOnly) {
            $sql .= " AND status = 'active' AND role = 'staff'";
        }

        $sql .= " ORDER BY first_name, last_name";

        return $this->db->query($sql)->bind(1, $branchId)->resultSet();
    }

    /**
     * Get branch availability for a date
     */
    public function getAvailability($branchId, $date) {
        // Get branch opening hours
        $branch = $this->find($branchId);
        if (!$branch || !$branch['opening_hours']) {
            return null;
        }

        $openingHours = json_decode($branch['opening_hours'], true);
        $dayOfWeek = date('l', strtotime($date)); // Monday, Tuesday, etc.

        if (!isset($openingHours[strtolower($dayOfWeek)])) {
            return null;
        }

        $hours = $openingHours[strtolower($dayOfWeek)];
        return [
            'branch' => $branch,
            'date' => $date,
            'day_of_week' => $dayOfWeek,
            'open_time' => $hours['open'] ?? null,
            'close_time' => $hours['close'] ?? null,
            'is_open' => isset($hours['open']) && isset($hours['close'])
        ];
    }

    /**
     * Check if branch is open at given time
     */
    public function isOpenAt($branchId, $datetime) {
        $date = date('Y-m-d', strtotime($datetime));
        $time = date('H:i:s', strtotime($datetime));

        $availability = $this->getAvailability($branchId, $date);

        if (!$availability || !$availability['is_open']) {
            return false;
        }

        return ($time >= $availability['open_time'] && $time <= $availability['close_time']);
    }

    /**
     * Get branch statistics
     */
    public function getStats($branchId, $dateFrom = null, $dateTo = null) {
        $sql = "
            SELECT
                COUNT(DISTINCT b.id) as total_bookings,
                COUNT(DISTINCT CASE WHEN b.status = 'completed' THEN b.id END) as completed_bookings,
                COUNT(DISTINCT CASE WHEN b.status = 'cancelled' THEN b.id END) as cancelled_bookings,
                COUNT(DISTINCT s.id) as total_services,
                COUNT(DISTINCT st.id) as total_staff,
                SUM(CASE WHEN b.status = 'completed' THEN b.total_amount ELSE 0 END) as total_revenue,
                AVG(CASE WHEN b.status = 'completed' THEN b.total_amount ELSE NULL END) as avg_booking_value
            FROM branches br
            LEFT JOIN bookings b ON br.id = b.branch_id
            LEFT JOIN services s ON br.id = s.branch_id AND s.is_active = 1
            LEFT JOIN users st ON br.id = st.branch_id AND st.role = 'staff' AND st.status = 'active'
            WHERE br.id = ?
        ";
        $params = [$branchId];

        if ($dateFrom) {
            $sql .= " AND DATE(b.created_at) >= ?";
            $params[] = $dateFrom;
        }

        if ($dateTo) {
            $sql .= " AND DATE(b.created_at) <= ?";
            $params[] = $dateTo;
        }

        $query = $this->db->query($sql);
        foreach ($params as $index => $param) {
            $query->bind($index + 1, $param);
        }

        return $query->single();
    }

    /**
     * Check if slug exists for company
     */
    public function slugExists($slug, $companyId, $excludeId = null) {
        $sql = "SELECT COUNT(*) as count FROM {$this->table} WHERE slug = ? AND company_id = ?";
        $params = [$slug, $companyId];

        if ($excludeId) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
        }

        $result = $this->db->query($sql)->single();
        return $result['count'] > 0;
    }

    /**
     * Update branch opening hours
     */
    public function updateOpeningHours($branchId, $openingHours) {
        return $this->update($branchId, ['opening_hours' => json_encode($openingHours)]);
    }
}
