<?php
/**
 * Database Connection Class
 * Salon/Spa/Beauty Parlor Booking System
 */

class Database {
    private static $instance = null;
    private $pdo;
    private $stmt;

    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
            ];

            $this->pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            if (IS_DEVELOPMENT) {
                die("Database connection failed: " . $e->getMessage());
            } else {
                error_log("Database connection failed: " . $e->getMessage());
                die("Database connection error. Please try again later.");
            }
        }
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function query($sql) {
        $this->stmt = $this->pdo->prepare($sql);
        return $this;
    }

    public function bind($param, $value, $type = null) {
        if ($type === null) {
            switch (true) {
                case is_int($value):
                    $type = PDO::PARAM_INT;
                    break;
                case is_bool($value):
                    $type = PDO::PARAM_BOOL;
                    break;
                case is_null($value):
                    $type = PDO::PARAM_NULL;
                    break;
                default:
                    $type = PDO::PARAM_STR;
            }
        }
        $this->stmt->bindValue($param, $value, $type);
        return $this;
    }

    public function execute() {
        return $this->stmt->execute();
    }

    public function resultSet() {
        $this->execute();
        return $this->stmt->fetchAll();
    }

    public function single() {
        $this->execute();
        return $this->stmt->fetch();
    }

    public function rowCount() {
        return $this->stmt->rowCount();
    }

    public function lastInsertId() {
        return $this->pdo->lastInsertId();
    }

    public function beginTransaction() {
        return $this->pdo->beginTransaction();
    }

    public function commit() {
        return $this->pdo->commit();
    }

    public function rollback() {
        return $this->pdo->rollback();
    }

    public function debugDumpParams() {
        return $this->stmt->debugDumpParams();
    }

    // Helper methods for common queries
    public function insert($table, $data) {
        $columns = implode(', ', array_keys($data));
        $placeholders = ':' . implode(', :', array_keys($data));

        $sql = "INSERT INTO {$table} ({$columns}) VALUES ({$placeholders})";
        $this->query($sql);

        foreach ($data as $key => $value) {
            $this->bind(":{$key}", $value);
        }

        return $this->execute();
    }

    public function update($table, $data, $where) {
        $setParts = [];
        foreach ($data as $key => $value) {
            $setParts[] = "{$key} = :{$key}";
        }
        $setClause = implode(', ', $setParts);

        $whereParts = [];
        foreach ($where as $key => $value) {
            $whereParts[] = "{$key} = :where_{$key}";
        }
        $whereClause = implode(' AND ', $whereParts);

        $sql = "UPDATE {$table} SET {$setClause} WHERE {$whereClause}";
        $this->query($sql);

        foreach ($data as $key => $value) {
            $this->bind(":{$key}", $value);
        }

        foreach ($where as $key => $value) {
            $this->bind(":where_{$key}", $value);
        }

        return $this->execute();
    }

    public function delete($table, $where) {
        $whereParts = [];
        foreach ($where as $key => $value) {
            $whereParts[] = "{$key} = :{$key}";
        }
        $whereClause = implode(' AND ', $whereParts);

        $sql = "DELETE FROM {$table} WHERE {$whereClause}";
        $this->query($sql);

        foreach ($where as $key => $value) {
            $this->bind(":{$key}", $value);
        }

        return $this->execute();
    }

    // Table existence check
    public function tableExists($table) {
        try {
            $result = $this->query("SHOW TABLES LIKE ?")
                          ->bind(1, $table)
                          ->single();
            return !empty($result);
        } catch (Exception $e) {
            return false;
        }
    }

    // Get table columns
    public function getTableColumns($table) {
        return $this->query("DESCRIBE {$table}")->resultSet();
    }
}
