<?php
/**
 * Staff Booking Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class BookingController extends Controller {
    public function index() {
        $staffId = $this->auth->id();

        // Get filter parameters
        $status = $this->getQuery('status', 'all');
        $date = $this->getQuery('date');
        $search = $this->getQuery('search');
        $page = (int) $this->getQuery('page', 1);

        // Build where clause
        $where = ["b.staff_id = ?"];
        $params = [$staffId];

        if ($status !== 'all') {
            $where[] = "b.status = ?";
            $params[] = $status;
        }

        if ($date) {
            $where[] = "DATE(b.start_at) = ?";
            $params[] = $date;
        }

        if ($search) {
            $where[] = "(u.first_name LIKE ? OR u.last_name LIKE ? OR u.email LIKE ? OR b.booking_number LIKE ?)";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
        }

        $whereClause = implode(' AND ', $where);

        // Get bookings with pagination
        $offset = ($page - 1) * ADMIN_ITEMS_PER_PAGE;

        $bookings = $this->db->query("
            SELECT b.*, s.name as service_name, s.duration_minutes,
                   u.first_name, u.last_name, u.email, u.phone,
                   br.name as branch_name
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN branches br ON b.branch_id = br.id
            WHERE {$whereClause}
            ORDER BY b.start_at DESC
            LIMIT ? OFFSET ?
        ", array_merge($params, [ADMIN_ITEMS_PER_PAGE, $offset]))->resultSet();

        // Get total count
        $totalBookings = $this->db->query("
            SELECT COUNT(*) as total FROM bookings b
            LEFT JOIN users u ON b.customer_id = u.id
            WHERE {$whereClause}
        ", $params)->single()['total'];

        $this->staffView('bookings/index', [
            'bookings' => $bookings,
            'filters' => [
                'status' => $status,
                'date' => $date,
                'search' => $search,
                'page' => $page
            ],
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalBookings / ADMIN_ITEMS_PER_PAGE),
                'total_items' => $totalBookings
            ],
            'pageTitle' => 'My Bookings'
        ]);
    }

    public function show($id) {
        $staffId = $this->auth->id();

        $booking = $this->db->query("
            SELECT b.*, s.name as service_name, s.duration_minutes, s.price,
                   u.first_name, u.last_name, u.email, u.phone,
                   br.name as branch_name, br.address as branch_address,
                   st.first_name as staff_first_name, st.last_name as staff_last_name
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN branches br ON b.branch_id = br.id
            LEFT JOIN users st ON b.staff_id = st.id
            WHERE b.id = ? AND b.staff_id = ?
        ")->bind(1, $id)->bind(2, $staffId)->single();

        if (!$booking) {
            $this->staffView('errors/404');
            return;
        }

        // Get booking notes
        $notes = $this->getBookingNotes($id);

        $this->staffView('bookings/show', [
            'booking' => $booking,
            'notes' => $notes,
            'pageTitle' => 'Booking Details - ' . $booking['booking_number']
        ]);
    }

    public function updateStatus($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['error' => 'Method not allowed'], 405);
        }

        $staffId = $this->auth->id();
        $newStatus = $this->getPost('status');
        $notes = $this->getPost('notes');

        // Validate status
        $allowedStatuses = ['confirmed', 'completed', 'cancelled', 'no_show'];
        if (!in_array($newStatus, $allowedStatuses)) {
            $this->json(['error' => 'Invalid status'], 400);
        }

        $booking = $this->db->query("SELECT * FROM bookings WHERE id = ? AND staff_id = ?")
                           ->bind(1, $id)->bind(2, $staffId)->single();

        if (!$booking) {
            $this->json(['error' => 'Booking not found'], 404);
        }

        try {
            $bookingModel = new Booking();
            $bookingModel->updateStatus($id, $newStatus, $staffId);

            // Add status change note
            if ($notes) {
                $this->addBookingNote($id, "Status changed to: {$newStatus}. Notes: {$notes}", $staffId);
            }

            // Send notifications if needed
            $this->sendStatusChangeNotifications($id, $newStatus);

            $this->json([
                'success' => true,
                'message' => 'Booking status updated successfully'
            ]);

        } catch (Exception $e) {
            error_log("Booking status update error: " . $e->getMessage());
            $this->json(['error' => 'Failed to update booking status'], 500);
        }
    }

    public function addNote($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['error' => 'Method not allowed'], 405);
        }

        $staffId = $this->auth->id();
        $note = $this->getPost('note');

        if (empty($note)) {
            $this->json(['error' => 'Note cannot be empty'], 400);
        }

        // Verify booking belongs to staff
        $booking = $this->db->query("SELECT id FROM bookings WHERE id = ? AND staff_id = ?")
                           ->bind(1, $id)->bind(2, $staffId)->single();

        if (!$booking) {
            $this->json(['error' => 'Booking not found'], 404);
        }

        try {
            $this->addBookingNote($id, $note, $staffId);

            $this->json([
                'success' => true,
                'message' => 'Note added successfully'
            ]);

        } catch (Exception $e) {
            error_log("Add note error: " . $e->getMessage());
            $this->json(['error' => 'Failed to add note'], 500);
        }
    }

    public function today() {
        $staffId = $this->auth->id();
        $today = date('Y-m-d');

        $bookings = $this->db->query("
            SELECT b.*, s.name as service_name, s.duration_minutes,
                   u.first_name, u.last_name, u.phone,
                   br.name as branch_name
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN branches br ON b.branch_id = br.id
            WHERE b.staff_id = ? AND DATE(b.start_at) = ? AND b.status IN ('confirmed', 'pending')
            ORDER BY b.start_at ASC
        ")->bind(1, $staffId)->bind(2, $today)->resultSet();

        $this->staffView('bookings/today', [
            'bookings' => $bookings,
            'date' => $today,
            'pageTitle' => 'Today\'s Appointments'
        ]);
    }

    public function markCompleted($id) {
        $staffId = $this->auth->id();

        $booking = $this->db->query("SELECT * FROM bookings WHERE id = ? AND staff_id = ?")
                           ->bind(1, $id)->bind(2, $staffId)->single();

        if (!$booking) {
            $this->json(['error' => 'Booking not found'], 404);
        }

        if ($booking['status'] !== 'confirmed') {
            $this->json(['error' => 'Only confirmed bookings can be marked as completed'], 400);
        }

        try {
            $bookingModel = new Booking();
            $bookingModel->updateStatus($id, 'completed', $staffId);

            // Add completion note
            $this->addBookingNote($id, 'Service completed by staff', $staffId);

            $this->json([
                'success' => true,
                'message' => 'Booking marked as completed'
            ]);

        } catch (Exception $e) {
            error_log("Mark completed error: " . $e->getMessage());
            $this->json(['error' => 'Failed to mark booking as completed'], 500);
        }
    }

    public function performance() {
        $staffId = $this->auth->id();

        // Get performance stats
        $stats = $this->db->query("
            SELECT
                COUNT(*) as total_bookings,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_bookings,
                COUNT(CASE WHEN status = 'no_show' THEN 1 END) as no_shows,
                COUNT(CASE WHEN status = 'cancelled' THEN 1 END) as cancellations,
                COALESCE(SUM(CASE WHEN status = 'completed' THEN total_amount END), 0) as total_revenue,
                COALESCE(AVG(CASE WHEN status = 'completed' THEN total_amount END), 0) as avg_booking_value,
                COUNT(DISTINCT DATE(start_at)) as days_worked
            FROM bookings
            WHERE staff_id = ? AND start_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        ")->bind(1, $staffId)->single();

        // Calculate rates
        $completionRate = $stats['total_bookings'] > 0 ? ($stats['completed_bookings'] / $stats['total_bookings']) * 100 : 0;
        $noShowRate = $stats['total_bookings'] > 0 ? ($stats['no_shows'] / $stats['total_bookings']) * 100 : 0;

        // Get monthly performance
        $monthlyPerformance = $this->db->query("
            SELECT
                DATE_FORMAT(start_at, '%Y-%m') as month,
                COUNT(*) as bookings,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed,
                COALESCE(SUM(CASE WHEN status = 'completed' THEN total_amount END), 0) as revenue
            FROM bookings
            WHERE staff_id = ? AND start_at >= DATE_SUB(NOW(), INTERVAL 6 MONTH)
            GROUP BY DATE_FORMAT(start_at, '%Y-%m')
            ORDER BY month DESC
        ")->bind(1, $staffId)->resultSet();

        $this->staffView('bookings/performance', [
            'stats' => $stats,
            'completionRate' => round($completionRate, 1),
            'noShowRate' => round($noShowRate, 1),
            'monthlyPerformance' => $monthlyPerformance,
            'pageTitle' => 'My Performance'
        ]);
    }

    /**
     * Get booking notes
     */
    private function getBookingNotes($bookingId) {
        // For now, return empty array - would implement notes system
        return [];
    }

    /**
     * Add booking note
     */
    private function addBookingNote($bookingId, $note, $userId) {
        // For now, just log - would implement notes system
        error_log("Staff booking note added: {$note}");
    }

    /**
     * Send status change notifications
     */
    private function sendStatusChangeNotifications($bookingId, $status) {
        // This would integrate with notification system
        error_log("Staff status change notifications sent for booking ID: {$bookingId}, status: {$status}");
    }
}
