<?php
/**
 * Shop Checkout Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class CheckoutController extends Controller {
    public function index() {
        // Require login for checkout
        if (!$this->auth->check()) {
            $_SESSION['redirect_after_login'] = BASE_URL . '/shop/checkout';
            $this->redirect(BASE_URL . '/login');
        }

        $cart = $this->getCart();
        $cartItems = $this->getCartItems($cart);

        if (empty($cartItems)) {
            $this->redirect(BASE_URL . '/shop/cart');
        }

        // Calculate totals
        $subtotal = 0;
        foreach ($cartItems as $item) {
            $subtotal += $item['total_price'];
        }

        $taxRate = 0.08; // 8% tax
        $tax = $subtotal * $taxRate;
        $shipping = $subtotal > 50 ? 0 : 5.99;
        $total = $subtotal + $tax + $shipping;

        // Get user addresses (placeholder for now)
        $addresses = [];

        // Get payment methods
        $paymentMethods = $this->db->query("
            SELECT * FROM payment_methods
            WHERE company_id = ? AND is_active = 1
            ORDER BY name ASC
        ")->bind(1, $this->auth->companyId() ?: 1)->resultSet();

        $this->view('shop/checkout/index', [
            'cartItems' => $cartItems,
            'addresses' => $addresses,
            'paymentMethods' => $paymentMethods,
            'subtotal' => $subtotal,
            'tax' => $tax,
            'shipping' => $shipping,
            'total' => $total,
            'pageTitle' => 'Checkout'
        ]);
    }

    public function process() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/shop/checkout');
        }

        if (!$this->auth->check()) {
            $this->redirect(BASE_URL . '/login');
        }

        $cart = $this->getCart();
        $cartItems = $this->getCartItems($cart);

        if (empty($cartItems)) {
            $this->redirect(BASE_URL . '/shop/cart');
        }

        // Get form data
        $paymentMethodId = $this->getPost('payment_method_id');
        $shippingAddress = [
            'first_name' => $this->getPost('shipping_first_name'),
            'last_name' => $this->getPost('shipping_last_name'),
            'email' => $this->getPost('shipping_email'),
            'phone' => $this->getPost('shipping_phone'),
            'address' => $this->getPost('shipping_address'),
            'city' => $this->getPost('shipping_city'),
            'state' => $this->getPost('shipping_state'),
            'postal_code' => $this->getPost('shipping_postal_code'),
            'country' => $this->getPost('shipping_country')
        ];

        $billingAddress = $this->getPost('same_address') ?
            $shippingAddress :
            [
                'first_name' => $this->getPost('billing_first_name'),
                'last_name' => $this->getPost('billing_last_name'),
                'email' => $this->getPost('billing_email'),
                'phone' => $this->getPost('billing_phone'),
                'address' => $this->getPost('billing_address'),
                'city' => $this->getPost('billing_city'),
                'state' => $this->getPost('billing_state'),
                'postal_code' => $this->getPost('billing_postal_code'),
                'country' => $this->getPost('billing_country')
            ];

        $notes = $this->getPost('notes');

        // Validate input
        $errors = $this->validateCheckoutData($shippingAddress, $billingAddress, $paymentMethodId);
        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/shop/checkout');
        }

        try {
            // Calculate totals
            $subtotal = 0;
            foreach ($cartItems as $item) {
                $subtotal += $item['total_price'];
            }

            $taxRate = 0.08;
            $tax = $subtotal * $taxRate;
            $shipping = $subtotal > 50 ? 0 : 5.99;
            $total = $subtotal + $tax + $shipping;

            // Create order
            $orderData = [
                'order_number' => $this->generateOrderNumber(),
                'company_id' => $this->auth->companyId() ?: 1,
                'customer_id' => $this->auth->id(),
                'status' => 'pending',
                'subtotal' => $subtotal,
                'tax_amount' => $tax,
                'shipping_amount' => $shipping,
                'total_amount' => $total,
                'shipping_address' => json_encode($shippingAddress),
                'billing_address' => json_encode($billingAddress),
                'notes' => $notes
            ];

            $orderId = $this->db->insert('orders', $orderData);

            // Add order items
            foreach ($cartItems as $item) {
                $this->db->insert('order_items', [
                    'order_id' => $orderId,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'price' => $item['price'],
                    'total' => $item['total_price']
                ]);

                // Update product stock
                if ($item['stock_quantity'] !== null) {
                    $newStock = $item['stock_quantity'] - $item['quantity'];
                    $this->db->update('products', ['stock_quantity' => $newStock], ['id' => $item['product_id']]);
                }
            }

            // Process payment
            $paymentResult = $this->processPayment($orderId, $total, $paymentMethodId);

            if ($paymentResult['success']) {
                // Update order status
                $this->db->update('orders', ['status' => 'processing'], ['id' => $orderId]);

                // Clear cart
                $this->db->query("DELETE FROM cart_items WHERE cart_id = ?")->bind(1, $cart['id'])->execute();

                // Send confirmation email
                $this->sendOrderConfirmation($orderId);

                $this->redirect(BASE_URL . '/shop/order/success/' . $orderId);
            } else {
                // Payment failed - cancel order
                $this->db->update('orders', ['status' => 'cancelled'], ['id' => $orderId]);

                $_SESSION['form_errors'] = ['payment' => 'Payment failed: ' . $paymentResult['message']];
                $this->redirect(BASE_URL . '/shop/checkout');
            }

        } catch (Exception $e) {
            error_log("Checkout error: " . $e->getMessage());

            $_SESSION['form_errors'] = ['general' => 'Checkout failed. Please try again.'];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/shop/checkout');
        }
    }

    public function success($orderId) {
        $order = $this->db->query("
            SELECT o.*, u.first_name, u.last_name, u.email
            FROM orders o
            LEFT JOIN users u ON o.customer_id = u.id
            WHERE o.id = ? AND o.customer_id = ?
        ")->bind(1, $orderId)->bind(2, $this->auth->id())->single();

        if (!$order) {
            $this->view('errors/404');
            return;
        }

        $this->view('shop/checkout/success', [
            'order' => $order,
            'pageTitle' => 'Order Confirmed'
        ]);
    }

    public function orderDetails($orderId) {
        $order = $this->db->query("
            SELECT o.*, u.first_name, u.last_name, u.email
            FROM orders o
            LEFT JOIN users u ON o.customer_id = u.id
            WHERE o.id = ? AND o.customer_id = ?
        ")->bind(1, $orderId)->bind(2, $this->auth->id())->single();

        if (!$order) {
            $this->view('errors/404');
            return;
        }

        // Get order items
        $orderItems = $this->db->query("
            SELECT oi.*, p.name, p.images
            FROM order_items oi
            JOIN products p ON oi.product_id = p.id
            WHERE oi.order_id = ?
        ")->bind(1, $orderId)->resultSet();

        $this->view('shop/checkout/order-details', [
            'order' => $order,
            'orderItems' => $orderItems,
            'pageTitle' => 'Order Details - ' . $order['order_number']
        ]);
    }

    /**
     * Validate checkout data
     */
    private function validateCheckoutData($shippingAddress, $billingAddress, $paymentMethodId) {
        $errors = [];

        // Validate shipping address
        if (empty($shippingAddress['first_name'])) $errors['shipping_first_name'] = 'First name is required';
        if (empty($shippingAddress['last_name'])) $errors['shipping_last_name'] = 'Last name is required';
        if (empty($shippingAddress['email']) || !filter_var($shippingAddress['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['shipping_email'] = 'Valid email is required';
        }
        if (empty($shippingAddress['phone'])) $errors['shipping_phone'] = 'Phone number is required';
        if (empty($shippingAddress['address'])) $errors['shipping_address'] = 'Address is required';
        if (empty($shippingAddress['city'])) $errors['shipping_city'] = 'City is required';
        if (empty($shippingAddress['country'])) $errors['shipping_country'] = 'Country is required';

        // Validate billing address (if different)
        if (!$this->getPost('same_address')) {
            if (empty($billingAddress['first_name'])) $errors['billing_first_name'] = 'First name is required';
            if (empty($billingAddress['last_name'])) $errors['billing_last_name'] = 'Last name is required';
            if (empty($billingAddress['email']) || !filter_var($billingAddress['email'], FILTER_VALIDATE_EMAIL)) {
                $errors['billing_email'] = 'Valid email is required';
            }
            if (empty($billingAddress['phone'])) $errors['billing_phone'] = 'Phone number is required';
            if (empty($billingAddress['address'])) $errors['billing_address'] = 'Address is required';
            if (empty($billingAddress['city'])) $errors['billing_city'] = 'City is required';
            if (empty($billingAddress['country'])) $errors['billing_country'] = 'Country is required';
        }

        // Validate payment method
        if (empty($paymentMethodId)) {
            $errors['payment_method_id'] = 'Payment method is required';
        } else {
            $paymentMethod = $this->db->query("SELECT id FROM payment_methods WHERE id = ? AND is_active = 1")
                                     ->bind(1, $paymentMethodId)->single();
            if (!$paymentMethod) {
                $errors['payment_method_id'] = 'Invalid payment method';
            }
        }

        return $errors;
    }

    /**
     * Process payment
     */
    private function processPayment($orderId, $amount, $paymentMethodId) {
        $paymentMethod = $this->db->query("SELECT * FROM payment_methods WHERE id = ?")
                                 ->bind(1, $paymentMethodId)->single();

        if (!$paymentMethod) {
            return ['success' => false, 'message' => 'Invalid payment method'];
        }

        // For now, simulate payment processing
        // In production, integrate with actual payment gateways

        $paymentResult = false;
        $transactionId = null;

        switch ($paymentMethod['type']) {
            case 'mpesa':
                $paymentResult = $this->processMpesaPayment($amount, $orderId);
                break;
            case 'card':
                $paymentResult = $this->processCardPayment($amount, $orderId);
                break;
            case 'cash':
            case 'bank_transfer':
                // Manual payment methods - mark as pending
                $paymentResult = true;
                $transactionId = 'manual_' . time();
                break;
            default:
                return ['success' => false, 'message' => 'Unsupported payment method'];
        }

        // Create payment record
        $this->db->insert('payments', [
            'order_id' => $orderId,
            'amount' => $amount,
            'payment_method_id' => $paymentMethodId,
            'transaction_id' => $transactionId,
            'status' => $paymentResult ? 'completed' : 'failed',
            'processed_by' => $this->auth->id()
        ]);

        return [
            'success' => $paymentResult,
            'message' => $paymentResult ? 'Payment processed successfully' : 'Payment processing failed',
            'transaction_id' => $transactionId
        ];
    }

    /**
     * Process M-Pesa payment (stub)
     */
    private function processMpesaPayment($amount, $orderId) {
        // Integrate with M-Pesa API
        // For now, simulate success
        return (rand(1, 10) <= 9); // 90% success rate
    }

    /**
     * Process card payment (stub)
     */
    private function processCardPayment($amount, $orderId) {
        // Integrate with payment gateway (Stripe, PayPal, etc.)
        // For now, simulate success
        return (rand(1, 10) <= 9); // 90% success rate
    }

    /**
     * Generate order number
     */
    private function generateOrderNumber() {
        return 'ORD-' . date('Y') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
    }

    /**
     * Get cart
     */
    private function getCart() {
        $sessionId = session_id();
        $userId = $this->auth->check() ? $this->auth->id() : null;

        $cart = null;

        if ($userId) {
            $cart = $this->db->query("SELECT * FROM carts WHERE user_id = ? AND status = 'active'")
                            ->bind(1, $userId)->single();
        }

        if (!$cart) {
            $cart = $this->db->query("SELECT * FROM carts WHERE session_id = ? AND status = 'active'")
                            ->bind(1, $sessionId)->single();
        }

        return $cart;
    }

    /**
     * Get cart items
     */
    private function getCartItems($cart) {
        return $this->db->query("
            SELECT ci.*, p.name, p.slug, p.images, p.stock_quantity,
                   (ci.price * ci.quantity) as total_price
            FROM cart_items ci
            JOIN products p ON ci.product_id = p.id
            WHERE ci.cart_id = ?
        ")->bind(1, $cart['id'])->resultSet();
    }

    /**
     * Send order confirmation
     */
    private function sendOrderConfirmation($orderId) {
        // This would integrate with email system
        error_log("Order confirmation sent for order ID: {$orderId}");
    }
}
