<?php
/**
 * Shop Catalog Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class CatalogController extends Controller {
    public function index() {
        $companyId = $this->getQuery('company_id', 1); // Default to first company

        // Get filter parameters
        $category = $this->getQuery('category');
        $search = $this->getQuery('search');
        $sort = $this->getQuery('sort', 'name');
        $minPrice = $this->getQuery('min_price');
        $maxPrice = $this->getQuery('max_price');
        $page = (int) $this->getQuery('page', 1);

        // Get product categories
        $categories = $this->db->query("
            SELECT id, name, slug FROM product_categories
            WHERE company_id = ? OR company_id IS NULL
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        // Build where clause for products
        $where = ["p.company_id = ?", "p.is_active = 1"];
        $params = [$companyId];

        if ($category) {
            $where[] = "p.category_id = ?";
            $params[] = $category;
        }

        if ($search) {
            $where[] = "(p.name LIKE ? OR p.description LIKE ? OR p.short_description LIKE ?)";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
        }

        if ($minPrice) {
            $where[] = "p.price >= ?";
            $params[] = $minPrice;
        }

        if ($maxPrice) {
            $where[] = "p.price <= ?";
            $params[] = $maxPrice;
        }

        $whereClause = implode(' AND ', $where);

        // Build order clause
        $orderClause = "p.name ASC";
        switch ($sort) {
            case 'name':
                $orderClause = "p.name ASC";
                break;
            case 'price_low':
                $orderClause = "p.price ASC";
                break;
            case 'price_high':
                $orderClause = "p.price DESC";
                break;
            case 'newest':
                $orderClause = "p.created_at DESC";
                break;
            case 'popular':
                $orderClause = "p.created_at DESC"; // Would need sales data
                break;
        }

        // Get products with pagination
        $offset = ($page - 1) * ITEMS_PER_PAGE;

        $products = $this->db->query("
            SELECT p.*, pc.name as category_name
            FROM products p
            LEFT JOIN product_categories pc ON p.category_id = pc.id
            WHERE {$whereClause}
            ORDER BY {$orderClause}
            LIMIT ? OFFSET ?
        ", array_merge($params, [ITEMS_PER_PAGE, $offset]))->resultSet();

        // Get total count
        $totalProducts = $this->db->query("
            SELECT COUNT(*) as total FROM products p WHERE {$whereClause}
        ", $params)->single()['total'];

        // Get featured products
        $featuredProducts = $this->db->query("
            SELECT p.*, pc.name as category_name
            FROM products p
            LEFT JOIN product_categories pc ON p.category_id = pc.id
            WHERE p.company_id = ? AND p.is_active = 1 AND p.is_featured = 1
            ORDER BY p.created_at DESC
            LIMIT 4
        ")->bind(1, $companyId)->resultSet();

        $this->view('shop/catalog/index', [
            'products' => $products,
            'featuredProducts' => $featuredProducts,
            'categories' => $categories,
            'filters' => [
                'category' => $category,
                'search' => $search,
                'sort' => $sort,
                'min_price' => $minPrice,
                'max_price' => $maxPrice,
                'page' => $page
            ],
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalProducts / ITEMS_PER_PAGE),
                'total_items' => $totalProducts
            ],
            'pageTitle' => 'Shop'
        ]);
    }

    public function show($slug) {
        $product = $this->db->query("
            SELECT p.*, pc.name as category_name, pc.slug as category_slug
            FROM products p
            LEFT JOIN product_categories pc ON p.category_id = pc.id
            WHERE p.slug = ? AND p.is_active = 1
        ")->bind(1, $slug)->single();

        if (!$product) {
            $this->view('errors/404');
            return;
        }

        // Get related products
        $relatedProducts = $this->getRelatedProducts($product, 4);

        // Get product images
        $images = $product['images'] ? json_decode($product['images'], true) : [];

        $this->view('shop/catalog/show', [
            'product' => $product,
            'images' => $images,
            'relatedProducts' => $relatedProducts,
            'pageTitle' => $product['name'],
            'metaDescription' => $product['short_description'] ?: substr($product['description'], 0, 155)
        ]);
    }

    public function category($categorySlug) {
        $category = $this->db->query("SELECT * FROM product_categories WHERE slug = ?")
                           ->bind(1, $categorySlug)->single();

        if (!$category) {
            $this->view('errors/404');
            return;
        }

        $companyId = $category['company_id'] ?: 1;

        // Get products in this category
        $page = (int) $this->getQuery('page', 1);
        $offset = ($page - 1) * ITEMS_PER_PAGE;

        $products = $this->db->query("
            SELECT p.*, pc.name as category_name
            FROM products p
            LEFT JOIN product_categories pc ON p.category_id = pc.id
            WHERE p.category_id = ? AND p.is_active = 1
            ORDER BY p.name ASC
            LIMIT ? OFFSET ?
        ")->bind(1, $category['id'])->bind(2, ITEMS_PER_PAGE)->bind(3, $offset)->resultSet();

        // Get total count
        $totalProducts = $this->db->query("
            SELECT COUNT(*) as total FROM products WHERE category_id = ? AND is_active = 1
        ")->bind(1, $category['id'])->single()['total'];

        $this->view('shop/catalog/category', [
            'category' => $category,
            'products' => $products,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalProducts / ITEMS_PER_PAGE),
                'total_items' => $totalProducts
            ],
            'pageTitle' => $category['name']
        ]);
    }

    public function search() {
        $query = $this->getQuery('q');
        $companyId = $this->getQuery('company_id', 1);

        if (empty($query)) {
            $this->redirect(BASE_URL . '/shop');
        }

        // Search products
        $products = $this->db->query("
            SELECT p.*, pc.name as category_name,
                   MATCH(p.name, p.description, p.short_description) AGAINST(? IN NATURAL LANGUAGE MODE) as relevance
            FROM products p
            LEFT JOIN product_categories pc ON p.category_id = pc.id
            WHERE p.company_id = ? AND p.is_active = 1
            AND MATCH(p.name, p.description, p.short_description) AGAINST(? IN NATURAL LANGUAGE MODE)
            ORDER BY relevance DESC, p.name ASC
            LIMIT 50
        ")->bind(1, $query)->bind(2, $companyId)->bind(3, $query)->resultSet();

        $this->view('shop/catalog/search', [
            'query' => $query,
            'products' => $products,
            'totalResults' => count($products),
            'pageTitle' => 'Search Results - ' . htmlspecialchars($query)
        ]);
    }

    /**
     * Get related products
     */
    private function getRelatedProducts($product, $limit = 4) {
        // Get products from same category
        if ($product['category_id']) {
            $related = $this->db->query("
                SELECT p.*, pc.name as category_name
                FROM products p
                LEFT JOIN product_categories pc ON p.category_id = pc.id
                WHERE p.category_id = ? AND p.id != ? AND p.is_active = 1
                ORDER BY p.created_at DESC
                LIMIT ?
            ")->bind(1, $product['category_id'])->bind(2, $product['id'])->bind(3, $limit)->resultSet();
        } else {
            // Get other products from same company
            $related = $this->db->query("
                SELECT p.*, pc.name as category_name
                FROM products p
                LEFT JOIN product_categories pc ON p.category_id = pc.id
                WHERE p.company_id = ? AND p.id != ? AND p.is_active = 1
                ORDER BY p.created_at DESC
                LIMIT ?
            ")->bind(1, $product['company_id'])->bind(2, $product['id'])->bind(3, $limit)->resultSet();
        }

        return $related;
    }
}
