<?php
/**
 * Shop Cart Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class CartController extends Controller {
    public function index() {
        $cart = $this->getCart();
        $cartItems = $this->getCartItems($cart);

        // Calculate totals
        $subtotal = 0;
        $tax = 0;
        $shipping = 0;

        foreach ($cartItems as $item) {
            $subtotal += $item['total_price'];
        }

        // Apply tax (configurable)
        $taxRate = 0.08; // 8% tax
        $tax = $subtotal * $taxRate;

        // Shipping (simple flat rate for now)
        $shipping = $subtotal > 50 ? 0 : 5.99;

        $total = $subtotal + $tax + $shipping;

        $this->view('shop/cart/index', [
            'cartItems' => $cartItems,
            'cartCount' => count($cartItems),
            'subtotal' => $subtotal,
            'tax' => $tax,
            'shipping' => $shipping,
            'total' => $total,
            'pageTitle' => 'Shopping Cart'
        ]);
    }

    public function add() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['error' => 'Method not allowed'], 405);
        }

        $productId = $this->getPost('product_id');
        $quantity = (int) $this->getPost('quantity', 1);
        $variantId = $this->getPost('variant_id');

        if (!$productId || $quantity < 1) {
            $this->json(['error' => 'Invalid product or quantity'], 400);
        }

        // Validate product exists and is active
        $product = $this->db->query("SELECT * FROM products WHERE id = ? AND is_active = 1")
                           ->bind(1, $productId)->single();

        if (!$product) {
            $this->json(['error' => 'Product not found'], 404);
        }

        // Check stock
        if ($product['stock_quantity'] !== null && $product['stock_quantity'] < $quantity) {
            $this->json(['error' => 'Insufficient stock'], 400);
        }

        // Get or create cart
        $cart = $this->getCart();

        // Check if item already in cart
        $existingItem = $this->db->query("
            SELECT id, quantity FROM cart_items
            WHERE cart_id = ? AND product_id = ?
        ")->bind(1, $cart['id'])->bind(2, $productId)->single();

        if ($existingItem) {
            // Update quantity
            $newQuantity = $existingItem['quantity'] + $quantity;
            $this->db->update('cart_items', ['quantity' => $newQuantity], ['id' => $existingItem['id']]);
        } else {
            // Add new item
            $this->db->insert('cart_items', [
                'cart_id' => $cart['id'],
                'product_id' => $productId,
                'quantity' => $quantity,
                'variant_id' => $variantId,
                'price' => $product['sale_price'] ?: $product['price']
            ]);
        }

        // Update cart total
        $this->updateCartTotal($cart['id']);

        $cartCount = $this->getCartCount();

        $this->json([
            'success' => true,
            'message' => 'Product added to cart',
            'cart_count' => $cartCount
        ]);
    }

    public function update() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['error' => 'Method not allowed'], 405);
        }

        $itemId = $this->getPost('item_id');
        $quantity = (int) $this->getPost('quantity');

        if (!$itemId || $quantity < 0) {
            $this->json(['error' => 'Invalid item or quantity'], 400);
        }

        $cart = $this->getCart();

        if ($quantity === 0) {
            // Remove item
            $this->db->query("DELETE FROM cart_items WHERE id = ? AND cart_id = ?")
                    ->bind(1, $itemId)->bind(2, $cart['id'])->execute();
        } else {
            // Check stock
            $item = $this->db->query("
                SELECT ci.*, p.stock_quantity FROM cart_items ci
                JOIN products p ON ci.product_id = p.id
                WHERE ci.id = ? AND ci.cart_id = ?
            ")->bind(1, $itemId)->bind(2, $cart['id'])->single();

            if ($item && $item['stock_quantity'] !== null && $item['stock_quantity'] < $quantity) {
                $this->json(['error' => 'Insufficient stock'], 400);
            }

            // Update quantity
            $this->db->update('cart_items', ['quantity' => $quantity], ['id' => $itemId]);
        }

        // Update cart total
        $this->updateCartTotal($cart['id']);

        $this->json([
            'success' => true,
            'message' => 'Cart updated'
        ]);
    }

    public function remove($itemId) {
        $cart = $this->getCart();

        $this->db->query("DELETE FROM cart_items WHERE id = ? AND cart_id = ?")
                ->bind(1, $itemId)->bind(2, $cart['id'])->execute();

        // Update cart total
        $this->updateCartTotal($cart['id']);

        $this->redirect(BASE_URL . '/shop/cart');
    }

    public function clear() {
        $cart = $this->getCart();

        $this->db->query("DELETE FROM cart_items WHERE cart_id = ?")->bind(1, $cart['id'])->execute();
        $this->updateCartTotal($cart['id']);

        $this->redirect(BASE_URL . '/shop/cart');
    }

    public function count() {
        $count = $this->getCartCount();

        $this->json([
            'success' => true,
            'count' => $count
        ]);
    }

    /**
     * Get or create cart for current session
     */
    private function getCart() {
        $sessionId = session_id();
        $userId = $this->auth->check() ? $this->auth->id() : null;

        // Try to find existing cart
        $cart = null;

        if ($userId) {
            $cart = $this->db->query("SELECT * FROM carts WHERE user_id = ? AND status = 'active' ORDER BY created_at DESC LIMIT 1")
                            ->bind(1, $userId)->single();
        }

        if (!$cart) {
            $cart = $this->db->query("SELECT * FROM carts WHERE session_id = ? AND status = 'active' ORDER BY created_at DESC LIMIT 1")
                            ->bind(1, $sessionId)->single();
        }

        // Create new cart if none exists
        if (!$cart) {
            $cartId = $this->db->insert('carts', [
                'session_id' => $sessionId,
                'user_id' => $userId,
                'status' => 'active'
            ]);

            $cart = $this->db->query("SELECT * FROM carts WHERE id = ?")->bind(1, $cartId)->single();
        }

        // Update cart with user ID if user just logged in
        if ($userId && !$cart['user_id']) {
            $this->db->update('carts', ['user_id' => $userId], ['id' => $cart['id']]);
            $cart['user_id'] = $userId;
        }

        return $cart;
    }

    /**
     * Get cart items with product details
     */
    private function getCartItems($cart) {
        return $this->db->query("
            SELECT ci.*, p.name, p.slug, p.images, p.stock_quantity,
                   (ci.price * ci.quantity) as total_price
            FROM cart_items ci
            JOIN products p ON ci.product_id = p.id
            WHERE ci.cart_id = ?
            ORDER BY ci.created_at ASC
        ")->bind(1, $cart['id'])->resultSet();
    }

    /**
     * Get cart item count
     */
    private function getCartCount() {
        $cart = $this->getCart();

        return $this->db->query("SELECT SUM(quantity) as count FROM cart_items WHERE cart_id = ?")
                       ->bind(1, $cart['id'])->single()['count'] ?? 0;
    }

    /**
     * Update cart total
     */
    private function updateCartTotal($cartId) {
        $total = $this->db->query("SELECT SUM(price * quantity) as total FROM cart_items WHERE cart_id = ?")
                         ->bind(1, $cartId)->single()['total'] ?? 0;

        $this->db->update('carts', ['total' => $total], ['id' => $cartId]);
    }
}
