<?php
/**
 * Service Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class ServiceController extends Controller {
    public function index() {
        // Get filters
        $category = $this->getQuery('category');
        $companyId = $this->getQuery('company_id') ?: 1; // Default to first company

        // Get service categories
        $categories = $this->db->query("
            SELECT * FROM service_categories
            WHERE (company_id = ? OR company_id IS NULL) AND is_active = 1
            ORDER BY sort_order ASC, name ASC
        ")->bind(1, $companyId)->resultSet();

        // Get services
        $serviceModel = new Service();
        if ($category) {
            $services = $serviceModel->getByCategory($category, true);
        } else {
            $services = $serviceModel->getByCompany($companyId, true);
        }

        // Get company info
        $company = $this->db->query("SELECT * FROM companies WHERE id = ? AND status = 'active'")
                           ->bind(1, $companyId)
                           ->single();

        $this->view('services/index', [
            'company' => $company,
            'categories' => $categories,
            'services' => $services,
            'currentCategory' => $category,
            'pageTitle' => 'Our Services'
        ]);
    }

    public function show($slug) {
        $serviceModel = new Service();
        $service = $serviceModel->findBySlug($slug);

        if (!$service) {
            $this->view('errors/404');
            return;
        }

        // Get service variants
        $variants = $serviceModel->getVariants($service['id']);

        // Get service staff
        $staff = $serviceModel->getServiceStaff($service['id']);

        // Get related services
        $relatedServices = $this->getRelatedServices($service, 3);

        // Get company and branch info
        $company = $this->db->query("SELECT * FROM companies WHERE id = ?")
                           ->bind(1, $service['company_id'])
                           ->single();

        // Get branches that offer this service
        $branches = $this->db->query("
            SELECT b.* FROM branches b
            INNER JOIN services s ON b.id = s.branch_id
            WHERE s.id = ? AND b.status = 'active'
            ORDER BY b.is_main DESC, b.name ASC
        ")->bind(1, $service['id'])->resultSet();

        // SEO data
        $pageTitle = $service['seo_title'] ?: $service['name'];
        $metaDescription = $service['seo_description'] ?: truncateText($service['short_description'], 155);

        $this->view('services/show', [
            'service' => $service,
            'variants' => $variants,
            'staff' => $staff,
            'relatedServices' => $relatedServices,
            'company' => $company,
            'branches' => $branches,
            'pageTitle' => $pageTitle,
            'metaDescription' => $metaDescription
        ]);
    }

    /**
     * Get related services
     */
    private function getRelatedServices($currentService, $limit = 3) {
        $serviceModel = new Service();

        // Get services from same category
        if ($currentService['category_id']) {
            $related = $serviceModel->getByCategory($currentService['category_id'], true);
            $related = array_filter($related, function($s) use ($currentService) {
                return $s['id'] != $currentService['id'];
            });
        } else {
            // Get other services from same company
            $related = $serviceModel->getByCompany($currentService['company_id'], true);
            $related = array_filter($related, function($s) use ($currentService) {
                return $s['id'] != $currentService['id'] && !$s['is_featured'];
            });
        }

        return array_slice($related, 0, $limit);
    }
}
