<?php
/**
 * Booking Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class BookingController extends Controller {
    public function create() {
        // Get booking parameters from query string
        $serviceId = $this->getQuery('service');
        $branchId = $this->getQuery('branch');
        $staffId = $this->getQuery('staff');
        $date = $this->getQuery('date');

        // Get service details
        $service = null;
        if ($serviceId) {
            $service = $this->db->query("
                SELECT s.*, c.name as category_name
                FROM services s
                LEFT JOIN service_categories c ON s.category_id = c.id
                WHERE s.id = ? AND s.is_active = 1
            ")->bind(1, $serviceId)->single();
        }

        // Get available branches
        $branches = $this->db->query("
            SELECT b.*, COUNT(s.id) as services_count
            FROM branches b
            LEFT JOIN services s ON b.company_id = s.company_id AND s.is_active = 1
            WHERE b.status = 'active'
            GROUP BY b.id
            ORDER BY b.is_main DESC, b.name ASC
        ")->resultSet();

        // Get company info
        $company = $this->db->query("SELECT * FROM companies WHERE status = 'active' LIMIT 1")->single();

        $this->view('booking/create', [
            'service' => $service,
            'branches' => $branches,
            'company' => $company,
            'selectedBranch' => $branchId,
            'selectedStaff' => $staffId,
            'selectedDate' => $date,
            'pageTitle' => 'Book an Appointment'
        ]);
    }

    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/book');
        }

        // Get form data
        $data = [
            'service_id' => $this->getPost('service_id'),
            'branch_id' => $this->getPost('branch_id'),
            'staff_id' => $this->getPost('staff_id') ?: null,
            'start_at' => $this->getPost('start_at'),
            'notes' => $this->getPost('notes'),
            'customer_notes' => $this->getPost('customer_notes'),
            'variant_ids' => $this->getPost('variant_ids') ?: []
        ];

        // Handle guest booking
        if (!$this->auth->check()) {
            $data['guest_details'] = [
                'first_name' => $this->getPost('first_name'),
                'last_name' => $this->getPost('last_name'),
                'email' => $this->getPost('email'),
                'phone' => $this->getPost('phone')
            ];
        }

        // Validate input
        $errors = $this->validateBookingInput($data);

        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/book?' . http_build_query([
                'service' => $data['service_id'],
                'branch' => $data['branch_id'],
                'staff' => $data['staff_id']
            ]));
        }

        try {
            // Create booking
            $bookingModel = new Booking();
            $bookingId = $bookingModel->createBooking($data);

            // Send confirmation notifications
            $this->sendBookingNotifications($bookingId);

            // Redirect to success page
            $this->redirect(BASE_URL . '/booking/success/' . $bookingId, 'Booking created successfully!');

        } catch (Exception $e) {
            error_log("Booking creation error: " . $e->getMessage());
            $_SESSION['form_errors'] = ['general' => 'Failed to create booking. Please try again.'];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/book');
        }
    }

    public function success($id) {
        $bookingModel = new Booking();
        $booking = $bookingModel->findWithDetails($id);

        if (!$booking) {
            $this->view('errors/404');
            return;
        }

        // Check if user can view this booking
        if (!$this->canViewBooking($booking)) {
            $this->view('errors/403');
            return;
        }

        $this->view('booking/success', [
            'booking' => $booking,
            'pageTitle' => 'Booking Confirmed'
        ]);
    }

    public function myBookings() {
        if (!$this->auth->check()) {
            $this->redirect(BASE_URL . '/login');
        }

        $bookingModel = new Booking();
        $bookings = $bookingModel->getByCustomer($this->auth->id());

        $this->view('booking/my-bookings', [
            'bookings' => $bookings,
            'pageTitle' => 'My Bookings'
        ]);
    }

    public function show($id) {
        if (!$this->auth->check()) {
            $this->redirect(BASE_URL . '/login');
        }

        $bookingModel = new Booking();
        $booking = $bookingModel->findWithDetails($id);

        if (!$booking || !$this->canViewBooking($booking)) {
            $this->view('errors/404');
            return;
        }

        $this->view('booking/show', [
            'booking' => $booking,
            'pageTitle' => 'Booking Details'
        ]);
    }

    public function cancel($id) {
        if (!$this->auth->check()) {
            $this->redirect(BASE_URL . '/login');
        }

        $bookingModel = new Booking();
        $booking = $bookingModel->findWithDetails($id);

        if (!$booking || !$this->canModifyBooking($booking)) {
            $this->view('errors/404');
            return;
        }

        // Check if cancellation is allowed
        if (!$this->canCancelBooking($booking)) {
            $_SESSION['flash_message'] = 'This booking cannot be cancelled at this time.';
            $_SESSION['flash_type'] = 'danger';
            $this->redirect(BASE_URL . '/booking/' . $id);
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $reason = $this->getPost('reason');
                $bookingModel->updateStatus($id, 'cancelled', $this->auth->id());

                // Update cancellation reason
                if ($reason) {
                    $bookingModel->update($id, ['cancellation_reason' => $reason]);
                }

                // Send cancellation notifications
                $this->sendCancellationNotifications($id);

                $this->redirect(BASE_URL . '/booking/' . $id, 'Booking cancelled successfully.');

            } catch (Exception $e) {
                error_log("Booking cancellation error: " . $e->getMessage());
                $_SESSION['flash_message'] = 'Failed to cancel booking. Please try again.';
                $_SESSION['flash_type'] = 'danger';
                $this->redirect(BASE_URL . '/booking/' . $id);
            }
        }

        $this->view('booking/cancel', [
            'booking' => $booking,
            'pageTitle' => 'Cancel Booking'
        ]);
    }

    /**
     * Validate booking input
     */
    private function validateBookingInput($data) {
        $errors = [];

        // Validate service
        $service = $this->db->query("SELECT * FROM services WHERE id = ? AND is_active = 1")
                           ->bind(1, $data['service_id'])
                           ->single();
        if (!$service) {
            $errors['service_id'] = 'Invalid service selected';
        }

        // Validate branch
        $branch = $this->db->query("SELECT * FROM branches WHERE id = ? AND status = 'active'")
                          ->bind(1, $data['branch_id'])
                          ->single();
        if (!$branch) {
            $errors['branch_id'] = 'Invalid branch selected';
        }

        // Validate staff (if provided)
        if (!empty($data['staff_id'])) {
            $staff = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'staff' AND status = 'active'")
                             ->bind(1, $data['staff_id'])
                             ->single();
            if (!$staff) {
                $errors['staff_id'] = 'Invalid staff member selected';
            }
        }

        // Validate datetime
        if (empty($data['start_at']) || !strtotime($data['start_at'])) {
            $errors['start_at'] = 'Invalid date and time';
        } else {
            $bookingTime = strtotime($data['start_at']);
            $now = time();

            if ($bookingTime <= $now + (MIN_ADVANCE_BOOKING_HOURS * 3600)) {
                $errors['start_at'] = 'Bookings must be made at least ' . MIN_ADVANCE_BOOKING_HOURS . ' hours in advance';
            }

            if ($bookingTime > strtotime('+' . MAX_ADVANCE_BOOKING_DAYS . ' days')) {
                $errors['start_at'] = 'Cannot book more than ' . MAX_ADVANCE_BOOKING_DAYS . ' days in advance';
            }
        }

        // Validate guest details (if not logged in)
        if (!$this->auth->check() && isset($data['guest_details'])) {
            $guestErrors = $this->validateGuestDetails($data['guest_details']);
            if (!empty($guestErrors)) {
                $errors = array_merge($errors, $guestErrors);
            }
        }

        return $errors;
    }

    /**
     * Validate guest details
     */
    private function validateGuestDetails($guest) {
        $errors = [];

        if (empty($guest['first_name']) || strlen($guest['first_name']) < 2) {
            $errors['first_name'] = 'First name must be at least 2 characters';
        }

        if (empty($guest['last_name']) || strlen($guest['last_name']) < 2) {
            $errors['last_name'] = 'Last name must be at least 2 characters';
        }

        if (empty($guest['email']) || !filter_var($guest['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Valid email address is required';
        }

        if (empty($guest['phone']) || !preg_match('/^[\+]?[1-9][\d]{0,15}$/', $guest['phone'])) {
            $errors['phone'] = 'Valid phone number is required';
        }

        return $errors;
    }

    /**
     * Check if user can view booking
     */
    private function canViewBooking($booking) {
        if ($this->auth->isAdmin()) {
            return true;
        }

        // Customers can view their own bookings
        return $booking['customer_id'] == $this->auth->id() ||
               ($booking['guest_email'] && $booking['guest_email'] === $this->auth->user()['email']);
    }

    /**
     * Check if user can modify booking
     */
    private function canModifyBooking($booking) {
        // Customers can modify their pending bookings
        return $booking['customer_id'] == $this->auth->id() &&
               in_array($booking['status'], ['pending', 'confirmed']);
    }

    /**
     * Check if booking can be cancelled
     */
    private function canCancelBooking($booking) {
        // Can cancel if booking is in future and not completed/cancelled
        $bookingTime = strtotime($booking['start_at']);
        $now = time();

        return $bookingTime > $now &&
               !in_array($booking['status'], ['completed', 'cancelled', 'no_show']);
    }

    /**
     * Send booking notifications
     */
    private function sendBookingNotifications($bookingId) {
        // This would integrate with notification system
        // For now, just log the action
        error_log("Booking confirmation notifications sent for booking ID: {$bookingId}");
    }

    /**
     * Send cancellation notifications
     */
    private function sendCancellationNotifications($bookingId) {
        // This would integrate with notification system
        error_log("Booking cancellation notifications sent for booking ID: {$bookingId}");
    }
}
