<?php
/**
 * Service API Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class ServiceController extends Controller {
    public function index() {
        try {
            $branchId = $this->getQuery('branch_id');
            $categoryId = $this->getQuery('category_id');
            $companyId = $this->getQuery('company_id');
            $featured = $this->getQuery('featured') === '1';
            $limit = (int) $this->getQuery('limit', 20);
            $offset = (int) $this->getQuery('offset', 0);

            $serviceModel = new Service();

            if ($branchId) {
                // Get services for specific branch
                $services = $serviceModel->getByBranch($branchId, true);
            } elseif ($companyId) {
                // Get services for specific company
                $services = $serviceModel->getByCompany($companyId, true);
            } elseif ($featured) {
                // Get featured services (need company context)
                $companyId = $this->getQuery('company_id');
                if (!$companyId) {
                    return $this->json(['error' => 'company_id required for featured services'], 400);
                }
                $services = $serviceModel->getFeatured($companyId, $limit);
            } else {
                // Get all active services
                $services = $serviceModel->all('name ASC', $limit);
            }

            // Filter by category if specified
            if ($categoryId) {
                $services = array_filter($services, function($service) use ($categoryId) {
                    return $service['category_id'] == $categoryId;
                });
            }

            // Apply pagination
            $services = array_slice($services, $offset, $limit);

            // Format response
            $formattedServices = array_map(function($service) {
                return [
                    'id' => $service['id'],
                    'name' => $service['name'],
                    'slug' => $service['slug'],
                    'description' => $service['short_description'] ?: $service['description'],
                    'price' => (float) $service['price'],
                    'duration_minutes' => (int) $service['duration_minutes'],
                    'image' => $service['images'] ? getServiceImageUrl($service['images']) : null,
                    'is_featured' => (bool) $service['is_featured'],
                    'category' => $service['category_name'] ?? null,
                    'seo_title' => $service['seo_title'],
                    'seo_description' => $service['seo_description']
                ];
            }, $services);

            $this->json([
                'success' => true,
                'data' => $formattedServices,
                'meta' => [
                    'total' => count($services),
                    'limit' => $limit,
                    'offset' => $offset
                ]
            ]);

        } catch (Exception $e) {
            error_log("Service API Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to fetch services'], 500);
        }
    }

    public function show($slug) {
        try {
            $serviceModel = new Service();
            $service = $serviceModel->findBySlug($slug);

            if (!$service) {
                return $this->json(['error' => 'Service not found'], 404);
            }

            // Get service variants
            $variants = $serviceModel->getVariants($service['id']);

            // Get service staff
            $staff = $serviceModel->getServiceStaff($service['id']);

            // Get related services (same category)
            $relatedServices = [];
            if ($service['category_id']) {
                $relatedServices = $serviceModel->getByCategory($service['category_id'], true);
                $relatedServices = array_filter($relatedServices, function($s) use ($service) {
                    return $s['id'] != $service['id'];
                });
                $relatedServices = array_slice($relatedServices, 0, 3);
            }

            $formattedService = [
                'id' => $service['id'],
                'name' => $service['name'],
                'slug' => $service['slug'],
                'description' => $service['description'],
                'short_description' => $service['short_description'],
                'price' => (float) $service['price'],
                'duration_minutes' => (int) $service['duration_minutes'],
                'buffer_before_minutes' => (int) $service['buffer_before_minutes'],
                'buffer_after_minutes' => (int) $service['buffer_after_minutes'],
                'max_bookings_per_slot' => (int) $service['max_bookings_per_slot'],
                'images' => array_map(function($image) {
                    return getServiceImageUrl([$image]);
                }, $service['images'] ? json_decode($service['images'], true) : []),
                'is_featured' => (bool) $service['is_featured'],
                'category' => $service['category_name'] ?? null,
                'seo_title' => $service['seo_title'],
                'seo_description' => $service['seo_description'],
                'seo_keywords' => $service['seo_keywords'],
                'variants' => array_map(function($variant) {
                    return [
                        'id' => $variant['id'],
                        'name' => $variant['name'],
                        'description' => $variant['description'],
                        'price_modifier' => (float) $variant['price_modifier'],
                        'duration_modifier' => (int) $variant['duration_modifier'],
                        'is_required' => (bool) $variant['is_required']
                    ];
                }, $variants),
                'available_staff' => array_map(function($member) {
                    return [
                        'id' => $member['id'],
                        'name' => $member['first_name'] . ' ' . $member['last_name'],
                        'avatar' => getAvatarUrl($member['avatar'])
                    ];
                }, $staff),
                'related_services' => array_map(function($related) {
                    return [
                        'id' => $related['id'],
                        'name' => $related['name'],
                        'slug' => $related['slug'],
                        'price' => (float) $related['price'],
                        'image' => getServiceImageUrl($related['images'])
                    ];
                }, $relatedServices)
            ];

            // Add schema.org structured data
            if ($service['schema_data']) {
                $formattedService['schema_org'] = json_decode($service['schema_data'], true);
            }

            $this->json([
                'success' => true,
                'data' => $formattedService
            ]);

        } catch (Exception $e) {
            error_log("Service API Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to fetch service details'], 500);
        }
    }
}
