<?php
/**
 * Branch API Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class BranchController extends Controller {
    public function index() {
        try {
            $companyId = $this->getQuery('company_id');
            $nearby = $this->getQuery('nearby');
            $latitude = $this->getQuery('lat');
            $longitude = $this->getQuery('lng');
            $radius = (int) $this->getQuery('radius', 50); // km

            if (!$companyId && !$nearby) {
                return $this->json(['error' => 'company_id or nearby parameter required'], 400);
            }

            $branchModel = new Branch();

            if ($nearby && $latitude && $longitude) {
                // Find nearby branches
                $branches = $this->getNearbyBranches($latitude, $longitude, $radius, $companyId);
            } else {
                // Get branches by company
                $branches = $branchModel->getByCompany($companyId, true);
            }

            // Format response
            $formattedBranches = array_map(function($branch) {
                return [
                    'id' => $branch['id'],
                    'name' => $branch['name'],
                    'slug' => $branch['slug'],
                    'address' => $branch['address'],
                    'city' => $branch['city'],
                    'state' => $branch['state'],
                    'country' => $branch['country'],
                    'postal_code' => $branch['postal_code'],
                    'phone' => $branch['phone'],
                    'email' => $branch['email'],
                    'coordinates' => [
                        'latitude' => (float) $branch['latitude'],
                        'longitude' => (float) $branch['longitude']
                    ],
                    'timezone' => $branch['timezone'],
                    'is_main' => (bool) $branch['is_main'],
                    'opening_hours' => $branch['opening_hours'] ? json_decode($branch['opening_hours'], true) : null,
                    'services_count' => $branch['services_count'] ?? 0,
                    'staff_count' => $branch['staff_count'] ?? 0
                ];
            }, $branches);

            $this->json([
                'success' => true,
                'data' => $formattedBranches,
                'meta' => [
                    'total' => count($formattedBranches),
                    'radius' => $nearby ? $radius : null
                ]
            ]);

        } catch (Exception $e) {
            error_log("Branch API Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to fetch branches'], 500);
        }
    }

    public function show($id) {
        try {
            $branchModel = new Branch();
            $branch = $branchModel->findWithCompany($id);

            if (!$branch) {
                return $this->json(['error' => 'Branch not found'], 404);
            }

            if ($branch['status'] !== 'active') {
                return $this->json(['error' => 'Branch is not available'], 403);
            }

            // Get branch services
            $services = $branchModel->getServices($id, true);

            // Get branch staff
            $staff = $branchModel->getStaff($id, true);

            // Get today's availability
            $today = date('Y-m-d');
            $todayAvailability = $branchModel->getAvailability($id, $today);

            $formattedBranch = [
                'id' => $branch['id'],
                'name' => $branch['name'],
                'slug' => $branch['slug'],
                'company' => [
                    'id' => $branch['company_id'],
                    'name' => $branch['company_name'],
                    'slug' => $branch['company_slug'],
                    'primary_color' => $branch['primary_color'],
                    'secondary_color' => $branch['secondary_color']
                ],
                'address' => $branch['address'],
                'city' => $branch['city'],
                'state' => $branch['state'],
                'country' => $branch['country'],
                'postal_code' => $branch['postal_code'],
                'phone' => $branch['phone'],
                'email' => $branch['email'],
                'coordinates' => [
                    'latitude' => (float) $branch['latitude'],
                    'longitude' => (float) $branch['longitude']
                ],
                'timezone' => $branch['timezone'],
                'is_main' => (bool) $branch['is_main'],
                'opening_hours' => $branch['opening_hours'] ? json_decode($branch['opening_hours'], true) : null,
                'today_availability' => $todayAvailability,
                'services' => array_map(function($service) {
                    return [
                        'id' => $service['id'],
                        'name' => $service['name'],
                        'slug' => $service['slug'],
                        'description' => $service['short_description'],
                        'price' => (float) $service['price'],
                        'duration_minutes' => (int) $service['duration_minutes'],
                        'image' => getServiceImageUrl($service['images'])
                    ];
                }, $services),
                'staff' => array_map(function($member) {
                    return [
                        'id' => $member['id'],
                        'name' => $member['first_name'] . ' ' . $member['last_name'],
                        'email' => $member['email'],
                        'phone' => $member['phone'],
                        'avatar' => getAvatarUrl($member['avatar'])
                    ];
                }, $staff)
            ];

            $this->json([
                'success' => true,
                'data' => $formattedBranch
            ]);

        } catch (Exception $e) {
            error_log("Branch API Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to fetch branch details'], 500);
        }
    }

    /**
     * Get nearby branches using Haversine formula
     */
    private function getNearbyBranches($latitude, $longitude, $radiusKm, $companyId = null) {
        $branches = $this->db->query("
            SELECT *,
                   (6371 * acos(cos(radians(?)) * cos(radians(latitude)) * cos(radians(longitude) - radians(?)) + sin(radians(?)) * sin(radians(latitude)))) AS distance
            FROM branches
            WHERE status = 'active'
            " . ($companyId ? "AND company_id = ?" : "") . "
            HAVING distance < ?
            ORDER BY distance ASC
        ", $companyId ? [$latitude, $longitude, $latitude, $companyId, $radiusKm] : [$latitude, $longitude, $latitude, $radiusKm])
        ->resultSet();

        return $branches;
    }
}
