<?php
/**
 * Auth API Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class AuthController extends Controller {
    public function login() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);

            if (!$input) {
                return $this->json(['error' => 'Invalid JSON input'], 400);
            }

            $email = $input['email'] ?? '';
            $password = $input['password'] ?? '';

            // Validate input
            if (empty($email) || empty($password)) {
                return $this->json(['error' => 'Email and password are required'], 422);
            }

            if (!$this->validateEmail($email)) {
                return $this->json(['error' => 'Invalid email format'], 422);
            }

            // Attempt login
            if ($this->auth->attempt($email, $password)) {
                $user = $this->auth->user();

                // Generate access token (simplified - in production use JWT)
                $token = bin2hex(random_bytes(32));
                $_SESSION['api_token'] = $token;

                $this->json([
                    'success' => true,
                    'message' => 'Login successful',
                    'data' => [
                        'user' => $this->formatUserResponse($user),
                        'token' => $token,
                        'expires_in' => SESSION_LIFETIME
                    ]
                ]);
            } else {
                return $this->json(['error' => 'Invalid credentials'], 401);
            }

        } catch (Exception $e) {
            error_log("Auth API Login Error: " . $e->getMessage());
            $this->json(['error' => 'Login failed'], 500);
        }
    }

    public function register() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);

            if (!$input) {
                return $this->json(['error' => 'Invalid JSON input'], 400);
            }

            // Validate required fields
            $requiredFields = ['first_name', 'last_name', 'email', 'password'];
            $validationErrors = $this->validateRequired($input, $requiredFields);

            if (!empty($validationErrors)) {
                return $this->json(['error' => 'Validation failed', 'details' => $validationErrors], 422);
            }

            // Additional validation
            $errors = [];

            if (!$this->validateEmail($input['email'])) {
                $errors['email'] = 'Invalid email format';
            }

            if (strlen($input['password']) < 8) {
                $errors['password'] = 'Password must be at least 8 characters long';
            }

            if ($input['password'] !== ($input['password_confirm'] ?? '')) {
                $errors['password_confirm'] = 'Passwords do not match';
            }

            // Check if email already exists
            $userModel = new User();
            if ($userModel->emailExists($input['email'])) {
                $errors['email'] = 'Email address already registered';
            }

            if (!empty($errors)) {
                return $this->json(['error' => 'Validation failed', 'details' => $errors], 422);
            }

            // Create user
            $userData = [
                'first_name' => $input['first_name'],
                'last_name' => $input['last_name'],
                'email' => $input['email'],
                'phone' => $input['phone'] ?? '',
                'status' => 'active'
            ];

            $userId = $this->auth->register($userData);

            if ($userId) {
                // Auto-login after registration
                $this->auth->attempt($input['email'], $input['password']);
                $user = $this->auth->user();

                // Generate access token
                $token = bin2hex(random_bytes(32));
                $_SESSION['api_token'] = $token;

                $this->json([
                    'success' => true,
                    'message' => 'Account created successfully',
                    'data' => [
                        'user' => $this->formatUserResponse($user),
                        'token' => $token,
                        'expires_in' => SESSION_LIFETIME
                    ]
                ], 201);
            } else {
                $this->json(['error' => 'Registration failed'], 500);
            }

        } catch (Exception $e) {
            error_log("Auth API Register Error: " . $e->getMessage());
            $this->json(['error' => 'Registration failed'], 500);
        }
    }

    public function logout() {
        try {
            $this->auth->logout();

            $this->json([
                'success' => true,
                'message' => 'Logout successful'
            ]);

        } catch (Exception $e) {
            error_log("Auth API Logout Error: " . $e->getMessage());
            $this->json(['error' => 'Logout failed'], 500);
        }
    }

    public function profile() {
        try {
            if (!$this->auth->check()) {
                return $this->json(['error' => 'Authentication required'], 401);
            }

            $user = $this->auth->user();

            $this->json([
                'success' => true,
                'data' => $this->formatUserResponse($user)
            ]);

        } catch (Exception $e) {
            error_log("Auth API Profile Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to fetch profile'], 500);
        }
    }

    public function updateProfile() {
        try {
            if (!$this->auth->check()) {
                return $this->json(['error' => 'Authentication required'], 401);
            }

            $input = json_decode(file_get_contents('php://input'), true);

            if (!$input) {
                return $this->json(['error' => 'Invalid JSON input'], 400);
            }

            $userModel = new User();
            $userId = $this->auth->id();

            // Validate input
            $updateData = [];
            $errors = [];

            if (isset($input['first_name'])) {
                if (empty($input['first_name']) || strlen($input['first_name']) < 2) {
                    $errors['first_name'] = 'First name must be at least 2 characters';
                } else {
                    $updateData['first_name'] = $input['first_name'];
                }
            }

            if (isset($input['last_name'])) {
                if (empty($input['last_name']) || strlen($input['last_name']) < 2) {
                    $errors['last_name'] = 'Last name must be at least 2 characters';
                } else {
                    $updateData['last_name'] = $input['last_name'];
                }
            }

            if (isset($input['phone'])) {
                if (!empty($input['phone']) && !preg_match('/^[\+]?[1-9][\d]{0,15}$/', $input['phone'])) {
                    $errors['phone'] = 'Invalid phone number format';
                } else {
                    $updateData['phone'] = $input['phone'];
                }
            }

            if (!empty($errors)) {
                return $this->json(['error' => 'Validation failed', 'details' => $errors], 422);
            }

            if (!empty($updateData)) {
                $result = $userModel->update($userId, $updateData);

                if ($result) {
                    // Refresh user session
                    $this->auth->refreshUser();
                    $updatedUser = $this->auth->user();

                    $this->json([
                        'success' => true,
                        'message' => 'Profile updated successfully',
                        'data' => $this->formatUserResponse($updatedUser)
                    ]);
                } else {
                    $this->json(['error' => 'Failed to update profile'], 500);
                }
            } else {
                $this->json(['error' => 'No valid fields to update'], 400);
            }

        } catch (Exception $e) {
            error_log("Auth API Update Profile Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to update profile'], 500);
        }
    }

    public function changePassword() {
        try {
            if (!$this->auth->check()) {
                return $this->json(['error' => 'Authentication required'], 401);
            }

            $input = json_decode(file_get_contents('php://input'), true);

            if (!$input) {
                return $this->json(['error' => 'Invalid JSON input'], 400);
            }

            $currentPassword = $input['current_password'] ?? '';
            $newPassword = $input['new_password'] ?? '';
            $confirmPassword = $input['confirm_password'] ?? '';

            // Validate input
            $errors = [];

            if (empty($currentPassword)) {
                $errors['current_password'] = 'Current password is required';
            }

            if (empty($newPassword) || strlen($newPassword) < 8) {
                $errors['new_password'] = 'New password must be at least 8 characters long';
            }

            if ($newPassword !== $confirmPassword) {
                $errors['confirm_password'] = 'Passwords do not match';
            }

            if (!empty($errors)) {
                return $this->json(['error' => 'Validation failed', 'details' => $errors], 422);
            }

            // Verify current password
            $user = $this->auth->user();
            if (!password_verify($currentPassword, $user['password_hash'])) {
                return $this->json(['error' => 'Current password is incorrect'], 422);
            }

            // Update password
            $userModel = new User();
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);

            $result = $userModel->update($this->auth->id(), ['password_hash' => $hashedPassword]);

            if ($result) {
                $this->json([
                    'success' => true,
                    'message' => 'Password changed successfully'
                ]);
            } else {
                $this->json(['error' => 'Failed to change password'], 500);
            }

        } catch (Exception $e) {
            error_log("Auth API Change Password Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to change password'], 500);
        }
    }

    public function forgotPassword() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);

            if (!$input || empty($input['email'])) {
                return $this->json(['error' => 'Email is required'], 400);
            }

            $email = $input['email'];

            if (!$this->validateEmail($email)) {
                return $this->json(['error' => 'Invalid email format'], 422);
            }

            // Check if user exists
            $userModel = new User();
            $user = $userModel->findByEmail($email);

            if ($user) {
                // Generate reset token
                $token = $this->auth->generatePasswordResetToken($email);

                if ($token) {
                    // Send reset email (would integrate with email service)
                    $this->sendPasswordResetEmail($email, $token);

                    $this->json([
                        'success' => true,
                        'message' => 'Password reset instructions sent to your email'
                    ]);
                } else {
                    $this->json(['error' => 'Failed to generate reset token'], 500);
                }
            } else {
                // Don't reveal if email exists for security
                $this->json([
                    'success' => true,
                    'message' => 'If an account with this email exists, password reset instructions have been sent'
                ]);
            }

        } catch (Exception $e) {
            error_log("Auth API Forgot Password Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to process password reset request'], 500);
        }
    }

    public function resetPassword() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);

            if (!$input) {
                return $this->json(['error' => 'Invalid JSON input'], 400);
            }

            $token = $input['token'] ?? '';
            $password = $input['password'] ?? '';
            $confirmPassword = $input['confirm_password'] ?? '';

            // Validate input
            $errors = [];

            if (empty($token)) {
                $errors['token'] = 'Reset token is required';
            }

            if (empty($password) || strlen($password) < 8) {
                $errors['password'] = 'Password must be at least 8 characters long';
            }

            if ($password !== $confirmPassword) {
                $errors['confirm_password'] = 'Passwords do not match';
            }

            if (!empty($errors)) {
                return $this->json(['error' => 'Validation failed', 'details' => $errors], 422);
            }

            // Reset password
            $result = $this->auth->resetPassword($token, $password);

            if ($result) {
                $this->json([
                    'success' => true,
                    'message' => 'Password reset successfully'
                ]);
            } else {
                $this->json(['error' => 'Invalid or expired reset token'], 422);
            }

        } catch (Exception $e) {
            error_log("Auth API Reset Password Error: " . $e->getMessage());
            $this->json(['error' => 'Failed to reset password'], 500);
        }
    }

    /**
     * Format user response data
     */
    private function formatUserResponse($user) {
        return [
            'id' => $user['id'],
            'first_name' => $user['first_name'],
            'last_name' => $user['last_name'],
            'email' => $user['email'],
            'phone' => $user['phone'],
            'avatar' => getAvatarUrl($user['avatar']),
            'role' => $user['role'],
            'status' => $user['status'],
            'company' => $user['company_name'] ? [
                'id' => $user['company_id'],
                'name' => $user['company_name'],
                'slug' => $user['company_slug']
            ] : null,
            'branch' => $user['branch_name'] ? [
                'id' => $user['branch_id'],
                'name' => $user['branch_name'],
                'slug' => $user['branch_slug']
            ] : null,
            'permissions' => $user['permissions'] ?? [],
            'roles' => $user['roles'] ?? [],
            'last_login' => $user['last_login'],
            'created_at' => $user['created_at']
        ];
    }

    /**
     * Send password reset email (stub)
     */
    private function sendPasswordResetEmail($email, $token) {
        // This would integrate with email service
        // For now, just log the action
        error_log("Password reset email sent to {$email} with token {$token}");
    }
}
