<?php
/**
 * Admin Reports Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class ReportController extends Controller {
    public function index() {
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();

        // Get report filters from query parameters
        $dateFrom = $this->getQuery('date_from', date('Y-m-d', strtotime('-30 days')));
        $dateTo = $this->getQuery('date_to', date('Y-m-d'));
        $branch = $this->getQuery('branch');
        $reportType = $this->getQuery('type', 'overview');

        // Get branches for filter
        $branches = $this->db->query("
            SELECT id, name FROM branches
            WHERE company_id = ? AND status = 'active'
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        // Generate report data based on type
        $reportData = $this->generateReport($companyId, $branchId, $reportType, $dateFrom, $dateTo, $branch);

        $this->adminView('reports/index', [
            'reportData' => $reportData,
            'reportType' => $reportType,
            'branches' => $branches,
            'filters' => [
                'date_from' => $dateFrom,
                'date_to' => $dateTo,
                'branch' => $branch
            ],
            'pageTitle' => 'Reports & Analytics'
        ]);
    }

    public function export() {
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();
        $reportType = $this->getQuery('type', 'overview');
        $dateFrom = $this->getQuery('date_from', date('Y-m-d', strtotime('-30 days')));
        $dateTo = $this->getQuery('date_to', date('Y-m-d'));
        $branch = $this->getQuery('branch');
        $format = $this->getQuery('format', 'csv');

        $reportData = $this->generateReport($companyId, $branchId, $reportType, $dateFrom, $dateTo, $branch);

        switch ($format) {
            case 'csv':
                $this->exportCSV($reportData, $reportType);
                break;
            case 'pdf':
                $this->exportPDF($reportData, $reportType);
                break;
            default:
                $this->json(['error' => 'Unsupported export format'], 400);
        }
    }

    /**
     * Generate report data
     */
    private function generateReport($companyId, $branchId, $type, $dateFrom, $dateTo, $branchFilter = null) {
        $whereClause = "b.created_at BETWEEN ? AND ?";
        $params = [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'];

        // Apply branch filter
        if ($branchFilter) {
            $whereClause .= " AND b.branch_id = ?";
            $params[] = $branchFilter;
        } elseif ($branchId) {
            $whereClause .= " AND b.branch_id = ?";
            $params[] = $branchId;
        } elseif ($companyId) {
            $whereClause .= " AND b.company_id = ?";
            $params[] = $companyId;
        }

        switch ($type) {
            case 'bookings':
                return $this->generateBookingsReport($whereClause, $params, $dateFrom, $dateTo);
            case 'revenue':
                return $this->generateRevenueReport($whereClause, $params, $dateFrom, $dateTo);
            case 'services':
                return $this->generateServicesReport($whereClause, $params, $dateFrom, $dateTo);
            case 'staff':
                return $this->generateStaffReport($whereClause, $params, $dateFrom, $dateTo);
            case 'customers':
                return $this->generateCustomersReport($whereClause, $params, $dateFrom, $dateTo);
            default:
                return $this->generateOverviewReport($whereClause, $params, $dateFrom, $dateTo);
        }
    }

    /**
     * Generate overview report
     */
    private function generateOverviewReport($whereClause, $params, $dateFrom, $dateTo) {
        // Total bookings
        $totalBookings = $this->db->query("
            SELECT COUNT(*) as total FROM bookings b WHERE {$whereClause}
        ", $params)->single()['total'];

        // Completed bookings
        $completedBookings = $this->db->query("
            SELECT COUNT(*) as total FROM bookings b WHERE {$whereClause} AND b.status = 'completed'
        ", $params)->single()['total'];

        // Cancelled bookings
        $cancelledBookings = $this->db->query("
            SELECT COUNT(*) as total FROM bookings b WHERE {$whereClause} AND b.status = 'cancelled'
        ", $params)->single()['total'];

        // Total revenue
        $totalRevenue = $this->db->query("
            SELECT COALESCE(SUM(b.total_amount), 0) as revenue FROM bookings b
            WHERE {$whereClause} AND b.status = 'completed'
        ", $params)->single()['revenue'];

        // Average booking value
        $avgBookingValue = $totalRevenue / max($completedBookings, 1);

        // Top services
        $topServices = $this->db->query("
            SELECT s.name, COUNT(b.id) as bookings, COALESCE(SUM(b.total_amount), 0) as revenue
            FROM services s
            LEFT JOIN bookings b ON s.id = b.service_id AND {$whereClause}
            WHERE s.company_id = ?
            GROUP BY s.id, s.name
            ORDER BY bookings DESC
            LIMIT 10
        ", array_merge($params, [$this->auth->companyId()]))->resultSet();

        // Revenue by day
        $revenueByDay = $this->db->query("
            SELECT DATE(b.created_at) as date, COALESCE(SUM(b.total_amount), 0) as revenue
            FROM bookings b
            WHERE {$whereClause} AND b.status = 'completed'
            GROUP BY DATE(b.created_at)
            ORDER BY date ASC
        ", $params)->resultSet();

        // Bookings by status
        $bookingsByStatus = $this->db->query("
            SELECT b.status, COUNT(*) as count
            FROM bookings b
            WHERE {$whereClause}
            GROUP BY b.status
            ORDER BY count DESC
        ", $params)->resultSet();

        return [
            'type' => 'overview',
            'period' => ['from' => $dateFrom, 'to' => $dateTo],
            'summary' => [
                'total_bookings' => $totalBookings,
                'completed_bookings' => $completedBookings,
                'cancelled_bookings' => $cancelledBookings,
                'total_revenue' => $totalRevenue,
                'avg_booking_value' => $avgBookingValue,
                'completion_rate' => $totalBookings > 0 ? ($completedBookings / $totalBookings) * 100 : 0
            ],
            'top_services' => $topServices,
            'revenue_chart' => $revenueByDay,
            'status_distribution' => $bookingsByStatus
        ];
    }

    /**
     * Generate bookings report
     */
    private function generateBookingsReport($whereClause, $params, $dateFrom, $dateTo) {
        $bookings = $this->db->query("
            SELECT b.*, s.name as service_name, br.name as branch_name,
                   u.first_name, u.last_name, u.email, u.phone,
                   st.first_name as staff_first_name, st.last_name as staff_last_name
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN branches br ON b.branch_id = br.id
            LEFT JOIN users u ON b.customer_id = u.id
            LEFT JOIN users st ON b.staff_id = st.id
            WHERE {$whereClause}
            ORDER BY b.start_at DESC
        ", $params)->resultSet();

        // Group by status
        $byStatus = [];
        foreach ($bookings as $booking) {
            $status = $booking['status'];
            if (!isset($byStatus[$status])) {
                $byStatus[$status] = [];
            }
            $byStatus[$status][] = $booking;
        }

        return [
            'type' => 'bookings',
            'period' => ['from' => $dateFrom, 'to' => $dateTo],
            'total_bookings' => count($bookings),
            'bookings_by_status' => $byStatus,
            'all_bookings' => $bookings
        ];
    }

    /**
     * Generate revenue report
     */
    private function generateRevenueReport($whereClause, $params, $dateFrom, $dateTo) {
        // Revenue by service
        $revenueByService = $this->db->query("
            SELECT s.name as service_name, COUNT(b.id) as bookings,
                   COALESCE(SUM(b.total_amount), 0) as revenue,
                   COALESCE(AVG(b.total_amount), 0) as avg_revenue
            FROM services s
            LEFT JOIN bookings b ON s.id = b.service_id AND {$whereClause} AND b.status = 'completed'
            WHERE s.company_id = ?
            GROUP BY s.id, s.name
            ORDER BY revenue DESC
        ", array_merge($params, [$this->auth->companyId()]))->resultSet();

        // Revenue by day
        $revenueByDay = $this->db->query("
            SELECT DATE(b.created_at) as date, COUNT(b.id) as bookings,
                   COALESCE(SUM(b.total_amount), 0) as revenue
            FROM bookings b
            WHERE {$whereClause} AND b.status = 'completed'
            GROUP BY DATE(b.created_at)
            ORDER BY date ASC
        ", $params)->resultSet();

        // Revenue by payment method
        $revenueByPayment = $this->db->query("
            SELECT COALESCE(pm.name, 'Unknown') as method,
                   COUNT(p.id) as transactions,
                   COALESCE(SUM(p.amount), 0) as revenue
            FROM payments p
            LEFT JOIN payment_methods pm ON p.payment_method_id = pm.id
            LEFT JOIN bookings b ON p.booking_id = b.id
            WHERE {$whereClause} AND p.status = 'completed'
            GROUP BY pm.name
            ORDER BY revenue DESC
        ", $params)->resultSet();

        $totalRevenue = array_sum(array_column($revenueByService, 'revenue'));

        return [
            'type' => 'revenue',
            'period' => ['from' => $dateFrom, 'to' => $dateTo],
            'total_revenue' => $totalRevenue,
            'revenue_by_service' => $revenueByService,
            'revenue_by_day' => $revenueByDay,
            'revenue_by_payment' => $revenueByPayment
        ];
    }

    /**
     * Generate services report
     */
    private function generateServicesReport($whereClause, $params, $dateFrom, $dateTo) {
        $services = $this->db->query("
            SELECT s.name, s.price, COUNT(b.id) as total_bookings,
                   COUNT(CASE WHEN b.status = 'completed' THEN 1 END) as completed_bookings,
                   COUNT(CASE WHEN b.status = 'cancelled' THEN 1 END) as cancelled_bookings,
                   COALESCE(SUM(CASE WHEN b.status = 'completed' THEN b.total_amount END), 0) as total_revenue,
                   COALESCE(AVG(CASE WHEN b.status = 'completed' THEN b.total_amount END), 0) as avg_revenue,
                   MIN(CASE WHEN b.status = 'completed' THEN b.total_amount END) as min_revenue,
                   MAX(CASE WHEN b.status = 'completed' THEN b.total_amount END) as max_revenue
            FROM services s
            LEFT JOIN bookings b ON s.id = b.service_id AND {$whereClause}
            WHERE s.company_id = ?
            GROUP BY s.id, s.name, s.price
            ORDER BY total_bookings DESC
        ", array_merge($params, [$this->auth->companyId()]))->resultSet();

        return [
            'type' => 'services',
            'period' => ['from' => $dateFrom, 'to' => $dateTo],
            'services' => $services,
            'total_services' => count($services)
        ];
    }

    /**
     * Generate staff report
     */
    private function generateStaffReport($whereClause, $params, $dateFrom, $dateTo) {
        $staff = $this->db->query("
            SELECT u.first_name, u.last_name, u.id,
                   COUNT(b.id) as total_bookings,
                   COUNT(CASE WHEN b.status = 'completed' THEN 1 END) as completed_bookings,
                   COUNT(CASE WHEN b.status = 'cancelled' THEN 1 END) as cancelled_bookings,
                   COUNT(CASE WHEN b.status = 'no_show' THEN 1 END) as no_show_bookings,
                   COALESCE(SUM(CASE WHEN b.status = 'completed' THEN b.total_amount END), 0) as total_revenue,
                   COALESCE(AVG(CASE WHEN b.status = 'completed' THEN b.total_amount END), 0) as avg_booking_value
            FROM users u
            LEFT JOIN bookings b ON u.id = b.staff_id AND {$whereClause}
            WHERE u.company_id = ? AND u.role = 'staff' AND u.status = 'active'
            GROUP BY u.id, u.first_name, u.last_name
            ORDER BY total_bookings DESC
        ", array_merge($params, [$this->auth->companyId()]))->resultSet();

        // Staff performance over time
        $performanceOverTime = $this->db->query("
            SELECT DATE(b.created_at) as date, st.first_name, st.last_name,
                   COUNT(b.id) as bookings, COALESCE(SUM(b.total_amount), 0) as revenue
            FROM users st
            LEFT JOIN bookings b ON st.id = b.staff_id AND {$whereClause} AND b.status = 'completed'
            WHERE st.company_id = ? AND st.role = 'staff' AND st.status = 'active'
            GROUP BY DATE(b.created_at), st.id, st.first_name, st.last_name
            ORDER BY date ASC, bookings DESC
        ", array_merge($params, [$this->auth->companyId()]))->resultSet();

        return [
            'type' => 'staff',
            'period' => ['from' => $dateFrom, 'to' => $dateTo],
            'staff_performance' => $staff,
            'performance_over_time' => $performanceOverTime,
            'total_staff' => count($staff)
        ];
    }

    /**
     * Generate customers report
     */
    private function generateCustomersReport($whereClause, $params, $dateFrom, $dateTo) {
        // New customers
        $newCustomers = $this->db->query("
            SELECT COUNT(*) as count FROM users u
            WHERE u.company_id = ? AND u.role = 'customer' AND u.created_at BETWEEN ? AND ?
        ", [$this->auth->companyId(), $params[0], $params[1]])->single()['count'];

        // Returning customers
        $returningCustomers = $this->db->query("
            SELECT COUNT(DISTINCT b.customer_id) as count
            FROM bookings b
            WHERE {$whereClause} AND b.customer_id IS NOT NULL
            AND b.customer_id IN (
                SELECT customer_id FROM bookings
                WHERE customer_id IS NOT NULL AND created_at < ?
                GROUP BY customer_id HAVING COUNT(*) > 1
            )
        ", array_merge($params, [$params[0]]))->single()['count'];

        // Top customers by spending
        $topCustomers = $this->db->query("
            SELECT u.first_name, u.last_name, u.email,
                   COUNT(b.id) as total_bookings,
                   COALESCE(SUM(b.total_amount), 0) as total_spent,
                   MAX(b.created_at) as last_booking
            FROM users u
            LEFT JOIN bookings b ON u.id = b.customer_id AND {$whereClause} AND b.status = 'completed'
            WHERE u.company_id = ? AND u.role = 'customer'
            GROUP BY u.id, u.first_name, u.last_name, u.email
            ORDER BY total_spent DESC
            LIMIT 20
        ", array_merge($params, [$this->auth->companyId()]))->resultSet();

        // Customer acquisition over time
        $customerAcquisition = $this->db->query("
            SELECT DATE(u.created_at) as date, COUNT(*) as new_customers
            FROM users u
            WHERE u.company_id = ? AND u.role = 'customer' AND u.created_at BETWEEN ? AND ?
            GROUP BY DATE(u.created_at)
            ORDER BY date ASC
        ", [$this->auth->companyId(), $params[0], $params[1]])->resultSet();

        return [
            'type' => 'customers',
            'period' => ['from' => $dateFrom, 'to' => $dateTo],
            'new_customers' => $newCustomers,
            'returning_customers' => $returningCustomers,
            'top_customers' => $topCustomers,
            'customer_acquisition' => $customerAcquisition
        ];
    }

    /**
     * Export to CSV
     */
    private function exportCSV($reportData, $type) {
        $filename = "report_{$type}_" . date('Y-m-d_H-i-s') . '.csv';

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        $output = fopen('php://output', 'w');

        // Write headers based on report type
        switch ($type) {
            case 'bookings':
                fputcsv($output, ['Booking Number', 'Customer', 'Service', 'Staff', 'Branch', 'Date & Time', 'Status', 'Amount']);
                foreach ($reportData['all_bookings'] as $booking) {
                    fputcsv($output, [
                        $booking['booking_number'],
                        $booking['customer_id'] ? $booking['first_name'] . ' ' . $booking['last_name'] : 'Guest',
                        $booking['service_name'],
                        $booking['staff_first_name'] ? $booking['staff_first_name'] . ' ' . $booking['staff_last_name'] : 'Unassigned',
                        $booking['branch_name'],
                        formatDateTime($booking['start_at']),
                        ucfirst($booking['status']),
                        '$' . number_format($booking['total_amount'], 2)
                    ]);
                }
                break;

            case 'revenue':
                fputcsv($output, ['Service', 'Bookings', 'Revenue', 'Average Revenue']);
                foreach ($reportData['revenue_by_service'] as $service) {
                    fputcsv($output, [
                        $service['service_name'],
                        $service['bookings'],
                        '$' . number_format($service['revenue'], 2),
                        '$' . number_format($service['avg_revenue'], 2)
                    ]);
                }
                break;

            case 'services':
                fputcsv($output, ['Service', 'Price', 'Total Bookings', 'Completed', 'Cancelled', 'Revenue', 'Avg Revenue']);
                foreach ($reportData['services'] as $service) {
                    fputcsv($output, [
                        $service['name'],
                        '$' . number_format($service['price'], 2),
                        $service['total_bookings'],
                        $service['completed_bookings'],
                        $service['cancelled_bookings'],
                        '$' . number_format($service['total_revenue'], 2),
                        '$' . number_format($service['avg_revenue'], 2)
                    ]);
                }
                break;

            case 'staff':
                fputcsv($output, ['Staff Name', 'Total Bookings', 'Completed', 'Cancelled', 'No Shows', 'Revenue', 'Avg Booking']);
                foreach ($reportData['staff_performance'] as $member) {
                    fputcsv($output, [
                        $member['first_name'] . ' ' . $member['last_name'],
                        $member['total_bookings'],
                        $member['completed_bookings'],
                        $member['cancelled_bookings'],
                        $member['no_show_bookings'],
                        '$' . number_format($member['total_revenue'], 2),
                        '$' . number_format($member['avg_booking_value'], 2)
                    ]);
                }
                break;

            case 'customers':
                fputcsv($output, ['Customer Name', 'Email', 'Total Bookings', 'Total Spent', 'Last Booking']);
                foreach ($reportData['top_customers'] as $customer) {
                    fputcsv($output, [
                        $customer['first_name'] . ' ' . $customer['last_name'],
                        $customer['email'],
                        $customer['total_bookings'],
                        '$' . number_format($customer['total_spent'], 2),
                        $customer['last_booking'] ? formatDate($customer['last_booking']) : 'Never'
                    ]);
                }
                break;
        }

        fclose($output);
        exit;
    }

    /**
     * Export to PDF (placeholder)
     */
    private function exportPDF($reportData, $type) {
        // PDF export would require a library like TCPDF or DomPDF
        // For now, return error
        $this->json(['error' => 'PDF export not implemented yet'], 501);
    }
}
