<?php
/**
 * Admin Customer Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class CustomerController extends Controller {
    public function index() {
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();

        // Get filter parameters
        $status = $this->getQuery('status', 'active');
        $search = $this->getQuery('search');
        $sort = $this->getQuery('sort', 'name');
        $page = (int) $this->getQuery('page', 1);

        // Build where clause
        $where = ["u.company_id = ?", "u.role = 'customer'"];
        $params = [$companyId];

        if ($status === 'active') {
            $where[] = "u.status = 'active'";
        } elseif ($status === 'inactive') {
            $where[] = "u.status = 'inactive'";
        }

        if ($search) {
            $where[] = "(u.first_name LIKE ? OR u.last_name LIKE ? OR u.email LIKE ? OR u.phone LIKE ?)";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
            $params[] = "%{$search}%";
        }

        $whereClause = implode(' AND ', $where);

        // Build order clause
        $orderClause = "u.first_name ASC, u.last_name ASC";
        switch ($sort) {
            case 'name':
                $orderClause = "u.first_name ASC, u.last_name ASC";
                break;
            case 'email':
                $orderClause = "u.email ASC";
                break;
            case 'recent':
                $orderClause = "u.last_login DESC, u.created_at DESC";
                break;
            case 'bookings':
                $orderClause = "total_bookings DESC";
                break;
        }

        // Get customers with pagination
        $offset = ($page - 1) * ADMIN_ITEMS_PER_PAGE;

        $customers = $this->db->query("
            SELECT u.*,
                   COUNT(b.id) as total_bookings,
                   COUNT(CASE WHEN b.status = 'completed' THEN 1 END) as completed_bookings,
                   COALESCE(SUM(CASE WHEN b.status = 'completed' THEN b.total_amount END), 0) as total_spent,
                   MAX(b.created_at) as last_booking_date,
                   MAX(u.last_login) as last_login_date
            FROM users u
            LEFT JOIN bookings b ON u.id = b.customer_id
            WHERE {$whereClause}
            GROUP BY u.id
            ORDER BY {$orderClause}
            LIMIT ? OFFSET ?
        ", array_merge($params, [ADMIN_ITEMS_PER_PAGE, $offset]))->resultSet();

        // Get total count
        $totalCustomers = $this->db->query("
            SELECT COUNT(*) as total FROM users u WHERE {$whereClause}
        ", $params)->single()['total'];

        $this->adminView('customers/index', [
            'customers' => $customers,
            'filters' => [
                'status' => $status,
                'search' => $search,
                'sort' => $sort,
                'page' => $page
            ],
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalCustomers / ADMIN_ITEMS_PER_PAGE),
                'total_items' => $totalCustomers
            ],
            'pageTitle' => 'Customer Management'
        ]);
    }

    public function show($id) {
        $customer = $this->db->query("
            SELECT u.*,
                   COUNT(b.id) as total_bookings,
                   COUNT(CASE WHEN b.status = 'completed' THEN 1 END) as completed_bookings,
                   COUNT(CASE WHEN b.status = 'cancelled' THEN 1 END) as cancelled_bookings,
                   COALESCE(SUM(CASE WHEN b.status = 'completed' THEN b.total_amount END), 0) as total_spent,
                   COALESCE(AVG(CASE WHEN b.status = 'completed' THEN b.total_amount END), 0) as avg_booking_value,
                   MIN(CASE WHEN b.status = 'completed' THEN b.created_at END) as first_booking_date,
                   MAX(CASE WHEN b.status = 'completed' THEN b.created_at END) as last_booking_date
            FROM users u
            LEFT JOIN bookings b ON u.id = b.customer_id
            WHERE u.id = ? AND u.role = 'customer'
            GROUP BY u.id
        ")->bind(1, $id)->single();

        if (!$customer || !$this->canAccessCustomer($customer)) {
            $this->adminView('errors/404');
            return;
        }

        // Get recent bookings
        $recentBookings = $this->db->query("
            SELECT b.*, s.name as service_name, br.name as branch_name,
                   st.first_name as staff_first_name, st.last_name as staff_last_name
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN branches br ON b.branch_id = br.id
            LEFT JOIN users st ON b.staff_id = st.id
            WHERE b.customer_id = ?
            ORDER BY b.created_at DESC
            LIMIT 10
        ")->bind(1, $id)->resultSet();

        // Get favorite services
        $favoriteServices = $this->db->query("
            SELECT s.name, COUNT(b.id) as booking_count
            FROM services s
            JOIN bookings b ON s.id = b.service_id
            WHERE b.customer_id = ? AND b.status = 'completed'
            GROUP BY s.id, s.name
            ORDER BY booking_count DESC
            LIMIT 5
        ")->bind(1, $id)->resultSet();

        $this->adminView('customers/show', [
            'customer' => $customer,
            'recentBookings' => $recentBookings,
            'favoriteServices' => $favoriteServices,
            'pageTitle' => 'Customer Details - ' . $customer['first_name'] . ' ' . $customer['last_name']
        ]);
    }

    public function create() {
        $this->adminView('customers/create', [
            'pageTitle' => 'Add New Customer'
        ]);
    }

    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/admin/customers/create');
        }

        $data = [
            'company_id' => $this->auth->companyId(),
            'first_name' => $this->getPost('first_name'),
            'last_name' => $this->getPost('last_name'),
            'email' => $this->getPost('email'),
            'phone' => $this->getPost('phone'),
            'role' => 'customer',
            'status' => 'active'
        ];

        // Validate input
        $errors = $this->validateCustomerInput($data);

        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/customers/create');
        }

        try {
            // Check if email already exists
            $existingUser = $this->db->query("SELECT id FROM users WHERE email = ?")
                                   ->bind(1, $data['email'])
                                   ->single();

            if ($existingUser) {
                $_SESSION['form_errors'] = ['email' => 'Email address already exists'];
                $_SESSION['form_data'] = $_POST;
                $this->redirect(BASE_URL . '/admin/customers/create');
            }

            // Generate temporary password
            $tempPassword = $this->generateTempPassword();
            $data['password_hash'] = password_hash($tempPassword, PASSWORD_DEFAULT);

            // Create customer
            $customerId = $this->db->insert('users', $data);

            // Send welcome email
            $this->sendWelcomeEmail($data['email'], $data['first_name'], $tempPassword);

            $this->redirect(BASE_URL . '/admin/customers', 'Customer added successfully!');

        } catch (Exception $e) {
            error_log("Customer creation error: " . $e->getMessage());
            $_SESSION['form_errors'] = ['general' => 'Failed to create customer'];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/customers/create');
        }
    }

    public function edit($id) {
        $customer = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'customer'")
                           ->bind(1, $id)
                           ->single();

        if (!$customer || !$this->canAccessCustomer($customer)) {
            $this->adminView('errors/404');
            return;
        }

        $this->adminView('customers/edit', [
            'customer' => $customer,
            'pageTitle' => 'Edit Customer - ' . $customer['first_name'] . ' ' . $customer['last_name']
        ]);
    }

    public function update($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/admin/customers/' . $id . '/edit');
        }

        $customer = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'customer'")
                           ->bind(1, $id)
                           ->single();

        if (!$customer || !$this->canAccessCustomer($customer)) {
            $this->adminView('errors/404');
            return;
        }

        $data = [
            'first_name' => $this->getPost('first_name'),
            'last_name' => $this->getPost('last_name'),
            'email' => $this->getPost('email'),
            'phone' => $this->getPost('phone'),
            'status' => $this->getPost('status')
        ];

        // Validate input
        $errors = $this->validateCustomerInput($data, $id);

        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/customers/' . $id . '/edit');
        }

        try {
            // Check if email already exists (excluding current user)
            $existingUser = $this->db->query("SELECT id FROM users WHERE email = ? AND id != ?")
                                   ->bind(1, $data['email'])
                                   ->bind(2, $id)
                                   ->single();

            if ($existingUser) {
                $_SESSION['form_errors'] = ['email' => 'Email address already exists'];
                $_SESSION['form_data'] = $_POST;
                $this->redirect(BASE_URL . '/admin/customers/' . $id . '/edit');
            }

            // Update customer
            $this->db->update('users', $data, ['id' => $id]);

            $this->redirect(BASE_URL . '/admin/customers', 'Customer updated successfully!');

        } catch (Exception $e) {
            error_log("Customer update error: " . $e->getMessage());
            $_SESSION['form_errors'] = ['general' => 'Failed to update customer'];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/customers/' . $id . '/edit');
        }
    }

    public function destroy($id) {
        $customer = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'customer'")
                           ->bind(1, $id)
                           ->single();

        if (!$customer || !$this->canAccessCustomer($customer)) {
            $this->json(['error' => 'Customer not found'], 404);
        }

        // Check if customer has bookings
        $bookingCount = $this->db->query("SELECT COUNT(*) as count FROM bookings WHERE customer_id = ?")
                                ->bind(1, $id)
                                ->single()['count'];

        if ($bookingCount > 0) {
            // Soft delete - mark as inactive instead of deleting
            $this->db->update('users', ['status' => 'inactive'], ['id' => $id]);

            $this->json([
                'success' => true,
                'message' => 'Customer marked as inactive (has booking history)'
            ]);
        } else {
            // Hard delete - no booking history
            $this->db->delete('users', ['id' => $id]);

            $this->json([
                'success' => true,
                'message' => 'Customer deleted successfully'
            ]);
        }
    }

    public function toggleStatus($id) {
        $customer = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'customer'")
                           ->bind(1, $id)
                           ->single();

        if (!$customer || !$this->canAccessCustomer($customer)) {
            $this->json(['error' => 'Customer not found'], 404);
        }

        $newStatus = $customer['status'] === 'active' ? 'inactive' : 'active';

        try {
            $this->db->update('users', ['status' => $newStatus], ['id' => $id]);

            $this->json([
                'success' => true,
                'message' => 'Customer status updated successfully',
                'new_status' => $newStatus
            ]);

        } catch (Exception $e) {
            error_log("Customer status toggle error: " . $e->getMessage());
            $this->json(['error' => 'Failed to update customer status'], 500);
        }
    }

    public function bookings($id) {
        $customer = $this->db->query("SELECT * FROM users WHERE id = ? AND role = 'customer'")
                           ->bind(1, $id)
                           ->single();

        if (!$customer || !$this->canAccessCustomer($customer)) {
            $this->adminView('errors/404');
            return;
        }

        $page = (int) $this->getQuery('page', 1);
        $offset = ($page - 1) * ADMIN_ITEMS_PER_PAGE;

        // Get customer bookings
        $bookings = $this->db->query("
            SELECT b.*, s.name as service_name, br.name as branch_name,
                   st.first_name as staff_first_name, st.last_name as staff_last_name
            FROM bookings b
            LEFT JOIN services s ON b.service_id = s.id
            LEFT JOIN branches br ON b.branch_id = br.id
            LEFT JOIN users st ON b.staff_id = st.id
            WHERE b.customer_id = ?
            ORDER BY b.created_at DESC
            LIMIT ? OFFSET ?
        ")->bind(1, $id)->bind(2, ADMIN_ITEMS_PER_PAGE)->bind(3, $offset)->resultSet();

        // Get total count
        $totalBookings = $this->db->query("SELECT COUNT(*) as total FROM bookings WHERE customer_id = ?")
                                 ->bind(1, $id)
                                 ->single()['total'];

        $this->adminView('customers/bookings', [
            'customer' => $customer,
            'bookings' => $bookings,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalBookings / ADMIN_ITEMS_PER_PAGE),
                'total_items' => $totalBookings
            ],
            'pageTitle' => 'Customer Bookings - ' . $customer['first_name'] . ' ' . $customer['last_name']
        ]);
    }

    public function export() {
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();

        $customers = $this->db->query("
            SELECT u.first_name, u.last_name, u.email, u.phone, u.created_at,
                   COUNT(b.id) as total_bookings,
                   COALESCE(SUM(b.total_amount), 0) as total_spent
            FROM users u
            LEFT JOIN bookings b ON u.id = b.customer_id AND b.status = 'completed'
            WHERE u.company_id = ? AND u.role = 'customer'
            GROUP BY u.id
            ORDER BY u.first_name, u.last_name
        ")->bind(1, $companyId)->resultSet();

        $filename = 'customers_export_' . date('Y-m-d_H-i-s') . '.csv';

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        $output = fopen('php://output', 'w');

        // CSV headers
        fputcsv($output, ['First Name', 'Last Name', 'Email', 'Phone', 'Registration Date', 'Total Bookings', 'Total Spent']);

        // CSV data
        foreach ($customers as $customer) {
            fputcsv($output, [
                $customer['first_name'],
                $customer['last_name'],
                $customer['email'],
                $customer['phone'],
                formatDate($customer['created_at']),
                $customer['total_bookings'],
                '$' . number_format($customer['total_spent'], 2)
            ]);
        }

        fclose($output);
        exit;
    }

    /**
     * Validate customer input
     */
    private function validateCustomerInput($data, $excludeId = null) {
        $errors = [];

        // Required fields
        if (empty($data['first_name'])) {
            $errors['first_name'] = 'First name is required';
        }

        if (empty($data['last_name'])) {
            $errors['last_name'] = 'Last name is required';
        }

        if (empty($data['email'])) {
            $errors['email'] = 'Email is required';
        } elseif (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Invalid email format';
        }

        return $errors;
    }

    /**
     * Generate temporary password
     */
    private function generateTempPassword() {
        return bin2hex(random_bytes(4)); // 8 character password
    }

    /**
     * Check if user can access customer
     */
    private function canAccessCustomer($customer) {
        $companyId = $this->auth->companyId();

        return $companyId && $customer['company_id'] == $companyId;
    }

    /**
     * Send welcome email
     */
    private function sendWelcomeEmail($email, $firstName, $tempPassword) {
        $emailAdapter = new EmailAdapter();
        $result = $emailAdapter->send($email, 'Welcome to Luxury Beauty Spa', "
            Dear {$firstName},

            Welcome to Luxury Beauty Spa! Your account has been created.

            Your temporary password is: {$tempPassword}

            Please log in and change your password.

            Login URL: " . BASE_URL . "/login

            Best regards,
            Luxury Beauty Spa Team
        ");

        if (!$result['success']) {
            error_log("Failed to send welcome email to {$email}: " . $result['message']);
        }
    }
}
