<?php
/**
 * Admin Booking Controller
 * Salon/Spa/Beauty Parlor Booking System
 */

class BookingController extends Controller {
    public function index() {
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();

        // Get filter parameters
        $status = $this->getQuery('status');
        $branch = $this->getQuery('branch');
        $staff = $this->getQuery('staff');
        $dateFrom = $this->getQuery('date_from');
        $dateTo = $this->getQuery('date_to');
        $search = $this->getQuery('search');
        $page = (int) $this->getQuery('page', 1);

        // Build filters
        $filters = [];
        if ($status) $filters['status'] = $status;
        if ($branch) $filters['branch_id'] = $branch;
        if ($staff) $filters['staff_id'] = $staff;
        if ($dateFrom) $filters['date_from'] = $dateFrom;
        if ($dateTo) $filters['date_to'] = $dateTo;

        // Get bookings
        $bookingModel = new Booking();
        $bookings = $bookingModel->paginate($page, ADMIN_ITEMS_PER_PAGE, $filters);

        // Get branches for filter
        $branches = $this->db->query("
            SELECT id, name FROM branches
            WHERE company_id = ? AND status = 'active'
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        // Get staff for filter
        $staffMembers = $this->db->query("
            SELECT id, first_name, last_name FROM users
            WHERE company_id = ? AND role = 'staff' AND status = 'active'
            ORDER BY first_name, last_name ASC
        ")->bind(1, $companyId)->resultSet();

        $this->adminView('bookings/index', [
            'bookings' => $bookings,
            'branches' => $branches,
            'staffMembers' => $staffMembers,
            'filters' => array_merge($filters, [
                'search' => $search,
                'page' => $page
            ]),
            'pageTitle' => 'Bookings Management'
        ]);
    }

    public function show($id) {
        $bookingModel = new Booking();
        $booking = $bookingModel->findWithDetails($id);

        if (!$booking) {
            $this->adminView('errors/404');
            return;
        }

        // Check permissions
        if (!$this->canViewBooking($booking)) {
            $this->adminView('errors/403');
            return;
        }

        // Get booking history/notes
        $notes = $this->getBookingNotes($id);

        // Get available staff for reassignment
        $availableStaff = $this->getAvailableStaff($booking['branch_id'], $booking['start_at']);

        $this->adminView('bookings/show', [
            'booking' => $booking,
            'notes' => $notes,
            'availableStaff' => $availableStaff,
            'pageTitle' => 'Booking Details - ' . $booking['booking_number']
        ]);
    }

    public function create() {
        $companyId = $this->auth->companyId();

        // Get branches
        $branches = $this->db->query("
            SELECT id, name FROM branches
            WHERE company_id = ? AND status = 'active'
            ORDER BY is_main DESC, name ASC
        ")->bind(1, $companyId)->resultSet();

        // Get services
        $services = $this->db->query("
            SELECT id, name, price FROM services
            WHERE company_id = ? AND is_active = 1
            ORDER BY name ASC
        ")->bind(1, $companyId)->resultSet();

        // Get customers
        $customers = $this->db->query("
            SELECT id, first_name, last_name, email, phone FROM users
            WHERE company_id = ? AND role = 'customer'
            ORDER BY first_name, last_name ASC
        ")->bind(1, $companyId)->resultSet();

        $this->adminView('bookings/create', [
            'branches' => $branches,
            'services' => $services,
            'customers' => $customers,
            'pageTitle' => 'Create New Booking'
        ]);
    }

    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect(BASE_URL . '/admin/bookings/create');
        }

        $data = [
            'service_id' => $this->getPost('service_id'),
            'branch_id' => $this->getPost('branch_id'),
            'customer_id' => $this->getPost('customer_id') ?: null,
            'staff_id' => $this->getPost('staff_id') ?: null,
            'start_at' => $this->getPost('start_at'),
            'notes' => $this->getPost('notes'),
            'customer_notes' => $this->getPost('customer_notes')
        ];

        // Validate input
        $errors = $this->validateBookingInput($data);

        if (!empty($errors)) {
            $_SESSION['form_errors'] = $errors;
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/bookings/create');
        }

        try {
            $bookingModel = new Booking();
            $bookingId = $bookingModel->createBooking($data);

            // Send notifications
            $this->sendBookingNotifications($bookingId);

            $this->redirect(BASE_URL . '/admin/bookings/' . $bookingId, 'Booking created successfully!');

        } catch (Exception $e) {
            error_log("Admin booking creation error: " . $e->getMessage());
            $_SESSION['form_errors'] = ['general' => 'Failed to create booking: ' . $e->getMessage()];
            $_SESSION['form_data'] = $_POST;
            $this->redirect(BASE_URL . '/admin/bookings/create');
        }
    }

    public function updateStatus($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['error' => 'Method not allowed'], 405);
        }

        $status = $this->getPost('status');
        $notes = $this->getPost('notes');

        if (!$status) {
            $this->json(['error' => 'Status is required'], 400);
        }

        $bookingModel = new Booking();
        $booking = $bookingModel->findWithDetails($id);

        if (!$booking || !$this->canModifyBooking($booking)) {
            $this->json(['error' => 'Booking not found or access denied'], 404);
        }

        try {
            $bookingModel->updateStatus($id, $status, $this->auth->id());

            // Add status change note
            if ($notes) {
                $this->addBookingNote($id, "Status changed to: {$status}. Notes: {$notes}", $this->auth->id());
            }

            // Send notifications for status changes
            $this->sendStatusChangeNotifications($id, $status);

            $this->json([
                'success' => true,
                'message' => 'Booking status updated successfully'
            ]);

        } catch (Exception $e) {
            error_log("Booking status update error: " . $e->getMessage());
            $this->json(['error' => 'Failed to update booking status'], 500);
        }
    }

    public function reassignStaff($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['error' => 'Method not allowed'], 405);
        }

        $staffId = $this->getPost('staff_id');

        $bookingModel = new Booking();
        $booking = $bookingModel->find($id);

        if (!$booking || !$this->canModifyBooking($booking)) {
            $this->json(['error' => 'Booking not found or access denied'], 404);
        }

        try {
            // Check if new staff is available
            if ($staffId) {
                $availability = $bookingModel->checkAvailability(
                    $booking['service_id'],
                    $staffId,
                    $booking['start_at'],
                    $booking['end_at'],
                    $id // exclude current booking
                );

                if (!$availability['available']) {
                    $this->json(['error' => 'Selected staff is not available at this time'], 400);
                }
            }

            $bookingModel->update($id, ['staff_id' => $staffId]);

            // Add reassignment note
            $staffName = $staffId ? $this->getStaffName($staffId) : 'Unassigned';
            $this->addBookingNote($id, "Staff reassigned to: {$staffName}", $this->auth->id());

            $this->json([
                'success' => true,
                'message' => 'Staff reassigned successfully'
            ]);

        } catch (Exception $e) {
            error_log("Staff reassignment error: " . $e->getMessage());
            $this->json(['error' => 'Failed to reassign staff'], 500);
        }
    }

    public function export() {
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();

        // Get filter parameters
        $status = $this->getQuery('status');
        $branch = $this->getQuery('branch');
        $dateFrom = $this->getQuery('date_from');
        $dateTo = $this->getQuery('date_to');

        // Build filters
        $filters = [];
        if ($status) $filters['status'] = $status;
        if ($branch) $filters['branch_id'] = $branch;
        if ($dateFrom) $filters['date_from'] = $dateFrom;
        if ($dateTo) $filters['date_to'] = $dateTo;

        $bookingModel = new Booking();
        $bookings = $bookingModel->getByCompany($companyId, $filters);

        // Generate CSV
        $filename = 'bookings_export_' . date('Y-m-d_H-i-s') . '.csv';

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        $output = fopen('php://output', 'w');

        // CSV headers
        fputcsv($output, [
            'Booking Number',
            'Customer',
            'Service',
            'Staff',
            'Branch',
            'Date & Time',
            'Status',
            'Amount',
            'Created'
        ]);

        // CSV data
        foreach ($bookings as $booking) {
            fputcsv($output, [
                $booking['booking_number'],
                $booking['first_name'] ? $booking['first_name'] . ' ' . $booking['last_name'] : 'Guest',
                $booking['service_name'],
                $booking['staff_first_name'] ? $booking['staff_first_name'] . ' ' . $booking['staff_last_name'] : 'Unassigned',
                $booking['branch_name'],
                formatDateTime($booking['start_at']),
                ucfirst($booking['status']),
                '$' . number_format($booking['total_amount'], 2),
                formatDate($booking['created_at'])
            ]);
        }

        fclose($output);
        exit;
    }

    /**
     * Validate booking input
     */
    private function validateBookingInput($data) {
        $errors = [];

        // Validate service exists and belongs to company
        $service = $this->db->query("
            SELECT s.* FROM services s
            WHERE s.id = ? AND s.is_active = 1
        ")->bind(1, $data['service_id'])->single();

        if (!$service) {
            $errors['service_id'] = 'Invalid or inactive service';
        }

        // Validate branch exists and accessible
        $branch = $this->db->query("
            SELECT b.* FROM branches b
            WHERE b.id = ? AND b.status = 'active'
        ")->bind(1, $data['branch_id'])->single();

        if (!$branch || !$this->canAccessBranch($branch)) {
            $errors['branch_id'] = 'Invalid or inaccessible branch';
        }

        // Validate staff (if provided)
        if (!empty($data['staff_id'])) {
            $staff = $this->db->query("
                SELECT u.* FROM users u
                WHERE u.id = ? AND u.role = 'staff' AND u.status = 'active'
            ")->bind(1, $data['staff_id'])->single();

            if (!$staff || !$this->canAccessStaff($staff)) {
                $errors['staff_id'] = 'Invalid or inaccessible staff member';
            }
        }

        // Validate customer (if provided)
        if (!empty($data['customer_id'])) {
            $customer = $this->db->query("
                SELECT u.* FROM users u
                WHERE u.id = ? AND u.role = 'customer'
            ")->bind(1, $data['customer_id'])->single();

            if (!$customer || !$this->canAccessCustomer($customer)) {
                $errors['customer_id'] = 'Invalid or inaccessible customer';
            }
        }

        // Validate datetime
        if (empty($data['start_at']) || !strtotime($data['start_at'])) {
            $errors['start_at'] = 'Invalid date and time';
        }

        return $errors;
    }

    /**
     * Check if user can view booking
     */
    private function canViewBooking($booking) {
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();

        if ($branchId) {
            return $booking['branch_id'] == $branchId;
        } elseif ($companyId) {
            return $booking['company_id'] == $companyId;
        }

        return false;
    }

    /**
     * Check if user can modify booking
     */
    private function canModifyBooking($booking) {
        return $this->canViewBooking($booking);
    }

    /**
     * Check if user can access branch
     */
    private function canAccessBranch($branch) {
        $companyId = $this->auth->companyId();

        return $companyId ? $branch['company_id'] == $companyId : false;
    }

    /**
     * Check if user can access staff
     */
    private function canAccessStaff($staff) {
        $companyId = $this->auth->companyId();
        $branchId = $this->auth->branchId();

        if ($branchId) {
            return $staff['branch_id'] == $branchId;
        } elseif ($companyId) {
            return $staff['company_id'] == $companyId;
        }

        return false;
    }

    /**
     * Check if user can access customer
     */
    private function canAccessCustomer($customer) {
        $companyId = $this->auth->companyId();

        return $companyId ? $customer['company_id'] == $companyId : false;
    }

    /**
     * Get available staff for a time slot
     */
    private function getAvailableStaff($branchId, $startTime) {
        $bookingModel = new Booking();
        $service = $this->db->query("SELECT duration_minutes FROM services WHERE id = ?")
                           ->bind(1, $_POST['service_id'] ?? 0)
                           ->single();

        $duration = $service ? $service['duration_minutes'] : 60;
        $endTime = date('Y-m-d H:i:s', strtotime($startTime) + ($duration * 60));

        $staffMembers = $this->db->query("
            SELECT u.id, u.first_name, u.last_name
            FROM users u
            WHERE u.branch_id = ? AND u.role = 'staff' AND u.status = 'active'
            ORDER BY u.first_name, u.last_name
        ")->bind(1, $branchId)->resultSet();

        $availableStaff = [];
        foreach ($staffMembers as $staff) {
            $availability = $bookingModel->checkAvailability(
                $_POST['service_id'] ?? 0,
                $staff['id'],
                $startTime,
                $endTime
            );

            if ($availability['available']) {
                $availableStaff[] = $staff;
            }
        }

        return $availableStaff;
    }

    /**
     * Get booking notes
     */
    private function getBookingNotes($bookingId) {
        // For now, return empty array - would implement notes system
        return [];
    }

    /**
     * Add booking note
     */
    private function addBookingNote($bookingId, $note, $userId) {
        // For now, just log - would implement notes system
        error_log("Booking note added: {$note}");
    }

    /**
     * Get staff name
     */
    private function getStaffName($staffId) {
        $staff = $this->db->query("SELECT first_name, last_name FROM users WHERE id = ?")
                         ->bind(1, $staffId)
                         ->single();
        return $staff ? $staff['first_name'] . ' ' . $staff['last_name'] : 'Unknown';
    }

    /**
     * Send booking notifications
     */
    private function sendBookingNotifications($bookingId) {
        error_log("Admin booking notifications sent for booking ID: {$bookingId}");
    }

    /**
     * Send status change notifications
     */
    private function sendStatusChangeNotifications($bookingId, $status) {
        error_log("Status change notifications sent for booking ID: {$bookingId}, status: {$status}");
    }
}
