<?php
/**
 * Authentication Class
 * Salon/Spa/Beauty Parlor Booking System
 */

class Auth {
    private static $instance = null;
    private $db;
    private $user = null;

    private function __construct() {
        $this->db = Database::getInstance();
        $this->startSession();
        $this->loadUser();
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Start session
     */
    private function startSession() {
        if (session_status() === PHP_SESSION_NONE) {
            session_name(SESSION_NAME);
            session_start();

            // Regenerate session ID for security
            if (!isset($_SESSION['regenerated'])) {
                session_regenerate_id(true);
                $_SESSION['regenerated'] = true;
            }

            // Check session expiry
            if (isset($_SESSION['last_activity']) &&
                (time() - $_SESSION['last_activity'] > SESSION_LIFETIME)) {
                $this->logout();
            }
            $_SESSION['last_activity'] = time();
        }
    }

    /**
     * Load user from session
     */
    private function loadUser() {
        if (isset($_SESSION['user_id'])) {
            $userId = $_SESSION['user_id'];
            $user = $this->db->query("
                SELECT u.*, c.name as company_name, c.slug as company_slug,
                       b.name as branch_name, b.slug as branch_slug
                FROM users u
                LEFT JOIN companies c ON u.company_id = c.id
                LEFT JOIN branches b ON u.branch_id = b.id
                WHERE u.id = ? AND u.status = 'active'
            ")->bind(1, $userId)->single();

            if ($user) {
                // Get user roles and permissions
                $userRoles = $this->getUserRoles($userId);
                $userPermissions = $this->getUserPermissions($userId);

                $user['roles'] = array_column($userRoles, 'slug');
                $user['permissions'] = $userPermissions;

                $this->user = $user;
                $_SESSION['user'] = $user;
            } else {
                $this->logout();
            }
        }
    }

    /**
     * Get user roles
     */
    private function getUserRoles($userId) {
        return $this->db->query("
            SELECT r.*
            FROM roles r
            INNER JOIN user_roles ur ON r.id = ur.role_id
            WHERE ur.user_id = ?
        ")->bind(1, $userId)->resultSet();
    }

    /**
     * Get user permissions
     */
    private function getUserPermissions($userId) {
        $permissions = [];

        $roles = $this->getUserRoles($userId);

        foreach ($roles as $role) {
            if ($role['permissions']) {
                $rolePermissions = json_decode($role['permissions'], true);
                if (is_array($rolePermissions)) {
                    $permissions = array_merge($permissions, $rolePermissions);
                }
            }
        }

        return array_unique($permissions);
    }

    /**
     * Attempt login
     */
    public function attempt($email, $password) {
        $user = $this->db->query("
            SELECT u.*, c.name as company_name, c.slug as company_slug,
                   b.name as branch_name, b.slug as branch_slug
            FROM users u
            LEFT JOIN companies c ON u.company_id = c.id
            LEFT JOIN branches b ON u.branch_id = b.id
            WHERE u.email = ? AND u.status = 'active'
        ")->bind(1, $email)->single();

        if ($user && password_verify($password, $user['password_hash'])) {
            // Update last login
            $this->db->update('users', ['last_login' => date('Y-m-d H:i:s')], ['id' => $user['id']]);

            // Set session
            $_SESSION['user_id'] = $user['id'];
            $this->loadUser();

            return true;
        }

        return false;
    }

    /**
     * Register new user
     */
    public function register($data) {
        // Check if email already exists
        $existing = $this->db->query("SELECT id FROM users WHERE email = ?")
                            ->bind(1, $data['email'])
                            ->single();

        if ($existing) {
            return false;
        }

        // Hash password
        $data['password_hash'] = password_hash($data['password'], PASSWORD_DEFAULT);
        unset($data['password']);

        // Create user
        $userId = $this->db->insert('users', $data);

        // Assign customer role
        $customerRole = $this->db->query("SELECT id FROM roles WHERE slug = 'customer' LIMIT 1")
                                ->single();

        if ($customerRole) {
            $this->db->insert('user_roles', [
                'user_id' => $userId,
                'role_id' => $customerRole['id'],
                'assigned_by' => $userId
            ]);
        }

        return $userId;
    }

    /**
     * Logout user
     */
    public function logout() {
        // Clear session
        session_unset();
        session_destroy();

        // Start new session to prevent session fixation
        $this->startSession();
    }

    /**
     * Get current user
     */
    public function user() {
        return $this->user;
    }

    /**
     * Check if user is logged in
     */
    public function check() {
        return $this->user !== null;
    }

    /**
     * Check if user has role
     */
    public function hasRole($role) {
        return $this->user && in_array($role, $this->user['roles']);
    }

    /**
     * Check if user has permission
     */
    public function hasPermission($permission) {
        return $this->user && in_array($permission, $this->user['permissions']);
    }

    /**
     * Check if user is admin
     */
    public function isAdmin() {
        return $this->hasRole('super_admin') ||
               $this->hasRole('company_admin') ||
               $this->hasRole('manager');
    }

    /**
     * Check if user is staff
     */
    public function isStaff() {
        return $this->hasRole('staff');
    }

    /**
     * Check if user is customer
     */
    public function isCustomer() {
        return $this->hasRole('customer');
    }

    /**
     * Get user ID
     */
    public function id() {
        return $this->user ? $this->user['id'] : null;
    }

    /**
     * Get user company ID
     */
    public function companyId() {
        return $this->user ? $this->user['company_id'] : null;
    }

    /**
     * Get user branch ID
     */
    public function branchId() {
        return $this->user ? $this->user['branch_id'] : null;
    }

    /**
     * Update user session data
     */
    public function refreshUser() {
        if ($this->user) {
            $this->loadUser();
        }
    }

    /**
     * Generate password reset token
     */
    public function generatePasswordResetToken($email) {
        $user = $this->db->query("SELECT id FROM users WHERE email = ?")
                        ->bind(1, $email)
                        ->single();

        if (!$user) {
            return false;
        }

        $token = bin2hex(random_bytes(32));
        $expires = date('Y-m-d H:i:s', strtotime('+1 hour'));

        // Store token (you might want to create a password_resets table)
        $_SESSION['password_reset'] = [
            'user_id' => $user['id'],
            'token' => $token,
            'expires' => $expires
        ];

        return $token;
    }

    /**
     * Reset password with token
     */
    public function resetPassword($token, $newPassword) {
        if (!isset($_SESSION['password_reset']) ||
            $_SESSION['password_reset']['token'] !== $token ||
            strtotime($_SESSION['password_reset']['expires']) < time()) {
            return false;
        }

        $userId = $_SESSION['password_reset']['user_id'];
        $passwordHash = password_hash($newPassword, PASSWORD_DEFAULT);

        $result = $this->db->update('users', ['password_hash' => $passwordHash], ['id' => $userId]);

        if ($result) {
            unset($_SESSION['password_reset']);
        }

        return $result;
    }
}
