<?php
/**
 * Automated Reminders Script
 * Salon/Spa/Beauty Parlor Booking System
 *
 * This script should be run via CRON job every hour
 * Example CRON: 0 * * * * php /path/to/salon-system/scripts/reminders.php
 */

// Include configuration
require_once __DIR__ . '/../config/config.php';

echo "Starting reminders script at " . date('Y-m-d H:i:s') . "\n";

try {
    // Initialize database
    $db = Database::getInstance();

    // Get settings for reminder timing
    $reminderSettings = $db->query("
        SELECT setting_key, setting_value
        FROM settings
        WHERE setting_key IN ('booking_reminder_24h', 'booking_reminder_2h', 'booking_reminder_enabled')
    ")->resultSet();

    $settings = [];
    foreach ($reminderSettings as $setting) {
        $settings[$setting['setting_key']] = $setting['setting_value'];
    }

    // Check if reminders are enabled
    if (isset($settings['booking_reminder_enabled']) && $settings['booking_reminder_enabled'] === 'false') {
        echo "Reminders are disabled\n";
        exit(0);
    }

    // Default reminder times (hours before appointment)
    $reminder24h = isset($settings['booking_reminder_24h']) ? (int) $settings['booking_reminder_24h'] : 24;
    $reminder2h = isset($settings['booking_reminder_2h']) ? (int) $settings['booking_reminder_2h'] : 2;

    echo "Sending {$reminder24h}h and {$reminder2h}h reminders\n";

    // Find bookings that need reminders
    $currentTime = date('Y-m-d H:i:s');

    // 24-hour reminders
    $reminder24hTime = date('Y-m-d H:i:s', strtotime("+{$reminder24h} hours"));
    $bookings24h = $db->query("
        SELECT b.*, s.name as service_name, s.duration_minutes,
               u.first_name, u.last_name, u.email, u.phone,
               br.name as branch_name, br.phone as branch_phone, br.email as branch_email,
               st.first_name as staff_first_name, st.last_name as staff_last_name
        FROM bookings b
        LEFT JOIN services s ON b.service_id = s.id
        LEFT JOIN users u ON b.customer_id = u.id
        LEFT JOIN branches br ON b.branch_id = br.id
        LEFT JOIN users st ON b.staff_id = st.id
        WHERE b.status = 'confirmed'
        AND b.start_at BETWEEN ? AND DATE_ADD(?, INTERVAL 1 HOUR)
        AND NOT EXISTS (
            SELECT 1 FROM notifications n
            WHERE n.related_booking_id = b.id
            AND n.type = 'email'
            AND n.subject LIKE '%reminder%'
            AND DATE(n.created_at) = CURDATE()
        )
    ", [$currentTime, $reminder24hTime])->resultSet();

    // 2-hour reminders
    $reminder2hTime = date('Y-m-d H:i:s', strtotime("+{$reminder2h} hours"));
    $bookings2h = $db->query("
        SELECT b.*, s.name as service_name, s.duration_minutes,
               u.first_name, u.last_name, u.email, u.phone,
               br.name as branch_name, br.phone as branch_phone, br.email as branch_email,
               st.first_name as staff_first_name, st.last_name as staff_last_name
        FROM bookings b
        LEFT JOIN services s ON b.service_id = s.id
        LEFT JOIN users u ON b.customer_id = u.id
        LEFT JOIN branches br ON b.branch_id = br.id
        LEFT JOIN users st ON b.staff_id = st.id
        WHERE b.status = 'confirmed'
        AND b.start_at BETWEEN ? AND DATE_ADD(?, INTERVAL 1 HOUR)
        AND NOT EXISTS (
            SELECT 1 FROM notifications n
            WHERE n.related_booking_id = b.id
            AND n.type = 'sms'
            AND n.subject LIKE '%reminder%'
            AND DATE(n.created_at) = CURDATE()
        )
    ", [$currentTime, $reminder2hTime])->resultSet();

    // Send 24-hour reminders (Email)
    foreach ($bookings24h as $booking) {
        sendReminderNotification($booking, '24h');
    }

    // Send 2-hour reminders (SMS/WhatsApp)
    foreach ($bookings2h as $booking) {
        sendReminderNotification($booking, '2h');
    }

    echo "Processed " . count($bookings24h) . " 24h reminders and " . count($bookings2h) . " 2h reminders\n";

} catch (Exception $e) {
    error_log("Reminders script error: " . $e->getMessage());
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}

echo "Reminders script completed at " . date('Y-m-d H:i:s') . "\n";

/**
 * Send reminder notification
 */
function sendReminderNotification($booking, $type) {
    global $db;

    // Prepare template data
    $templateData = [
        'customer_name' => $booking['customer_id'] ? $booking['first_name'] . ' ' . $booking['last_name'] : 'Valued Guest',
        'booking_number' => $booking['booking_number'],
        'service_name' => $booking['service_name'],
        'booking_date' => date('l, F j, Y', strtotime($booking['start_at'])),
        'booking_time' => date('g:i A', strtotime($booking['start_at'])),
        'staff_name' => $booking['staff_first_name'] ? $booking['staff_first_name'] . ' ' . $booking['staff_last_name'] : 'Our Expert',
        'branch_name' => $booking['branch_name'],
        'branch_phone' => $booking['branch_phone'],
        'branch_email' => $booking['branch_email'],
        'duration_minutes' => $booking['duration_minutes']
    ];

    if ($type === '24h') {
        // Send email reminder
        $emailAdapter = new EmailAdapter();
        $result = $emailAdapter->send(
            $booking['email'] ?: 'guest@example.com',
            'Appointment Reminder - ' . $booking['booking_number'],
            '', // Empty message - will use template
            'booking_reminder',
            $templateData
        );

        $notificationType = 'email';

    } else { // 2h
        // Send SMS/WhatsApp reminder
        $smsAdapter = new SMSAdapter();
        $result = $smsAdapter->send(
            $booking['phone'] ?: '',
            '', // Empty message - will use template
            'booking_reminder',
            $templateData
        );

        $notificationType = 'sms';

        // Also try WhatsApp if available
        if (!empty($booking['phone'])) {
            $whatsappAdapter = new WhatsAppAdapter();
            $whatsappResult = $whatsappAdapter->send(
                $booking['phone'],
                '', // Empty message - will use template
                'booking_reminder',
                $templateData
            );
        }
    }

    // Log notification
    $db->insert('notifications', [
        'company_id' => $booking['company_id'],
        'recipient_id' => $booking['customer_id'],
        'recipient_email' => $booking['email'],
        'recipient_phone' => $booking['phone'],
        'type' => $notificationType,
        'subject' => "Appointment Reminder - {$booking['booking_number']}",
        'content' => "Reminder sent for booking {$booking['booking_number']}",
        'status' => $result['success'] ? 'sent' : 'failed',
        'related_booking_id' => $booking['id']
    ]);

    echo "Sent {$type} reminder for booking {$booking['booking_number']}: " . ($result['success'] ? 'SUCCESS' : 'FAILED') . "\n";
}
