<?php
/**
 * Email Notification Adapter
 * Salon/Spa/Beauty Parlor Booking System
 */

class EmailAdapter {
    private $smtpHost;
    private $smtpPort;
    private $smtpUser;
    private $smtpPass;
    private $smtpEncryption;
    private $fromEmail;
    private $fromName;

    public function __construct() {
        $this->smtpHost = SMTP_HOST ?: null;
        $this->smtpPort = SMTP_PORT ?: 587;
        $this->smtpUser = SMTP_USER ?: null;
        $this->smtpPass = SMTP_PASS ?: null;
        $this->smtpEncryption = SMTP_ENCRYPTION ?: 'tls';
        $this->fromEmail = FROM_EMAIL ?: 'noreply@salon.com';
        $this->fromName = FROM_NAME ?: 'Salon System';
    }

    /**
     * Send email notification
     */
    public function send($to, $subject, $message, $templateId = null, $templateData = []) {
        try {
            // Check if SMTP is configured
            if (!$this->smtpHost || !$this->smtpUser) {
                // Log the email instead of sending
                error_log("Email not sent (SMTP not configured): To: {$to}, Subject: {$subject}");
                return [
                    'success' => true,
                    'message' => 'Email logged (SMTP not configured)',
                    'message_id' => 'logged_' . time()
                ];
            }

            // Load template if template ID is provided
            if ($templateId) {
                $template = $this->loadTemplate($templateId);
                if ($template) {
                    $subject = $this->replacePlaceholders($template['subject'], $templateData);
                    $message = $this->replacePlaceholders($template['content'], $templateData);
                }
            }

            // Send email using SMTP
            $result = $this->sendSMTPEmail($to, $subject, $message);

            return [
                'success' => $result,
                'message' => $result ? 'Email sent successfully' : 'Failed to send email',
                'message_id' => $result ? 'smtp_' . time() : null
            ];

        } catch (Exception $e) {
            error_log("Email sending error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Email sending failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Send bulk emails
     */
    public function sendBulk($recipients, $subject, $message, $templateId = null, $templateData = []) {
        $results = [];

        foreach ($recipients as $recipient) {
            if (is_array($recipient)) {
                $email = $recipient['email'];
                $data = array_merge($templateData, $recipient);
            } else {
                $email = $recipient;
                $data = $templateData;
            }

            $results[] = [
                'email' => $email,
                'result' => $this->send($email, $subject, $message, $templateId, $data)
            ];
        }

        return $results;
    }

    /**
     * Load email template
     */
    private function loadTemplate($templateId) {
        // This would load template from database
        // For now, return null to indicate template not found
        return null;
    }

    /**
     * Replace placeholders in template
     */
    private function replacePlaceholders($content, $data) {
        foreach ($data as $key => $value) {
            $content = str_replace("{{{$key}}}", $value, $content);
        }
        return $content;
    }

    /**
     * Send email using SMTP (simplified implementation)
     */
    private function sendSMTPEmail($to, $subject, $message) {
        // This is a simplified SMTP implementation
        // In production, you would use a proper SMTP library like PHPMailer or SwiftMailer

        $headers = [
            'MIME-Version: 1.0',
            'Content-type: text/html; charset=UTF-8',
            'From: ' . $this->fromName . ' <' . $this->fromEmail . '>',
            'Reply-To: ' . $this->fromEmail,
            'X-Mailer: PHP/' . phpversion()
        ];

        // Basic email validation
        if (!filter_var($to, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // For development/testing, log the email instead of sending
        if (IS_DEVELOPMENT) {
            error_log("EMAIL TO: {$to}");
            error_log("SUBJECT: {$subject}");
            error_log("MESSAGE: " . substr($message, 0, 200) . "...");
            return true;
        }

        // Attempt to send email using PHP's mail function
        // Note: This requires proper mail server configuration
        return mail($to, $subject, $message, implode("\r\n", $headers));
    }

    /**
     * Get email templates for booking confirmations
     */
    public static function getBookingConfirmationTemplate() {
        return [
            'subject' => 'Your Booking Confirmation - {{booking_number}}',
            'content' => '
                <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
                    <h2 style="color: #C9A86A;">Booking Confirmed!</h2>
                    <p>Dear {{customer_name}},</p>
                    <p>Your booking has been confirmed with the following details:</p>

                    <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
                        <h3>Booking Details</h3>
                        <p><strong>Booking Number:</strong> {{booking_number}}</p>
                        <p><strong>Service:</strong> {{service_name}}</p>
                        <p><strong>Date & Time:</strong> {{booking_date}} at {{booking_time}}</p>
                        <p><strong>Staff:</strong> {{staff_name}}</p>
                        <p><strong>Branch:</strong> {{branch_name}}</p>
                        <p><strong>Total Amount:</strong> ${{total_amount}}</p>
                    </div>

                    <p>If you need to make any changes or have questions, please contact us.</p>

                    <p>Thank you for choosing Luxury Beauty Spa!</p>

                    <hr style="margin: 30px 0; border: none; border-top: 1px solid #eee;">
                    <p style="color: #666; font-size: 12px;">
                        This is an automated message. Please do not reply to this email.
                    </p>
                </div>
            '
        ];
    }

    /**
     * Get email templates for booking reminders
     */
    public static function getBookingReminderTemplate() {
        return [
            'subject' => 'Appointment Reminder - {{booking_number}}',
            'content' => '
                <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
                    <h2 style="color: #C9A86A;">Appointment Reminder</h2>
                    <p>Dear {{customer_name}},</p>
                    <p>This is a friendly reminder of your upcoming appointment:</p>

                    <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0;">
                        <h3>Your Appointment</h3>
                        <p><strong>Service:</strong> {{service_name}}</p>
                        <p><strong>Date & Time:</strong> {{booking_date}} at {{booking_time}}</p>
                        <p><strong>Staff:</strong> {{staff_name}}</p>
                        <p><strong>Branch:</strong> {{branch_name}}</p>
                        <p><strong>Booking Number:</strong> {{booking_number}}</p>
                    </div>

                    <p>We look forward to seeing you!</p>

                    <p>Luxury Beauty Spa Team</p>
                </div>
            '
        ];
    }
}
