<?php
/**
 * Helper Functions
 * Salon/Spa/Beauty Parlor Booking System
 */

/**
 * Get environment variable with fallback
 */
function env($key, $default = null) {
    return getenv($key) ?: $default;
}

/**
 * Sanitize input data
 */
function sanitize($data) {
    if (is_array($data)) {
        return array_map('sanitize', $data);
    }
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

/**
 * Generate CSRF token
 */
function generateCsrfToken() {
    if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
        $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
    }
    return $_SESSION[CSRF_TOKEN_NAME];
}

/**
 * Verify CSRF token
 */
function verifyCsrfToken($token) {
    return isset($_SESSION[CSRF_TOKEN_NAME]) && hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
}

/**
 * Generate random string
 */
function generateRandomString($length = 10) {
    return bin2hex(random_bytes($length / 2));
}

/**
 * Format currency
 */
function formatCurrency($amount, $currency = 'USD') {
    $symbols = [
        'USD' => '$',
        'EUR' => '€',
        'GBP' => '£',
        'KES' => 'KSh'
    ];

    $symbol = $symbols[$currency] ?? $currency;
    return $symbol . number_format($amount, 2);
}

/**
 * Format date
 */
function formatDate($date, $format = 'M j, Y') {
    return date($format, strtotime($date));
}

/**
 * Format time
 */
function formatTime($time, $format = 'g:i A') {
    return date($format, strtotime($time));
}

/**
 * Format datetime
 */
function formatDateTime($datetime, $dateFormat = 'M j, Y', $timeFormat = 'g:i A') {
    $timestamp = strtotime($datetime);
    return date($dateFormat, $timestamp) . ' at ' . date($timeFormat, $timestamp);
}

/**
 * Generate booking number
 */
function generateBookingNumber() {
    return 'BK-' . date('Y') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
}

/**
 * Generate order number
 */
function generateOrderNumber() {
    return 'ORD-' . date('Y') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
}

/**
 * Generate ticket number
 */
function generateTicketNumber() {
    return 'TKT-' . date('Y') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
}

/**
 * Get user avatar URL
 */
function getAvatarUrl($avatar = null, $size = 100) {
    if ($avatar && file_exists(PUBLIC_PATH . $avatar)) {
        return BASE_URL . $avatar;
    }
    return "https://ui-avatars.com/api/?name=User&size={$size}&background=C9A86A&color=fff";
}

/**
 * Get service image URL
 */
function getServiceImageUrl($images, $index = 0) {
    if (is_array($images) && isset($images[$index])) {
        $image = $images[$index];
        if (file_exists(PUBLIC_PATH . $image)) {
            return BASE_URL . $image;
        }
    }
    return ASSETS_URL . '/images/placeholder-service.jpg';
}

/**
 * Get product image URL
 */
function getProductImageUrl($images, $index = 0) {
    if (is_array($images) && isset($images[$index])) {
        $image = $images[$index];
        if (file_exists(PUBLIC_PATH . $image)) {
            return BASE_URL . $image;
        }
    }
    return ASSETS_URL . '/images/placeholder-product.jpg';
}

/**
 * Truncate text
 */
function truncateText($text, $length = 100, $suffix = '...') {
    if (strlen($text) <= $length) {
        return $text;
    }
    return substr($text, 0, $length - strlen($suffix)) . $suffix;
}

/**
 * Check if user has permission
 */
function hasPermission($permission, $userPermissions = null) {
    if (!$userPermissions) {
        return false;
    }

    // Super admin has all permissions
    if (in_array('*', $userPermissions)) {
        return true;
    }

    return in_array($permission, $userPermissions);
}

/**
 * Check if user has role
 */
function hasRole($role, $userRoles = null) {
    if (!$userRoles) {
        return false;
    }

    return in_array($role, $userRoles);
}

/**
 * Get current user
 */
function currentUser() {
    return $_SESSION['user'] ?? null;
}

/**
 * Get current company
 */
function currentCompany() {
    return $_SESSION['company'] ?? null;
}

/**
 * Get current branch
 */
function currentBranch() {
    return $_SESSION['branch'] ?? null;
}

/**
 * Redirect with message
 */
function redirectWithMessage($url, $message, $type = 'success') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
    header("Location: $url");
    exit;
}

/**
 * Get flash message
 */
function getFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'] ?? 'success';
        unset($_SESSION['flash_message'], $_SESSION['flash_type']);
        return ['message' => $message, 'type' => $type];
    }
    return null;
}

/**
 * Validate email
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Validate phone number (basic)
 */
function isValidPhone($phone) {
    return preg_match('/^\+?[\d\s\-\(\)]+$/', $phone);
}

/**
 * Generate slug from string
 */
function generateSlug($string) {
    $string = strtolower($string);
    $string = preg_replace('/[^a-z0-9\s-]/', '', $string);
    $string = preg_replace('/[\s-]+/', '-', $string);
    return trim($string, '-');
}

/**
 * Get file extension
 */
function getFileExtension($filename) {
    return strtolower(pathinfo($filename, PATHINFO_EXTENSION));
}

/**
 * Check if file type is allowed
 */
function isAllowedFileType($mimeType, $allowedTypes) {
    return in_array($mimeType, $allowedTypes);
}

/**
 * Calculate distance between two coordinates (Haversine formula)
 */
function calculateDistance($lat1, $lon1, $lat2, $lon2) {
    $earthRadius = 6371; // km

    $latDelta = deg2rad($lat2 - $lat1);
    $lonDelta = deg2rad($lon2 - $lon1);

    $a = sin($latDelta / 2) * sin($latDelta / 2) +
         cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
         sin($lonDelta / 2) * sin($lonDelta / 2);

    $c = 2 * atan2(sqrt($a), sqrt(1 - $a));

    return $earthRadius * $c;
}

/**
 * Get user's IP address
 */
function getUserIP() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        return $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        return $_SERVER['HTTP_X_FORWARDED_FOR'];
    } else {
        return $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    }
}

/**
 * Get user agent
 */
function getUserAgent() {
    return $_SERVER['HTTP_USER_AGENT'] ?? '';
}

/**
 * Check if request is AJAX
 */
function isAjaxRequest() {
    return isset($_SERVER['HTTP_X_REQUESTED_WITH']) &&
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

/**
 * Send JSON response
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/**
 * Get request method
 */
function getRequestMethod() {
    return $_SERVER['REQUEST_METHOD'];
}

/**
 * Get request URI
 */
function getRequestUri() {
    $uri = $_SERVER['REQUEST_URI'];
    // Remove query string
    $uri = explode('?', $uri)[0];
    return $uri;
}

/**
 * Get query parameter
 */
function getQueryParam($key, $default = null) {
    return $_GET[$key] ?? $default;
}

/**
 * Get post parameter
 */
function getPostParam($key, $default = null) {
    return $_POST[$key] ?? $default;
}

/**
 * Check if user is logged in
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Require login
 */
function requireLogin() {
    if (!isLoggedIn()) {
        redirectWithMessage(BASE_URL . '/login', 'Please log in to continue.', 'warning');
    }
}

/**
 * Require role
 */
function requireRole($role) {
    requireLogin();
    $user = currentUser();
    if (!$user || !hasRole($role, $user['roles'])) {
        redirectWithMessage(BASE_URL . '/dashboard', 'Access denied.', 'danger');
    }
}

/**
 * Require permission
 */
function requirePermission($permission) {
    requireLogin();
    $user = currentUser();
    if (!$user || !hasPermission($permission, $user['permissions'])) {
        redirectWithMessage(BASE_URL . '/dashboard', 'Access denied.', 'danger');
    }
}
