/**
 * Main JavaScript Application
 * Salon/Spa/Beauty Parlor Booking System
 */

// DOM Content Loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeApp();
});

function initializeApp() {
    // Initialize components
    initializeMobileMenu();
    initializeFlashMessages();
    initializeFormValidation();
    initializeImageLazyLoading();
    initializeSmoothScrolling();
    initializeModalSystem();

    // Initialize page-specific features
    if (document.querySelector('.home-page')) {
        initializeHomePage();
    }

    if (document.querySelector('.booking-page')) {
        initializeBookingPage();
    }

    if (document.querySelector('.service-page')) {
        initializeServicePage();
    }

    // Initialize PWA features if supported
    if ('serviceWorker' in navigator) {
        initializePWA();
    }
}

// ===========================================
// MOBILE MENU
// ===========================================

function initializeMobileMenu() {
    const mobileMenuButton = document.getElementById('mobile-menu-button');
    const mobileMenu = document.getElementById('mobile-menu');

    if (mobileMenuButton && mobileMenu) {
        mobileMenuButton.addEventListener('click', function() {
            mobileMenu.classList.toggle('hidden');
            mobileMenu.classList.toggle('open');
        });

        // Close mobile menu when clicking outside
        document.addEventListener('click', function(event) {
            if (!mobileMenuButton.contains(event.target) && !mobileMenu.contains(event.target)) {
                mobileMenu.classList.add('hidden');
                mobileMenu.classList.remove('open');
            }
        });
    }
}

// ===========================================
// FLASH MESSAGES
// ===========================================

function initializeFlashMessages() {
    const flashMessages = document.querySelectorAll('.flash-message');

    flashMessages.forEach(function(message) {
        // Auto-hide after 5 seconds
        setTimeout(function() {
            message.style.opacity = '0';
            setTimeout(function() {
                message.remove();
            }, 300);
        }, 5000);

        // Allow manual dismissal
        const closeButton = message.querySelector('.flash-close');
        if (closeButton) {
            closeButton.addEventListener('click', function() {
                message.style.opacity = '0';
                setTimeout(function() {
                    message.remove();
                }, 300);
            });
        }
    });
}

// ===========================================
// FORM VALIDATION
// ===========================================

function initializeFormValidation() {
    const forms = document.querySelectorAll('form[data-validate]');

    forms.forEach(function(form) {
        form.addEventListener('submit', function(event) {
            if (!validateForm(form)) {
                event.preventDefault();
            }
        });
    });
}

function validateForm(form) {
    let isValid = true;
    const requiredFields = form.querySelectorAll('[required]');
    const emailFields = form.querySelectorAll('input[type="email"]');
    const phoneFields = form.querySelectorAll('input[type="tel"]');

    // Clear previous errors
    form.querySelectorAll('.field-error').forEach(function(error) {
        error.remove();
    });
    form.querySelectorAll('.field-invalid').forEach(function(field) {
        field.classList.remove('field-invalid');
    });

    // Validate required fields
    requiredFields.forEach(function(field) {
        if (!field.value.trim()) {
            showFieldError(field, 'This field is required');
            isValid = false;
        }
    });

    // Validate email fields
    emailFields.forEach(function(field) {
        if (field.value && !isValidEmail(field.value)) {
            showFieldError(field, 'Please enter a valid email address');
            isValid = false;
        }
    });

    // Validate phone fields
    phoneFields.forEach(function(field) {
        if (field.value && !isValidPhone(field.value)) {
            showFieldError(field, 'Please enter a valid phone number');
            isValid = false;
        }
    });

    return isValid;
}

function showFieldError(field, message) {
    field.classList.add('field-invalid');

    const errorElement = document.createElement('div');
    errorElement.className = 'field-error text-red-600 text-sm mt-1';
    errorElement.textContent = message;

    field.parentNode.insertBefore(errorElement, field.nextSibling);
}

function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

function isValidPhone(phone) {
    const phoneRegex = /^[\+]?[1-9][\d]{0,15}$/;
    return phoneRegex.test(phone.replace(/[\s\-\(\)]/g, ''));
}

// ===========================================
// IMAGE LAZY LOADING
// ===========================================

function initializeImageLazyLoading() {
    const images = document.querySelectorAll('img[data-src]');

    if ('IntersectionObserver' in window) {
        const imageObserver = new IntersectionObserver(function(entries) {
            entries.forEach(function(entry) {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    img.src = img.dataset.src;
                    img.classList.remove('lazy');
                    imageObserver.unobserve(img);
                }
            });
        });

        images.forEach(function(img) {
            imageObserver.observe(img);
        });
    } else {
        // Fallback for browsers without IntersectionObserver
        images.forEach(function(img) {
            img.src = img.dataset.src;
        });
    }
}

// ===========================================
// SMOOTH SCROLLING
// ===========================================

function initializeSmoothScrolling() {
    const links = document.querySelectorAll('a[href^="#"]');

    links.forEach(function(link) {
        link.addEventListener('click', function(event) {
            const targetId = this.getAttribute('href').substring(1);
            const targetElement = document.getElementById(targetId);

            if (targetElement) {
                event.preventDefault();
                targetElement.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
}

// ===========================================
// MODAL SYSTEM
// ===========================================

function initializeModalSystem() {
    // Modal triggers
    const modalTriggers = document.querySelectorAll('[data-modal]');

    modalTriggers.forEach(function(trigger) {
        trigger.addEventListener('click', function(event) {
            event.preventDefault();
            const modalId = this.dataset.modal;
            showModal(modalId);
        });
    });

    // Modal close buttons
    const modalCloseButtons = document.querySelectorAll('[data-modal-close]');

    modalCloseButtons.forEach(function(button) {
        button.addEventListener('click', function() {
            const modal = this.closest('.modal');
            hideModal(modal);
        });
    });

    // Close modal on backdrop click
    const modals = document.querySelectorAll('.modal');

    modals.forEach(function(modal) {
        modal.addEventListener('click', function(event) {
            if (event.target === modal) {
                hideModal(modal);
            }
        });
    });

    // Close modal on ESC key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            const openModal = document.querySelector('.modal.show');
            if (openModal) {
                hideModal(openModal);
            }
        }
    });
}

function showModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.add('show');
        document.body.style.overflow = 'hidden';
    }
}

function hideModal(modal) {
    modal.classList.remove('show');
    document.body.style.overflow = '';
}

// ===========================================
// HOME PAGE FEATURES
// ===========================================

function initializeHomePage() {
    // Hero slider (if implemented)
    initializeHeroSlider();

    // Service card hover effects
    initializeServiceCards();

    // Testimonial carousel
    initializeTestimonials();
}

function initializeHeroSlider() {
    // Simple hero slider implementation
    const slides = document.querySelectorAll('.hero-slide');
    let currentSlide = 0;

    if (slides.length > 1) {
        setInterval(function() {
            slides[currentSlide].classList.remove('active');
            currentSlide = (currentSlide + 1) % slides.length;
            slides[currentSlide].classList.add('active');
        }, 5000);
    }
}

function initializeServiceCards() {
    const serviceCards = document.querySelectorAll('.service-card');

    serviceCards.forEach(function(card) {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-10px)';
        });

        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });
}

function initializeTestimonials() {
    const testimonials = document.querySelectorAll('.testimonial');
    let currentTestimonial = 0;

    if (testimonials.length > 1) {
        // Auto-rotate testimonials
        setInterval(function() {
            testimonials[currentTestimonial].style.opacity = '0';
            currentTestimonial = (currentTestimonial + 1) % testimonials.length;
            testimonials[currentTestimonial].style.opacity = '1';
        }, 4000);
    }
}

// ===========================================
// BOOKING PAGE FEATURES
// ===========================================

function initializeBookingPage() {
    initializeCalendar();
    initializeTimeSlots();
    initializeBookingForm();
}

function initializeCalendar() {
    const calendarDays = document.querySelectorAll('.calendar-day:not(.disabled)');

    calendarDays.forEach(function(day) {
        day.addEventListener('click', function() {
            // Remove previous selection
            document.querySelectorAll('.calendar-day.selected').forEach(function(selected) {
                selected.classList.remove('selected');
            });

            // Select current day
            this.classList.add('selected');

            // Load available time slots for this date
            const date = this.dataset.date;
            loadTimeSlots(date);
        });
    });
}

function initializeTimeSlots() {
    const timeSlots = document.querySelectorAll('.time-slot');

    timeSlots.forEach(function(slot) {
        slot.addEventListener('click', function() {
            // Remove previous selection
            document.querySelectorAll('.time-slot.selected').forEach(function(selected) {
                selected.classList.remove('selected');
            });

            // Select current slot
            this.classList.add('selected');

            // Update booking summary
            updateBookingSummary();
        });
    });
}

function initializeBookingForm() {
    const bookingForm = document.getElementById('booking-form');

    if (bookingForm) {
        bookingForm.addEventListener('submit', function(event) {
            event.preventDefault();

            if (validateBookingForm()) {
                submitBooking();
            }
        });
    }
}

function loadTimeSlots(date) {
    // AJAX call to load time slots
    fetch(`/api/availability?date=${date}`)
        .then(response => response.json())
        .then(data => {
            updateTimeSlots(data.slots);
        })
        .catch(error => {
            console.error('Error loading time slots:', error);
        });
}

function updateTimeSlots(slots) {
    const timeSlotsContainer = document.getElementById('time-slots');

    if (timeSlotsContainer) {
        timeSlotsContainer.innerHTML = slots.map(slot => `
            <button type="button" class="time-slot ${slot.available ? '' : 'disabled'}" data-time="${slot.time}">
                ${slot.time}
            </button>
        `).join('');

        // Re-initialize time slot events
        initializeTimeSlots();
    }
}

function updateBookingSummary() {
    const selectedDate = document.querySelector('.calendar-day.selected')?.dataset.date;
    const selectedTime = document.querySelector('.time-slot.selected')?.dataset.time;
    const selectedService = document.querySelector('input[name="service_id"]:checked');

    const summaryElement = document.getElementById('booking-summary');

    if (summaryElement && selectedDate && selectedTime && selectedService) {
        const serviceName = selectedService.dataset.name;
        const servicePrice = selectedService.dataset.price;

        summaryElement.innerHTML = `
            <h3>Booking Summary</h3>
            <p><strong>Service:</strong> ${serviceName}</p>
            <p><strong>Date:</strong> ${formatDate(selectedDate)}</p>
            <p><strong>Time:</strong> ${selectedTime}</p>
            <p><strong>Price:</strong> $${servicePrice}</p>
        `;
    }
}

function validateBookingForm() {
    const selectedDate = document.querySelector('.calendar-day.selected');
    const selectedTime = document.querySelector('.time-slot.selected');
    const selectedService = document.querySelector('input[name="service_id"]:checked');

    if (!selectedDate || !selectedTime || !selectedService) {
        alert('Please select a service, date, and time.');
        return false;
    }

    return true;
}

function submitBooking() {
    const formData = new FormData(document.getElementById('booking-form'));

    fetch('/api/bookings', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showModal('booking-success-modal');
            // Redirect to confirmation page or show success message
        } else {
            alert('Booking failed: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error submitting booking:', error);
        alert('An error occurred. Please try again.');
    });
}

// ===========================================
// SERVICE PAGE FEATURES
// ===========================================

function initializeServicePage() {
    initializeImageGallery();
    initializeServiceBooking();
}

function initializeImageGallery() {
    const galleryImages = document.querySelectorAll('.gallery-image');
    const mainImage = document.getElementById('main-service-image');

    galleryImages.forEach(function(image) {
        image.addEventListener('click', function() {
            if (mainImage) {
                mainImage.src = this.src;
            }

            // Update active state
            document.querySelectorAll('.gallery-image').forEach(function(img) {
                img.classList.remove('active');
            });
            this.classList.add('active');
        });
    });
}

function initializeServiceBooking() {
    const bookNowButton = document.getElementById('book-now-button');

    if (bookNowButton) {
        bookNowButton.addEventListener('click', function() {
            // Scroll to booking widget or open modal
            const bookingWidget = document.getElementById('booking-widget');
            if (bookingWidget) {
                bookingWidget.scrollIntoView({ behavior: 'smooth' });
            }
        });
    }
}

// ===========================================
// PWA FEATURES
// ===========================================

function initializePWA() {
    // Register service worker
    if ('serviceWorker' in navigator) {
        navigator.serviceWorker.register('/sw.js')
            .then(function(registration) {
                console.log('Service Worker registered successfully:', registration.scope);
            })
            .catch(function(error) {
                console.log('Service Worker registration failed:', error);
            });
    }

    // Install prompt
    let deferredPrompt;

    window.addEventListener('beforeinstallprompt', function(event) {
        event.preventDefault();
        deferredPrompt = event;

        // Show install button
        const installButton = document.getElementById('pwa-install-button');
        if (installButton) {
            installButton.style.display = 'block';
            installButton.addEventListener('click', function() {
                deferredPrompt.prompt();
                deferredPrompt.userChoice.then(function(choiceResult) {
                    if (choiceResult.outcome === 'accepted') {
                        console.log('User accepted the install prompt');
                    }
                    deferredPrompt = null;
                });
            });
        }
    });

    // App installed
    window.addEventListener('appinstalled', function(event) {
        console.log('App was installed');
    });
}

// ===========================================
// UTILITY FUNCTIONS
// ===========================================

function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        weekday: 'long',
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
}

function formatTime(timeString) {
    const [hours, minutes] = timeString.split(':');
    const date = new Date();
    date.setHours(hours, minutes);
    return date.toLocaleTimeString('en-US', {
        hour: 'numeric',
        minute: '2-digit',
        hour12: true
    });
}

// AJAX helper
function ajax(url, options = {}) {
    const defaultOptions = {
        method: 'GET',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        }
    };

    const mergedOptions = { ...defaultOptions, ...options };

    if (mergedOptions.data && mergedOptions.method === 'GET') {
        const params = new URLSearchParams(mergedOptions.data);
        url += '?' + params.toString();
    }

    if (mergedOptions.data && mergedOptions.method !== 'GET') {
        mergedOptions.body = JSON.stringify(mergedOptions.data);
    }

    return fetch(url, mergedOptions)
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        });
}

// Debounce function
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = function() {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Throttle function
function throttle(func, limit) {
    let inThrottle;
    return function() {
        const args = arguments;
        const context = this;
        if (!inThrottle) {
            func.apply(context, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}
